/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.internal.classfile;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * A {@link ClassfileElement} that has complex structure defined in terms of
 * other classfile elements, such as a method, field, method body, or entire
 * class.  When encountering a {@linkplain CompoundElement}, clients have the
 * option to treat the element as a single entity (e.g., an entire method)
 * or to traverse the contents of that element with the methods in this class
 * (e.g., {@link #elements()}, {@link #forEachElement(Consumer)}, etc.)
 */
public sealed interface CompoundElement<E extends ClassfileElement>
        extends ClassfileElement, Iterable<E>
        permits ClassModel, CodeModel, FieldModel, MethodModel, jdk.internal.classfile.impl.AbstractUnboundModel {
    /**
     * Invoke the provided handler with each element contained in this
     * compound element
     * @param consumer the handler
     */
    void forEachElement(Consumer<E> consumer);

    /**
     * {@return an {@link Iterable} describing all the elements contained in this
     * compound element}
     */
    default Iterable<E> elements() {
        return elementList();
    }

    /**
     * {@return an {@link Iterator} describing all the elements contained in this
     * compound element}
     */
    @Override
    default Iterator<E> iterator() {
        return elements().iterator();
    }

    /**
     * {@return a {@link Stream} containing all the elements contained in this
     * compound element}
     */
    default Stream<E> elementStream() {
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(
                                            iterator(),
                                            Spliterator.IMMUTABLE | Spliterator.NONNULL | Spliterator.ORDERED),
                                    false);
    }

    /**
     * {@return an {@link List} containing all the elements contained in this
     * compound element}
     */
    default List<E> elementList() {
        List<E> list = new ArrayList<>();
        forEachElement(new Consumer<>() {
            @Override
            public void accept(E e) {
                list.add(e);
            }
        });
        return list;
    }

}
