/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.driver.launcher;

import com.oracle.svm.driver.launcher.BundleLauncherUtil;
import com.oracle.svm.driver.launcher.configuration.BundleContainerSettingsParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;

public class ContainerSupport {
    public String tool;
    public String bundleTool;
    public String toolVersion;
    public String bundleToolVersion;
    public String image;
    public String bundleImage;
    public Path dockerfile;
    public static final List<String> SUPPORTED_TOOLS = List.of("podman", "docker");
    public static final String TOOL_JSON_KEY = "containerTool";
    public static final String TOOL_VERSION_JSON_KEY = "containerToolVersion";
    public static final String IMAGE_JSON_KEY = "containerImage";
    public static final Path GRAAL_VM_HOME = Path.of("/graalvm", new String[0]);
    private final BiFunction<String, Throwable, Error> errorFunction;
    private final Consumer<String> warningPrinter;
    private final Consumer<String> messagePrinter;

    public ContainerSupport(Path bundleStageDir, BiFunction<String, Throwable, Error> errorFunction, Consumer<String> warningPrinter, Consumer<String> messagePrinter) {
        this.errorFunction = errorFunction;
        this.warningPrinter = warningPrinter;
        this.messagePrinter = messagePrinter;
        if (bundleStageDir != null) {
            this.dockerfile = bundleStageDir.resolve("Dockerfile");
            Path containerFile = bundleStageDir.resolve("container.json");
            if (Files.exists(containerFile, new LinkOption[0])) {
                try (BufferedReader reader = Files.newBufferedReader(containerFile);){
                    HashMap<String, String> containerSettings = new HashMap<String, String>();
                    new BundleContainerSettingsParser(containerSettings).parseAndRegister(reader);
                    this.bundleImage = containerSettings.getOrDefault(IMAGE_JSON_KEY, this.bundleImage);
                    this.bundleTool = containerSettings.getOrDefault(TOOL_JSON_KEY, this.bundleTool);
                    this.bundleToolVersion = containerSettings.getOrDefault(TOOL_VERSION_JSON_KEY, this.bundleToolVersion);
                }
                catch (IOException e) {
                    throw errorFunction.apply("Failed to read bundle-file " + String.valueOf(containerFile), e);
                }
                if (this.bundleTool != null) {
                    String containerToolVersionString = this.bundleToolVersion == null ? "" : String.format(" (%s)", this.bundleToolVersion);
                    messagePrinter.accept(String.format("%sBundled native-image was created in a container with %s%s.%n", "Native Image Bundles: ", this.bundleTool, containerToolVersionString));
                }
            }
        }
    }

    public int initializeImage() {
        try {
            this.image = BundleLauncherUtil.digest(Files.readString(this.dockerfile));
        }
        catch (IOException e) {
            throw this.errorFunction.apply("Could not read Dockerfile " + String.valueOf(this.dockerfile), e);
        }
        if (this.bundleImage != null && !this.bundleImage.equals(this.image)) {
            this.warningPrinter.accept("The bundled image was created with a different dockerfile.");
        }
        if (this.bundleTool != null && this.tool == null) {
            this.tool = this.bundleTool;
        }
        if (this.tool != null) {
            if (!ContainerSupport.isToolAvailable(this.tool)) {
                throw this.errorFunction.apply("Configured container tool not available.", null);
            }
            if (this.tool.equals("docker") && !this.isRootlessDocker()) {
                throw this.errorFunction.apply("Only rootless docker is supported for containerized builds.", null);
            }
            this.toolVersion = this.getToolVersion();
            if (this.bundleTool != null) {
                if (!this.tool.equals(this.bundleTool)) {
                    this.warningPrinter.accept(String.format("The bundled image was created with container tool '%s' (using '%s').%n", this.bundleTool, this.tool));
                } else if (this.toolVersion != null && this.bundleToolVersion != null && !this.toolVersion.equals(this.bundleToolVersion)) {
                    this.warningPrinter.accept(String.format("The bundled image was created with different %s version '%s' (installed '%s').%n", this.tool, this.bundleToolVersion, this.toolVersion));
                }
            }
        } else {
            for (String supportedTool : SUPPORTED_TOOLS) {
                if (!ContainerSupport.isToolAvailable(supportedTool)) continue;
                if (supportedTool.equals("docker") && !this.isRootlessDocker()) {
                    this.messagePrinter.accept("Native Image Bundles: Rootless context missing for docker.");
                    continue;
                }
                this.tool = supportedTool;
                this.toolVersion = this.getToolVersion();
                break;
            }
            if (this.tool == null) {
                throw this.errorFunction.apply(String.format("Please install one of the following tools before running containerized native image builds: %s", SUPPORTED_TOOLS), null);
            }
        }
        return this.createContainer();
    }

    private int createContainer() {
        ProcessBuilder pbCheckForImage = new ProcessBuilder(this.tool, "images", "-q", this.image + ":latest");
        ProcessBuilder pb = new ProcessBuilder(this.tool, "build", "-f", this.dockerfile.toString(), "-t", this.image, ".");
        String imageId = this.getFirstProcessResultLine(pbCheckForImage);
        if (imageId == null) {
            pb.inheritIO();
        } else {
            this.messagePrinter.accept(String.format("%sReusing container image %s.%n", "Native Image Bundles: ", this.image));
        }
        Process p = null;
        try {
            p = pb.start();
            int status = p.waitFor();
            if (status == 0 && imageId != null && !imageId.equals(this.getFirstProcessResultLine(pbCheckForImage))) {
                try (BufferedReader processResult = new BufferedReader(new InputStreamReader(p.getInputStream()));){
                    this.messagePrinter.accept(String.format("%sUpdated container image %s.%n", "Native Image Bundles: ", this.image));
                    processResult.lines().forEach(this.messagePrinter);
                }
            }
            int n = status;
            return n;
        }
        catch (IOException | InterruptedException e) {
            throw this.errorFunction.apply(e.getMessage(), e);
        }
        finally {
            if (p != null) {
                p.destroy();
            }
        }
    }

    private static boolean isToolAvailable(String toolName) {
        return Arrays.stream(System.getenv("PATH").split(":")).map(str -> Path.of(str, new String[0]).resolve(toolName)).anyMatch(Files::isExecutable);
    }

    private String getToolVersion() {
        ProcessBuilder pb = new ProcessBuilder(this.tool, "--version");
        return this.getFirstProcessResultLine(pb);
    }

    private boolean isRootlessDocker() {
        ProcessBuilder pb = new ProcessBuilder("docker", "context", "show");
        return this.getFirstProcessResultLine(pb).equals("rootless");
    }

    private String getFirstProcessResultLine(ProcessBuilder pb) {
        Process p = null;
        try {
            String string;
            p = pb.start();
            p.waitFor();
            BufferedReader processResult = new BufferedReader(new InputStreamReader(p.getInputStream()));
            try {
                string = processResult.readLine();
            }
            catch (Throwable throwable) {
                try {
                    try {
                        processResult.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException | InterruptedException e) {
                    throw this.errorFunction.apply(e.getMessage(), e);
                }
            }
            processResult.close();
            return string;
        }
        finally {
            if (p != null) {
                p.destroy();
            }
        }
    }

    public static Map<Path, TargetPath> mountMappingFor(Path javaHome, Path inputDir, Path outputDir) {
        HashMap<Path, TargetPath> mountMapping = new HashMap<Path, TargetPath>();
        Path containerRoot = Paths.get("/", new String[0]);
        mountMapping.put(javaHome, TargetPath.readonly(containerRoot.resolve(GRAAL_VM_HOME)));
        mountMapping.put(inputDir, TargetPath.readonly(containerRoot.resolve("input")));
        mountMapping.put(outputDir, TargetPath.of(containerRoot.resolve("output"), false));
        return mountMapping;
    }

    public List<String> createCommand(Map<String, String> containerEnvironment, Map<Path, TargetPath> mountMapping) {
        ArrayList<String> containerCommand = new ArrayList<String>();
        containerCommand.add(this.tool);
        containerCommand.add("run");
        containerCommand.add("--network=none");
        containerCommand.add("--rm");
        containerEnvironment.forEach((key, value) -> {
            containerCommand.add("-e");
            containerCommand.add(key + "=" + BundleLauncherUtil.quoteShellArg(value));
        });
        mountMapping.forEach((source, target) -> {
            containerCommand.add("--mount");
            ArrayList<Object> mountArgs = new ArrayList<Object>();
            mountArgs.add("type=bind");
            mountArgs.add("source=" + String.valueOf(source));
            mountArgs.add("target=" + String.valueOf(target.path));
            if (target.readonly) {
                mountArgs.add("readonly");
            }
            containerCommand.add(BundleLauncherUtil.quoteShellArg(String.join((CharSequence)",", mountArgs)));
        });
        containerCommand.add(this.image);
        return containerCommand;
    }

    public static void replacePaths(List<String> arguments, Path javaHome, Path bundleRoot) {
        arguments.replaceAll(arg -> arg.replace(javaHome.toString(), GRAAL_VM_HOME.toString()).replace(bundleRoot.toString(), ""));
    }

    public record TargetPath(Path path, boolean readonly) {
        public static TargetPath readonly(Path target) {
            return TargetPath.of(target, true);
        }

        public static TargetPath of(Path target, boolean readonly) {
            return new TargetPath(target, readonly);
        }
    }
}

