/*
 * Copyright (c) 2022, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.core.graal;

import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.hosted.Feature;

import com.oracle.svm.core.BuildPhaseProvider;
import com.oracle.svm.core.deopt.DeoptimizationSupport;
import com.oracle.svm.core.util.VMError;

import jdk.graal.compiler.api.replacements.Fold;

public final class RuntimeCompilation {
    /**
     * Returns whether the image supports runtime compilation using Graal. This method can be called
     * as early as during {@link Feature#afterRegistration}. {@code true} means that deoptimization
     * is also enabled, so that {@link DeoptimizationSupport#enabled} would return {@code true}.
     */
    @Fold
    public static boolean isEnabled() {
        VMError.guarantee(BuildPhaseProvider.isFeatureRegistrationFinished(), "RuntimeCompilation.isEnabled() must not be called before the feature registration is finished.");
        boolean enabled = ImageSingletons.contains(RuntimeCompilationCanaryFeature.class);
        assert !enabled || DeoptimizationSupport.enabled();
        return enabled;
    }

    private RuntimeCompilation() {
    }
}
