/*
 * Copyright (c) 2021, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
/*
 * Load static objects represented as binary data.
 */

/**
 * Stores the constant data as a binary buffer.
 *
 * This variable is initialized later.
 */
let binary_buffer = null;

class BinaryReader {
    constructor(offset) {
        this.offset = offset;
        this.view = new DataView(binary_buffer);
    }

    static readArray(readFun, numEl) {
        const arr = new Array(numEl);

        for (let i = 0; i < numEl; i++) {
            arr[i] = readFun();
        }

        return arr;
    }

    /**
     * Reads an unsigned byte value [0, 255].
     */
    readByte() {
        const b = this.view.getUint8(this.offset);
        this.offset++;
        return b;
    }

    readBytes(numBytes) {
        const bytes = new Uint8Array(binary_buffer, this.offset, numBytes);
        this.offset += numBytes;
        return bytes;
    }

    /**
     * Reads a signed byte value [-128,127].
     *
     * In java bytes are signed, so use this function when reading a java byte.
     */
    readSignedByte() {
        const b = this.view.getInt8(this.offset);
        this.offset++;
        return b;
    }

    readSignedBytes(numBytes) {
        const bytes = new Int8Array(binary_buffer, this.offset, numBytes);
        this.offset += numBytes;
        return bytes;
    }

    readSigned(numBits) {
        let ret;
        switch (numBits) {
            case 8:
                ret = this.view.getInt8(this.offset);
                break;
            case 16:
                ret = this.view.getInt16(this.offset, true);
                break;
            case 32:
                ret = this.view.getInt32(this.offset, true);
                break;
            default:
                throw Error("readSigned: " + numBits);
        }
        this.offset += numBits >> 3;
        return ret;
    }

    readSignedArray(numBits, numEl) {
        return BinaryReader.readArray(this.readSigned.bind(this, numBits), numEl);
    }

    readUnsigned(numBits) {
        let ret;
        switch (numBits) {
            case 8:
                ret = this.view.getUint8(this.offset);
                break;
            case 16:
                ret = this.view.getUint16(this.offset, true);
                break;
            case 32:
                ret = this.view.getUint32(this.offset, true);
                break;
            default:
                throw Error("readUnsigned: " + numBits);
        }
        this.offset += numBits >> 3;
        return ret;
    }

    readUnsignedArray(numBits, numEl) {
        return BinaryReader.readArray(this.readUnsigned.bind(this, numBits), numEl);
    }

    readU32() {
        return this.readUnsigned(32);
    }

    readI32() {
        return this.readSigned(32);
    }

    readLong() {
        const lo = this.readI32();
        const hi = this.readI32();
        return Long64.fromTwoInt(lo, hi);
    }

    readFloat() {
        const f = this.view.getFloat32(this.offset, true);
        this.offset += 4;
        return f;
    }

    readFloatArray(numEl) {
        return BinaryReader.readArray(this.readFloat.bind(this), numEl);
    }

    readDouble() {
        const d = this.view.getFloat64(this.offset, true);
        this.offset += 8;
        return d;
    }

    readDoubleArray(numEl) {
        return BinaryReader.readArray(this.readDouble.bind(this), numEl);
    }
}

class BinaryReaderLEB128 extends BinaryReader {
    readSigned(numBits) {
        let result = 0;
        let shift = 0;

        let msbit;
        let b;

        do {
            b = this.readByte();
            msbit = (b & 0x80) >> 7 === 1;
            result |= (b & 0x7f) << shift;
            shift += 7;
        } while (msbit);

        const signBitSet = (b & 0x40) == 0x40;

        if (shift < numBits && signBitSet) {
            result |= ~0 << shift;
        }

        return result;
    }

    readUnsigned(numBits) {
        let result = 0;
        let shift = 0;

        while (true) {
            const b = this.readByte();
            const msbit = (b & 0x80) >> 7 == 1;

            result |= (b & 0x7f) << shift;

            if (!msbit) {
                break;
            }

            shift += 7;
        }

        return result;
    }
}

function getBinaryReader(offset) {
    return new BinaryReaderLEB128(offset);
}

function loadJavaObject(reader) {
    let fields = [];

    const target = C[reader.readI32()];
    const fieldListId = reader.readI32();
    const hashCode = reader.readI32();

    const fieldList = fieldLists[fieldListId].slice(1);

    for (let i = 0; i < fieldList.length; i++) {
        const fieldType = fieldList[i][1];

        let val;

        switch (fieldType) {
            case 0:
                // Boolean
                val = reader.readByte() & 0x01;
                break;

            case 1:
                // Byte
                val = reader.readSignedByte();
                break;

            case 2:
                // Short
                val = reader.readSigned(16);
                break;

            case 3:
                // Char
                val = reader.readUnsigned(16);
                break;

            case 4:
                // Int
                val = reader.readI32();
                break;

            case 5:
                // Float
                val = reader.readFloat();
                break;

            case 6:
                // Long
                val = reader.readLong();
                break;

            case 7:
                // Double
                val = reader.readDouble();
                break;

            case 8:
                // Reference
                const id = reader.readI32();

                if (id == -1) {
                    val = null;
                } else {
                    val = C[id];
                }
                break;

            default:
                throw Error("Unknown field type: " + fieldType);
        }

        fields.push(val);
    }

    object_init(target, fieldListId, hashCode, ...fields);
}

function loadBoolArray(reader, numEl) {
    const numBytes = Math.ceil(numEl / 8);

    const arr = [];

    for (let i = 0; i < numBytes; i++) {
        let compound = reader.readByte();
        for (let j = 0; j < 8; j++) {
            const idx = i * 8 + j;

            if (idx >= numEl) {
                break;
            }

            arr.push(compound & 0x1);
            compound >>= 1;
        }
    }

    return arr;
}

function loadByteArray(reader, numEl) {
    return reader.readSignedBytes(numEl);
}

function loadShortArray(reader, numEl) {
    return reader.readSignedArray(16, numEl);
}

function loadCharArray(reader, numEl) {
    return reader.readUnsignedArray(16, numEl);
}

function loadIntArray(reader, numEl) {
    return reader.readSignedArray(32, numEl);
}

function loadFloatArray(reader, numEl) {
    return reader.readFloatArray(numEl);
}

function loadLongArray(reader, numEl) {
    const arr = [];

    for (let i = 0; i < numEl; i++) {
        arr.push(reader.readLong());
    }

    return arr;
}

function loadDoubleArray(reader, numEl) {
    return reader.readDoubleArray(numEl);
}

function loadRefArray(reader, numEl) {
    const ids = reader.readSignedArray(32, numEl);

    let refs = [];

    for (const id of ids) {
        if (id == -1) {
            refs.push(null);
        } else {
            refs.push(C[id]);
        }
    }

    return refs;
}

function loadArray(reader, componentKind, isAllEqual) {
    const target = C[reader.readI32()];
    const numEl = reader.readI32();
    const hub = C[reader.readI32()];
    const hashCode = reader.readI32();

    const numLoweredEl = isAllEqual ? 1 : numEl;

    let elements;

    let initWithSame = _a$h_;
    let initWithValues = _a$hi_;

    switch (componentKind) {
        case 0:
            // Boolean
            elements = loadBoolArray(reader, numLoweredEl);
            break;

        case 1:
            // Byte
            elements = loadByteArray(reader, numLoweredEl);
            break;

        case 2:
            // Short
            elements = loadShortArray(reader, numLoweredEl);
            break;

        case 3:
            // Char
            elements = loadCharArray(reader, numLoweredEl);
            break;

        case 4:
            // Int
            elements = loadIntArray(reader, numLoweredEl);
            break;

        case 5:
            // Float
            elements = loadFloatArray(reader, numLoweredEl);
            break;

        case 6:
            // Long
            elements = loadLongArray(reader, numLoweredEl);
            if (isAllEqual) {
                initWithSame = _a$h_BigInt64Array_;
            } else {
                initWithValues = _a$hi_BigInt64Array_;
            }
            break;

        case 7:
            // Double
            elements = loadDoubleArray(reader, numLoweredEl);
            break;

        case 8:
            // Reference
            elements = loadRefArray(reader, numLoweredEl);
            break;

        default:
            throw Error("Unknown component type: " + componentKind);
    }

    if (isAllEqual) {
        initWithSame(target, elements[0], hub, hashCode);
    } else {
        initWithValues(target, elements, hub, hashCode);
    }
}

function constructArray(reader, componentKind) {
    const targetIndex = reader.readI32();
    const numEl = reader.readI32();

    const ctr = arrayConstructors[componentKind];
    C[targetIndex] = new ctr(numEl);
}

function constructJavaString(reader) {
    const targetIndex = reader.readI32();
    const strlen = reader.readI32();
    const bytes = reader.readBytes(strlen);
    const str = new TextDecoder("utf-8").decode(bytes);
    C[targetIndex] = toJavaStringLazy(str);
}

function constructObject(reader) {
    const targetIndex = reader.readI32();
    const fieldListId = reader.readI32();
    const ctr = fieldLists[fieldListId][0];

    C[targetIndex] = new ctr();
}

function constructStaticObjects(objectListOffset) {
    const reader = getBinaryReader(objectListOffset);
    const numObjects = reader.readI32();

    for (let i = 0; i < numObjects; i++) {
        const offset = reader.readI32();

        const objReader = getBinaryReader(offset);
        const type = objReader.readByte();
        const kind = type & 0x0f;
        const isArray = !!((type >> 4) & 0x01);

        if (isArray) {
            constructArray(objReader, kind);
        } else if (kind == 15) {
            constructJavaString(objReader);
        } else {
            if (DEBUG_CHECKS) {
                if (type != 8) {
                    throw Error("Non-string, non-array object is not primitive");
                }
            }

            constructObject(objReader);
        }
    }
}

function loadStaticObjects(objectListOffset) {
    const reader = getBinaryReader(objectListOffset);
    const numObjects = reader.readI32();

    for (let i = 0; i < numObjects; i++) {
        const offset = reader.readI32();

        const objReader = getBinaryReader(offset);
        const type = objReader.readByte();
        const kind = type & 0x0f;
        const isArray = !!((type >> 4) & 0x01);

        if (isArray) {
            const isAllEqual = !!((type >> 6) & 0x01);
            loadArray(objReader, kind, isAllEqual);
        } else if (kind == 15) {
            // TODO [GR-32646] get JavaKind to int mapping directly from java
            // Strings are completely loaded during constructStaticObjects
            continue;
        } else {
            if (DEBUG_CHECKS) {
                if (type != 8) {
                    throw Error("Non-string, non-array object is not primitive");
                }
            }

            loadJavaObject(objReader);
        }
    }
}
