/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.image;

import com.oracle.svm.core.SubstrateUtil;
import com.oracle.svm.core.util.VMError;
import com.oracle.svm.util.ReflectionUtil;
import java.io.FileDescriptor;
import java.lang.ref.Cleaner;
import java.lang.ref.Reference;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.Buffer;
import java.nio.MappedByteBuffer;
import java.util.Random;
import java.util.SplittableRandom;
import java.util.concurrent.ThreadLocalRandom;
import java.util.random.RandomGenerator;
import java.util.zip.ZipFile;
import jdk.internal.ref.CleanerFactory;

public final class DisallowedImageHeapObjects {
    public static final Class<?> CANCELLABLE_CLASS = ReflectionUtil.lookupClass((boolean)false, (String)"sun.nio.fs.Cancellable");
    private static final Class<?> VIRTUAL_THREAD_CLASS = ReflectionUtil.lookupClass((boolean)false, (String)"java.lang.VirtualThread");
    public static final Class<?> CONTINUATION_CLASS = ReflectionUtil.lookupClass((boolean)false, (String)"jdk.internal.vm.Continuation");
    private static final Method CONTINUATION_IS_STARTED_METHOD = ReflectionUtil.lookupMethod(CONTINUATION_CLASS, (String)"isStarted", (Class[])new Class[0]);
    private static final Class<?> CLEANER_CLEANABLE_CLASS = ReflectionUtil.lookupClass((boolean)false, (String)"jdk.internal.ref.CleanerImpl$CleanerCleanable");
    public static final Class<?> LEGACY_CLEANER_CLASS = ReflectionUtil.lookupClass((boolean)false, (String)"jdk.internal.ref.Cleaner");
    private static final Field FILE_DESCRIPTOR_FIELD = ReflectionUtil.lookupField(MappedByteBuffer.class, (String)"fd");

    public static void check(Object obj, DisallowedObjectReporter reporter) {
        if (obj instanceof SplittableRandom) {
            SplittableRandom random = (SplittableRandom)obj;
            DisallowedImageHeapObjects.onSplittableRandomReachable(random, reporter);
        } else if (obj instanceof Random) {
            Random random = (Random)obj;
            DisallowedImageHeapObjects.onRandomReachable(random, reporter);
        }
        if (obj instanceof Thread) {
            Thread asThread = (Thread)obj;
            DisallowedImageHeapObjects.onThreadReachable(asThread, reporter);
        }
        if (SubstrateUtil.HOSTED && CONTINUATION_CLASS.isInstance(obj)) {
            DisallowedImageHeapObjects.onContinuationReachable(obj, reporter);
        }
        if (obj instanceof FileDescriptor) {
            DisallowedImageHeapObjects.onFileDescriptorReachable((FileDescriptor)obj, reporter);
        }
        if (obj instanceof Buffer) {
            Buffer buffer = (Buffer)obj;
            DisallowedImageHeapObjects.onBufferReachable(buffer, reporter);
        }
        if (obj instanceof Cleaner.Cleanable || LEGACY_CLEANER_CLASS.isInstance(obj)) {
            DisallowedImageHeapObjects.onCleanableReachable(obj, reporter);
        }
        if (obj instanceof Cleaner) {
            Cleaner cleaner = (Cleaner)obj;
            DisallowedImageHeapObjects.onCleanerReachable(cleaner, reporter);
        }
        if (obj instanceof ZipFile) {
            ZipFile zipFile = (ZipFile)obj;
            DisallowedImageHeapObjects.onZipFileReachable(zipFile, reporter);
        }
        if (CANCELLABLE_CLASS.isInstance(obj)) {
            DisallowedImageHeapObjects.onCancellableReachable(obj, reporter);
        }
    }

    public static void onRandomReachable(Random random, DisallowedObjectReporter reporter) {
        if (!(random instanceof ThreadLocalRandom)) {
            DisallowedImageHeapObjects.onRandomGeneratorReachable(random, reporter);
        }
    }

    public static void onSplittableRandomReachable(SplittableRandom random, DisallowedObjectReporter reporter) {
        DisallowedImageHeapObjects.onRandomGeneratorReachable(random, reporter);
    }

    private static void onRandomGeneratorReachable(RandomGenerator random, DisallowedObjectReporter reporter) {
        throw reporter.raise("Detected an instance of Random/SplittableRandom class in the image heap. Instances created during image generation have cached seed values and don't behave as expected.", random, "Try avoiding to initialize the class that caused initialization of the object.");
    }

    public static void onThreadReachable(Thread thread, DisallowedObjectReporter reporter) {
        if (!VIRTUAL_THREAD_CLASS.isInstance(thread) && thread.getState() != Thread.State.NEW && thread.getState() != Thread.State.TERMINATED) {
            throw reporter.raise("Detected a started Thread in the image heap. Thread name: " + thread.getName() + ". Threads running in the image generator are no longer running at image runtime.", thread, "Prevent threads from starting during image generation, or a started thread from being included in the image.");
        }
    }

    public static void onContinuationReachable(Object continuation, DisallowedObjectReporter reporter) {
        boolean isStarted;
        VMError.guarantee(CONTINUATION_CLASS.isInstance(continuation));
        try {
            isStarted = (Boolean)CONTINUATION_IS_STARTED_METHOD.invoke(continuation, new Object[0]);
        }
        catch (IllegalAccessException | InvocationTargetException ignored) {
            isStarted = false;
        }
        if (isStarted) {
            throw reporter.raise("Detected a started Continuation in the image heap. Continuation state from the image generator cannot be used at image runtime.", continuation, "Prevent continuations from starting during image generation, or started continuations from being included in the image.");
        }
    }

    public static void onFileDescriptorReachable(FileDescriptor descriptor, DisallowedObjectReporter reporter) {
        if (descriptor != FileDescriptor.in && descriptor != FileDescriptor.out && descriptor != FileDescriptor.err && descriptor.valid()) {
            throw reporter.raise("Detected a FileDescriptor in the image heap. File descriptors opened during image generation are no longer open at image runtime, and the files might not even be present anymore at image runtime.", descriptor, "Try avoiding to initialize the class that caused initialization of the FileDescriptor.");
        }
    }

    public static void onBufferReachable(Buffer buffer, DisallowedObjectReporter reporter) {
        if (buffer instanceof MappedByteBuffer) {
            MappedByteBuffer mappedBuffer = (MappedByteBuffer)buffer;
            if (mappedBuffer.capacity() != 0 || DisallowedImageHeapObjects.getFileDescriptor(mappedBuffer) != null) {
                throw reporter.raise("Detected a direct/mapped ByteBuffer in the image heap. A direct ByteBuffer has a pointer to unmanaged C memory, and C memory from the image generator is not available at image runtime. A mapped ByteBuffer references a file descriptor, which is no longer open and mapped at run time.", mappedBuffer, "Try avoiding to initialize the class that caused initialization of the MappedByteBuffer.");
            }
        } else if (buffer.isDirect()) {
            throw reporter.raise("Detected a direct Buffer in the image heap. A direct Buffer has a pointer to unmanaged C memory, and C memory from the image generator is not available at image runtime.", buffer, "Try avoiding to initialize the class that caused initialization of the direct Buffer.");
        }
    }

    public static void onCleanableReachable(Object cleanable, DisallowedObjectReporter reporter) {
        VMError.guarantee(cleanable instanceof Cleaner.Cleanable || LEGACY_CLEANER_CLASS.isInstance(cleanable));
        if (!(cleanable instanceof Reference && ((Reference)cleanable).refersTo(null) || CLEANER_CLEANABLE_CLASS.isInstance(cleanable))) {
            throw reporter.raise("Detected an active instance of Cleanable or jdk.internal.ref.Cleaner in the image heap. This usually means that a resource such as a Timer, native memory, a file descriptor or another resource is reachable which is not available at image runtime.", cleanable, "Prevent such objects being used during image generation, including by class initializers.");
        }
    }

    public static void onCleanerReachable(Cleaner cleaner, DisallowedObjectReporter reporter) {
        if (cleaner != CleanerFactory.cleaner()) {
            throw reporter.raise("Detected a java.lang.ref.Cleaner object in the image heap which uses a daemon thread that invokes cleaning actions, but threads running in the image generator are no longer running at image runtime.", cleaner, "Prevent such objects being used during image generation, including by class initializers.");
        }
    }

    public static void onZipFileReachable(ZipFile zipFile, DisallowedObjectReporter reporter) {
        throw reporter.raise("Detected a ZipFile object in the image heap. A ZipFile object contains pointers to unmanaged C memory and file descriptors, and these resources are no longer available at image runtime.", zipFile, "Try avoiding to initialize the class that caused initialization of the ZipFile.");
    }

    public static void onCancellableReachable(Object cancellable, DisallowedObjectReporter reporter) {
        VMError.guarantee(CANCELLABLE_CLASS.isInstance(cancellable));
        throw reporter.raise("Detected an instance of a class that extends " + CANCELLABLE_CLASS.getTypeName() + ": " + cancellable.getClass().getTypeName() + ". It contains a pointer to unmanaged C memory, which is no longer available at image runtime.", cancellable, "Try avoiding to initialize the class that caused initialization of the object.");
    }

    private static FileDescriptor getFileDescriptor(MappedByteBuffer buffer) {
        try {
            return (FileDescriptor)FILE_DESCRIPTOR_FIELD.get(buffer);
        }
        catch (ReflectiveOperationException ex) {
            throw VMError.shouldNotReachHere(ex);
        }
    }

    public static interface DisallowedObjectReporter {
        public RuntimeException raise(String var1, Object var2, String var3);
    }
}

