/*
 * Copyright (c) 2022, 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * Defines functionality that supports Java-to-JavaScript and JavaScript-to-Java call semantics.
 *
 * This file is included after all Java class definitions are emitted.
 */

// Java-to-JavaScript conversions

/**
 * Given a Java boxed Boolean, creates the corresponding JavaScript boolean value.
 *
 * Note: Java represents (in the generated code) primitive boolean values as JavaScript numbers,
 * hence the extra conversion in the body of this method.
 *
 * @param jlboolean The java.lang.Boolean object
 * @return {*} A JavaScript Boolean value
 */
function extractJavaScriptBoolean(jlboolean) {
    return !!$t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["extractJavaScriptBoolean"](jlboolean);
}

/**
 * Given a Java boxed Double, creates the corresponding JavaScript number value.
 *
 * Note: Java represents (in the generated code) primitive double values as JavaScript numbers,
 * hence no extra conversion is necessary.
 *
 * @param jldouble The java.lang.Double object
 * @return {*} A JavaScript Number value
 */
function extractJavaScriptNumber(jldouble) {
    return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["extractJavaScriptNumber"](jldouble);
}

/**
 * Given a Java String, creates the corresponding JavaScript string value.
 *
 * Note: the Java method called in this implementation will return (in the generated code)
 * an actual primitive Java string.
 *
 * @param jlstring The java.lang.String object
 * @return {*} A JavaScript String value
 */
function extractJavaScriptString(jlstring) {
    return jlstring.toJSString();
}

/**
 * Converts a Java array to a JavaScript array that contains JavaScript values
 * that correspond to the Java values of the input array.
 *
 * @param jarray A Java array
 * @returns {*} The resulting JavaScript array
 */
function extractJavaScriptArray(jarray) {
    const length = $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["lengthOf"](jarray);
    const jsarray = new Array(length);
    for (let i = 0; i < length; i++) {
        jsarray[i] = $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["javaToJavaScript"](jarray[i]);
    }
    return jsarray;
}

// JavaScript-to-Java conversions (standard Java classes)

/**
 * Creates a java.lang.Boolean object from a JavaScript boolean value.
 */
function createJavaBoolean(b) {
    return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaBoolean"](b);
}

/**
 * Creates a java.lang.Boolean object from a JavaScript number value.
 */
function createJavaDouble(x) {
    return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaDouble"](x);
}

/**
 * Box the given value if the specified type is primitive.
 *
 * The parameter type is the enum index as defined in jdk.vm.ci.meta.JavaKind.
 * The following is a summary:
 *
 *      0 - Boolean
 *      1 - Byte
 *      2 - Short
 *      3 - Char
 *      4 - Int
 *      5 - Float
 *      6 - Long
 *      7 - Double
 *      8 - Object
 *
 * @param {number=} type
 */
function boxIfNeeded(x, type) {
    switch (type) {
        case 0:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaBoolean"](x);
        case 1:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaByte"](x);
        case 2:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaShort"](x);
        case 3:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaCharacter"](x);
        case 4:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaInteger"](x);
        case 5:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaFloat"](x);
        case 6:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaLong"](x);
        case 7:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["createJavaDouble"](x);
        default:
            return x;
    }
}

/**
 * Unbox the given value if the specified type is primitive.
 *
 * See documentation for `boxIfNeeded`.
 */
function unboxIfNeeded(x, type) {
    switch (type) {
        case 0:
            return x.$t["java.lang.Boolean"].$m["booleanValue"](x);
        case 1:
            return x.$t["java.lang.Byte"].$m["byteValue"](x);
        case 2:
            return x.$t["java.lang.Short"].$m["shortValue"](x);
        case 3:
            return x.$t["java.lang.Character"].$m["charValue"](x);
        case 4:
            return x.$t["java.lang.Integer"].$m["intValue"](x);
        case 5:
            return x.$t["java.lang.Float"].$m["floatValue"](x);
        case 6:
            return x.$t["java.lang.Long"].$m["longValue"](x);
        case 7:
            return x.$t["java.lang.Double"].$m["doubleValue"](x);
        default:
            return x;
    }
}

// JavaScript-to-Java conversions (JSValue classes)

/**
 * Gets the Java singleton object that represents the JavaScript undefined value.
 */
function createJSUndefined() {
    return $t["org.graalvm.aotjs.api.JSValue"].$m["undefined"]();
}

/**
 * Wraps a JavaScript Boolean into a Java JSBoolean object.
 *
 * @param boolean The JavaScript boolean to wrap
 * @return {*} The Java JSBoolean object
 */
function createJSBoolean(boolean) {
    const jsboolean = new $t["org.graalvm.aotjs.api.JSBoolean"]();
    jsboolean[runtime.symbol.javaScriptNative] = boolean;
    return jsboolean;
}

/**
 * Wraps a JavaScript Number into a Java JSNumber object.
 *
 * @param number The JavaScript number to wrap
 * @return {*} The Java JSNumber object
 */
function createJSNumber(number) {
    const jsnumber = new $t["org.graalvm.aotjs.api.JSNumber"]();
    jsnumber[runtime.symbol.javaScriptNative] = number;
    return jsnumber;
}

/**
 * Wraps a JavaScript BigInt into a Java JSBigInt object.
 *
 * @param bigint The JavaScript BigInt value to wrap
 * @return {*} The Java JSBigInt object
 */
function createJSBigInt(bigint) {
    const jsbigint = new $t["org.graalvm.aotjs.api.JSBigInt"]();
    jsbigint[runtime.symbol.javaScriptNative] = bigint;
    return jsbigint;
}

/**
 * Wraps a JavaScript String into a Java JSString object.
 *
 * @param string The JavaScript String value to wrap
 * @return {*} The Java JSString object
 */
function createJSString(string) {
    const jsstring = new $t["org.graalvm.aotjs.api.JSString"]();
    jsstring[runtime.symbol.javaScriptNative] = string;
    return jsstring;
}

/**
 * Wraps a JavaScript Symbol into a Java JSSymbol object.
 *
 * @param symbol The JavaScript Symbol value to wrap
 * @return {*} The Java JSSymbol object
 */
function createJSSymbol(symbol) {
    const jssymbol = new $t["org.graalvm.aotjs.api.JSSymbol"]();
    jssymbol[runtime.symbol.javaScriptNative] = symbol;
    return jssymbol;
}

/**
 * Wraps a JavaScript object into a Java JSObject object.
 *
 * @param obj The JavaScript Object value to wrap
 * @returns {*} The Java JSObject object
 */
function createJSObject(obj) {
    const jsobj = new $t["org.graalvm.aotjs.api.JSObject"]();
    jsobj[runtime.symbol.javaScriptNative] = obj;
    return jsobj;
}

// Helper methods

/**
 * Checks if the specified object (which may be a JavaScript value or a Java value) is an internal Java object.
 */
function isInternalJavaObject(obj) {
    return obj instanceof $t["java.lang.Object"];
}

/**
 * Copies own fields from source to destination.
 *
 * Existing fields in the destination are overwritten.
 */
function copyOwnFields(src, dst) {
    for (let name of Object.getOwnPropertyNames(src)) {
        dst[name] = src[name];
    }
}

/**
 * Creates an anonymous JavaScript object, and does the mirror handshake.
 */
function createAnonymousJavaScriptObject() {
    const x = {};
    const jsObject = createJSObject(x);
    x[runtime.symbol.javaNative] = jsObject;
    return x;
}

// Java Proxies

/**
 * Handler for JavaScript Proxies that wrap Java objects.
 */
class ProxyHandler {
    constructor(javaScriptConstructor) {
        this.javaScriptConstructor = javaScriptConstructor;
        this.javaConstructorMethod = null;
        this.methods = {};
        this.staticMethods = {};
        // Function properties derived from accessible Java methods.
        this._createProxyMethods();
        // Default function properties.
        this._createDefaultMethods();
    }

    _createProxyMethods() {
        // Create proxy methods for the current class.
        const classMeta = this.javaScriptConstructor[runtime.symbol.classMeta];
        if (classMeta === undefined) {
            return;
        }
        const methodTable = classMeta.methodTable;
        if (methodTable === undefined) {
            return;
        }
        const proxyHandlerThis = this;
        for (const name in methodTable) {
            const overloads = methodTable[name];
            const instanceOverloads = [];
            const staticOverloads = [];
            for (const m of overloads) {
                if (m.isStatic) {
                    staticOverloads.push(m);
                } else {
                    instanceOverloads.push(m);
                }
            }
            if (instanceOverloads.length > 0) {
                this.methods[name] = function (...javaScriptArgs) {
                    // Note: the 'this' value is bound to the Proxy object.
                    return proxyHandlerThis._invokeProxyMethod(name, instanceOverloads, this, ...javaScriptArgs);
                };
            }
            if (staticOverloads.length > 0) {
                this.staticMethods[name] = function (...javaScriptArgs) {
                    // Note: the 'this' value is bound to the Proxy object.
                    return proxyHandlerThis._invokeProxyMethod(name, staticOverloads, null, ...javaScriptArgs);
                };
            }
        }
        if (methodTable[runtime.symbol.ctor] !== undefined) {
            const overloads = methodTable[runtime.symbol.ctor];
            this.javaConstructorMethod = function (javaScriptJavaProxy, ...javaScriptArgs) {
                // Note: the 'this' value is bound to the Proxy object.
                return proxyHandlerThis._invokeProxyMethod("<init>", overloads, javaScriptJavaProxy, ...javaScriptArgs);
            };
        } else {
            this.javaConstructorMethod = function (javaScriptJavaProxy, ...javaScriptArgs) {
                throw new Error(
                    "Cannot invoke the constructor. Make sure that the constructors are explicitly added to the image."
                );
            };
        }

        // Link the prototype chain of the superclass' proxy handler, to include super methods.
        if (this.javaScriptConstructor !== $t["java.lang.Object"]) {
            const parentClass = Object.getPrototypeOf(this.javaScriptConstructor);
            const parentProxyHandler = getOrCreateProxyHandler(parentClass);
            Object.setPrototypeOf(this.methods, parentProxyHandler.methods);
        }
    }

    _conforms(args, metadata) {
        if (metadata.paramHubs.length !== args.length) {
            return false;
        }
        for (let i = 0; i < args.length; i++) {
            const arg = args[i];
            let paramHub = metadata.paramHubs[i];
            if (paramHub === null) {
                // A null parameter hub means that the type-check always passes.
                continue;
            }
            if (paramHub.$t["java.lang.Class"].$m["isPrimitive"](paramHub)) {
                // A primitive hub must be replaced with the hub of the corresponding boxed type.
                paramHub = paramHub[runtime.symbol.boxedHub];
            }
            if (!isA(true, arg, paramHub)) {
                return false;
            }
        }
        return true;
    }

    _unboxJavaArguments(args, metadata) {
        // Precondition -- method metadata refers to a method with a correct arity.
        for (let i = 0; i < args.length; i++) {
            const paramHub = metadata.paramHubs[i];
            if (paramHub.$t["java.lang.Class"].$m["isPrimitive"](paramHub)) {
                args[i] = paramHub[runtime.symbol.unbox](args[i]);
            }
        }
    }

    _createDefaultMethods() {
        if (!this.methods.hasOwnProperty("toString")) {
            // The check must use hasOwnProperty, because toString always exists in the prototype.
            this.methods["toString"] = () => "[Java Proxy: " + this.javaScriptConstructor.name + "]";
        } else {
            const javaToString = this.methods["toString"];
            this.methods[runtime.symbol.toString] = javaToString;
            this.methods["toString"] = function () {
                // The `this` value must be bound to the proxy instance.
                //
                // The `toString` method is used often in JavaScript, and treated specially.
                // If its return type is a Java String, then that string is converted to a JavaScript string.
                // In other words, if the result of the call is a JavaScript proxy (see _invokeProxyMethod return value),
                // then proxies that represent java.lang.String are converted to JavaScript strings.
                const javaScriptResult = javaToString.call(this);
                if (typeof javaScriptResult === "function" || typeof javaScriptResult === "object") {
                    const javaResult = javaScriptResult[runtime.symbol.javaNative];
                    if (
                        javaResult !== undefined &&
                        $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["isJavaLangString"](javaResult)
                    ) {
                        return javaResult.toJSString();
                    } else {
                        return javaScriptResult;
                    }
                } else {
                    return javaScriptResult;
                }
            };
        }

        // Override Java methods that return valueOf.
        // JavaScript requires that valueOf returns a JavaScript primitive (in this case, string).
        this.methods["valueOf"] = () => "[Java Proxy: " + this.javaScriptConstructor.name + "]";

        const proxyHandlerThis = this;
        const asProperty = function (tpe) {
            // Note: this will be bound to the Proxy object.
            return coerceJavaProxyToJavaScriptType(proxyHandlerThis, this, tpe);
        };
        if (!("$as" in this.methods)) {
            this.methods["$as"] = asProperty;
        }
        this.methods[runtime.symbol.javaScriptCoerceAs] = asProperty;

        const vmProperty = vm;
        if (!("$vm" in this.methods)) {
            this.methods["$vm"] = vmProperty;
        }
    }

    _loadMethod(target, key) {
        const member = this.methods[key];
        if (member !== undefined) {
            return member;
        }
    }

    _methodNames() {
        return Object.keys(this.methods);
    }

    _invokeProxyMethod(name, overloads, javaScriptJavaProxy, ...javaScriptArgs) {
        // For static methods, javaScriptThis is set to null.
        const isStatic = javaScriptJavaProxy === null;
        const javaThis = isStatic ? null : javaScriptJavaProxy[runtime.symbol.javaNative];
        const javaArgs = eachJavaScriptToJava(javaScriptArgs);
        for (let i = 0; i < overloads.length; i++) {
            const metadata = overloads[i];
            if (this._conforms(javaArgs, metadata)) {
                // Where necessary, perform unboxing of Java arguments.
                this._unboxJavaArguments(javaArgs, metadata);
                let javaResult;
                try {
                    if (isStatic) {
                        javaResult = metadata.method.call(null, ...javaArgs);
                    } else {
                        javaResult = metadata.method.call(null, javaThis, ...javaArgs);
                    }
                } catch (error) {
                    throw javaToJavaScript(error);
                }
                if (javaResult === undefined) {
                    // This only happens when the return type is void.
                    return undefined;
                }
                // If necessary, box the Java return value.
                const retHub = metadata.returnHub;
                if (retHub.$t["java.lang.Class"].$m["isPrimitive"](retHub)) {
                    javaResult = retHub[runtime.symbol.box](javaResult);
                }
                const javaScriptResult = javaToJavaScript(javaResult);
                return javaScriptResult;
            }
        }
        const methodName = name !== null ? "method '" + name + "'" : "single abstract method";
        throw new Error("No matching signature for " + methodName + " and argument list '" + javaScriptArgs + "'");
    }

    getPrototypeOf(target) {
        return null;
    }

    setPrototypeOf(target, prototype) {
        return false;
    }

    isExtensible(target) {
        return false;
    }

    preventExtensions(target) {
        return true;
    }

    getOwnPropertyDescriptor(target, key) {
        const value = this._loadMethod(target, key);
        if (value === undefined) {
            return undefined;
        }
        return {
            value: value,
            writable: false,
            enumerable: false,
            configurable: false,
        };
    }

    defineProperty(target, key, descriptor) {
        return false;
    }

    has(target, key) {
        return this._loadMethod(target, key) !== undefined;
    }

    get(target, key) {
        if (key === runtime.symbol.javaNative) {
            return target(runtime.symbol.javaNative);
        } else {
            const javaObject = target(runtime.symbol.javaNative);
            if (Array.isArray(javaObject) && typeof key === "string") {
                const index = Number(key);
                if (0 <= index && index < javaObject.length) {
                    return javaToJavaScript(javaObject[key]);
                } else if (key === "length") {
                    return javaObject.length;
                }
            }
        }
        return this._loadMethod(target, key);
    }

    set(target, key, value, receiver) {
        const javaObject = target(runtime.symbol.javaNative);
        if (Array.isArray(javaObject)) {
            const index = Number(key);
            if (0 <= index && index < javaObject.length) {
                javaObject[key] = javaScriptToJava(value);
                return true;
            }
        }
        return false;
    }

    deleteProperty(target, key) {
        return false;
    }

    ownKeys(target) {
        return this._methodNames();
    }

    apply(target, javaScriptThisArg, javaScriptArgs) {
        // We need to convert the Proxy's target function to the Java Proxy.
        const javaScriptJavaProxy = toProxy(target(runtime.symbol.javaNative));
        // Note: the JavaScript this argument for the apply method is never exposed to Java, so we just ignore it.
        return this._applyWithObject(javaScriptJavaProxy, javaScriptArgs);
    }

    _applyWithObject(javaScriptJavaProxy, javaScriptArgs) {
        const javaThis = javaScriptJavaProxy[runtime.symbol.javaNative];
        const sam = javaThis.constructor[runtime.symbol.classMeta].singleAbstractMethod;
        if (sam === undefined) {
            throw new Error("Java Proxy is not a functional interface, so 'apply' cannot be called from JavaScript.");
        }
        return this._invokeProxyMethod(null, [sam], javaScriptJavaProxy, ...javaScriptArgs);
    }

    construct(target, argumentsList) {
        const javaThis = target(runtime.symbol.javaNative);
        if (javaThis.constructor !== $t["java.lang.Class"]) {
            throw new Error(
                "Cannot invoke the 'new' operator. The 'new' operator can only be used on Java Proxies that represent the 'java.lang.Class' type."
            );
        }
        // Step 1: Look up the generated JavaScript class (i.e., the JavaScript constructor).
        const javaScriptConstructor = javaThis[runtime.symbol.jsClass];
        // Step 2: Allocate the Java object using that generated JavaScript constructor.
        const javaInstance = new javaScriptConstructor();
        // Step 3: Obtain the Proxy Handler that corresponds to the JavaScript constructor.
        const instanceProxyHandler = getOrCreateProxyHandler(javaScriptConstructor);
        // Step 4: Check if a Java constructor method is available.
        const javaConstructorMethod = instanceProxyHandler.javaConstructorMethod;
        // Step 5: Call the Java constructor method.
        const javaScriptInstance = javaToJavaScript(javaInstance);
        javaConstructorMethod(javaScriptInstance, ...argumentsList);
        return javaScriptInstance;
    }
}

/**
 * Obtains the proxy handler of the class behind the specified constructor.
 */
function getOrCreateProxyHandler(ctor) {
    if (!ctor.hasOwnProperty(runtime.symbol.javaProxyHandler)) {
        ctor[runtime.symbol.javaProxyHandler] = new ProxyHandler(ctor);
    }
    return ctor[runtime.symbol.javaProxyHandler];
}

/**
 * Creates a proxy that intercepts messages that correspond to Java method calls and Java field accesses.
 *
 * @param obj The Java object to create a proxy for
 * @return {*} The proxy around the Java object
 */
function toProxy(obj) {
    const ctor = obj.constructor;
    let proxyHandler = getOrCreateProxyHandler(ctor);
    // The wrapper is a temporary object that allows having the non-identifier name of the target function.
    // We declare the property as a function, to ensure that it is constructable, so that the Proxy handler's construct method is callable.
    let targetWrapper = {
        ["Java Proxy"]: function (key) {
            if (key === runtime.symbol.javaNative) {
                return obj;
            }
            return undefined;
        },
    };

    return new Proxy(targetWrapper["Java Proxy"], proxyHandler);
}

/**
 * Converts a JavaScript value to the corresponding Java representation.
 *
 * The exact rules of the mapping are documented in the Java JS annotation class.
 *
 * This method is only meant to be called from the conversion code generated for JS-annotated methods.
 *
 * @param x The JavaScript value to convert
 * @return {*} The Java representation of the JavaScript value
 */
function javaScriptToJava(x) {
    // Step 1: check null, which is mapped 1:1 to null in Java.
    if (x === null) {
        return null;
    }

    // Step 2: check undefined, which is a singleton in Java.
    if (x === undefined) {
        return createJSUndefined();
    }

    // Step 3: check if the javaNative property is set.
    // This covers objects that already have Java counterparts (for example, Aot.js proxies).
    const javaValue = x[runtime.symbol.javaNative];
    if (javaValue !== undefined) {
        return javaValue;
    }

    // Step 4: use the JavaScript type to select the appropriate Java representation.
    const tpe = typeof x;
    switch (tpe) {
        case "boolean":
            return createJSBoolean(x);
        case "number":
            return createJSNumber(x);
        case "bigint":
            return createJSBigInt(x);
        case "string":
            return createJSString(x);
        case "symbol":
            return createJSSymbol(x);
        case "object":
        case "function":
            // We know this is a normal object created in JavaScript,
            // otherwise it would have a runtime.symbol.javaNative property,
            // and the conversion would have returned in Step 3.
            return createJSObject(x);
        default:
            throw new Error("unexpected type: " + tpe);
    }
}

/**
 * Maps each JavaScript value in the input array to a Java value.
 * See {@code javaScriptToJava}.
 */
function eachJavaScriptToJava(javaScriptValues) {
    const javaValues = new Array(javaScriptValues.length);
    for (let i = 0; i < javaScriptValues.length; i++) {
        javaValues[i] = javaScriptToJava(javaScriptValues[i]);
    }
    return javaValues;
}

/**
 * Converts a Java value to JavaScript.
 */
function javaToJavaScript(x) {
    return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["javaToJavaScript"](x);
}

function throwClassCastException(value, tpe) {
    let tpeName;
    if (typeof tpe === "string") {
        tpeName = tpe;
    } else if (typeof tpe === "function") {
        tpeName = tpe.name;
    } else {
        tpeName = tpe.toString();
    }
    $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["throwClassCastException"](
        value,
        toJavaString(tpeName)
    );
}

/**
 * Converts the specified Java Proxy to the target JavaScript type, if possible.
 *
 * This method is meant to be called from Java Proxy object, either when implicit coercion is enabled,
 * or when the user explicitly invokes coercion on the Proxy object.
 *
 * @param proxyHandler handler for the proxy that must be converted
 * @param proxy the Java Proxy object that should be coerced
 * @param tpe target JavaScript type name (result of the typeof operator) or constructor function
 * @return {*} the resulting JavaScript value
 */
function coerceJavaProxyToJavaScriptType(proxyHandler, proxy, tpe) {
    const o = proxy[runtime.symbol.javaNative];
    switch (tpe) {
        case "boolean":
            // Due to Java booleans being numbers, the double-negation is necessary.
            return !!$t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptBoolean"](o);
        case "number":
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptNumber"](o);
        case "bigint":
            const bs = $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptBigInt"](o);
            return BigInt(bs);
        case "string":
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptString"](o);
        case "object":
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptObject"](o);
        case "function":
            const sam = o.constructor[runtime.symbol.classMeta].singleAbstractMethod;
            if (sam !== undefined) {
                return (...args) => proxyHandler._applyWithObject(proxy, args);
            }
            throwClassCastException(o, tpe);
        case Uint8Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptUint8Array"](o);
        case Int8Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptInt8Array"](o);
        case Uint16Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptUint16Array"](o);
        case Int16Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptInt16Array"](o);
        case Int32Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptInt32Array"](o);
        case Float32Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptFloat32Array"](o);
        case BigInt64Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptBigInt64Array"](o);
        case Float64Array:
            return $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["coerceToJavaScriptFloat64Array"](o);
        default:
            throwClassCastException(o, tpe);
    }
}

/**
 * Try to convert the JavaScript object to a Java facade class, or return null.
 *
 * @param obj JavaScript object whose Java facade class we search for
 * @param cls target Java class in the form of its JavaScript counterpart
 * @return {*} the mirror instance wrapped into a JavaScript Java Proxy, or null
 */
function tryExtractFacadeClass(obj, cls) {
    const facades = runtime.findFacadesFor(obj.constructor);
    const rawJavaHub = cls[runtime.symbol.javaNative];
    const internalJavaClass = rawJavaHub[runtime.symbol.jsClass];
    if (facades.has(internalJavaClass)) {
        const rawJavaMirror = new internalJavaClass();
        // Note: only one-way handshake, since the JavaScript object could be recast to a different Java facade class.
        rawJavaMirror[runtime.symbol.javaScriptNative] = obj;
        return toProxy(rawJavaMirror);
    } else {
        return null;
    }
}

/**
 * Coerce the specified JavaScript value to the specified Java type.
 *
 * See VM.as for the specification of this function.
 */
function coerceJavaScriptToJavaType(javaScriptValue, type) {
    let typeHub;
    if (typeof type === "string") {
        typeHub = runtime.hubs[type];
    } else if (typeof type === "object") {
        const javaType = type[runtime.symbol.javaNative];
        if (javaType !== undefined && javaType.constructor === runtime.classHub) {
            typeHub = javaType;
        }
    }
    if (typeHub === undefined) {
        throw new Error(
            "Cannot coerce JavaScript value with the requested type descriptor (use String or Java Class): " + type
        );
    }
    // Check if the current object is a Java Proxy, in which case no coercion is possible.
    let javaValue = javaScriptValue[runtime.symbol.javaNative];
    if (javaValue !== undefined) {
        const valueHub = runtime.hubOf(javaValue);
        if (runtime.isSupertype(typeHub, valueHub)) {
            return javaValue;
        } else {
            throw new Error("Cannot coerce Java Proxy of type '" + valueHub + "' to the type '" + typeHub + "'");
        }
    }
    // Do a normal conversion, and invoke the as method on the JSValue.
    javaValue = runtime.javaScriptToJava(javaScriptValue);
    return runtime.javaToJavaScript(javaValue.$t["org.graalvm.aotjs.api.JSValue"].$m["as"](javaValue, typeHub));
}

runtime.javaToJavaScript = javaToJavaScript;

runtime.javaScriptToJava = javaScriptToJava;

runtime.classHub = $t["java.lang.Class"];

runtime.hubOf = $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["hubOf"];

runtime.isSupertype = $t["com.oracle.svm.aotjs.functionintrinsics.c"].$m["isSupertype"];

vm.as = coerceJavaScriptToJavaType;
