/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.jipher.internal.tools.asn1;

import com.oracle.jipher.internal.tools.asn1.Asn1;
import com.oracle.jipher.internal.tools.asn1.Asn1ContentDecodeException;
import com.oracle.jipher.internal.tools.asn1.Asn1DecodeException;
import com.oracle.jipher.internal.tools.asn1.Asn1DerDecodeException;
import com.oracle.jipher.internal.tools.asn1.Asn1Exception;
import com.oracle.jipher.internal.tools.asn1.TagClass;
import com.oracle.jipher.internal.tools.asn1.UniversalTag;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.SimpleTimeZone;

public final class Asn1BerValue {
    static final TagClass[] TAG_CLASS_VALUES = TagClass.values();
    static final String INVALID_BER_CONTENT_ENCODING = "Invalid BER content encoding: ";
    static final String CONTENT_NOT_DER_COMPLIANT = "Content not DER compliant: ";
    static final String BUFFER_UNDERFLOW_AT_OFFSET = "Buffer underflow while decoding value; at offset ";
    static final String CONTENT_LENGTH_EXCEEDS_IMPLEMENTATION_LIMIT = "Content length exceeds implementation limit";
    static final String AT_OFFSET = "; at offset ";
    static final int ID_TAG_CLASS_MASK = 192;
    static final int ID_TAG_CLASS_SHIFT = 6;
    static final int ID_CONSTRUCTED_MASK = 32;
    static final int ID_TAG_MASK = 31;
    static final int INDEFINITE_LENGTH = -1;
    public final TagClass tagClass;
    public final int tagValue;
    public final int offset;
    public final boolean constructed;
    public final boolean endOfContents;
    final SortOrder sortOrder;
    final boolean checkDer;
    final ByteBuffer contentBuffer;
    final List<Asn1BerValue> contentValues;
    boolean der = true;
    static final SetComparator SET_COMPARATOR = new SetComparator();
    static final SetOfComparator SET_OF_COMPARATOR = new SetOfComparator();

    Asn1BerValue(ByteBuffer src, boolean checkDer) {
        int len;
        this.sortOrder = SortOrder.SEQUENCE;
        this.checkDer = checkDer;
        ByteBuffer rbb = src.asReadOnlyBuffer();
        this.offset = rbb.position();
        try {
            byte id = rbb.get();
            this.tagClass = TAG_CLASS_VALUES[(id & 0xC0) >> 6];
            this.constructed = (id & 0x20) != 0;
            int tag = id & 0x1F;
            if (tag == 31) {
                tag = Asn1BerValue.decodeTagValue(rbb);
            }
            this.tagValue = tag;
            len = this.decodeLength(rbb);
        }
        catch (Asn1DecodeException ex) {
            throw new Asn1DecodeException(ex.getMessage() + AT_OFFSET + this.offset, ex);
        }
        catch (BufferUnderflowException ex) {
            throw new Asn1DecodeException(BUFFER_UNDERFLOW_AT_OFFSET + rbb.position(), ex);
        }
        if (checkDer) {
            this.der();
        }
        if (this.tagClass == TagClass.UNIVERSAL && this.tagValue == 0) {
            if (len != 0 || this.constructed || !this.der) {
                throw new Asn1DecodeException("Invalid End-of-contents marker; at offset " + this.offset);
            }
            this.endOfContents = true;
            src.position(rbb.position());
            this.contentValues = null;
            this.contentBuffer = null;
            return;
        }
        this.endOfContents = false;
        rbb.mark();
        if (len != -1) {
            if (len > rbb.remaining()) {
                throw new Asn1DecodeException(BUFFER_UNDERFLOW_AT_OFFSET + this.offset);
            }
            rbb.limit(rbb.position() + len);
        }
        if (this.constructed) {
            if (len == -1) {
                this.contentValues = Asn1.decodeUntilMark(rbb);
                rbb.limit(rbb.position());
            } else {
                this.contentValues = Asn1.decodeAll(rbb, checkDer);
                for (Asn1BerValue value : this.contentValues) {
                    if (value.isDer()) continue;
                    this.der = false;
                    break;
                }
            }
        } else {
            if (len == -1) {
                throw new Asn1DecodeException("Non-constructed BER value has indefinite length; at offset " + this.offset);
            }
            this.contentValues = null;
        }
        rbb.reset();
        this.contentBuffer = rbb;
        src.position(rbb.limit());
    }

    Asn1BerValue(TagClass tagClass, int tagValue, boolean constructed, SortOrder sortOrder, List<Asn1BerValue> contentValues, ByteBuffer contentBuffer) {
        this.tagClass = tagClass;
        this.tagValue = tagValue;
        this.offset = -1;
        this.constructed = constructed;
        this.endOfContents = false;
        this.sortOrder = sortOrder;
        this.checkDer = false;
        this.contentValues = contentValues;
        this.contentBuffer = contentBuffer;
    }

    public Asn1BerValue der() {
        if (!this.der) {
            throw new Asn1DerDecodeException("Not DER; at offset " + this.offset);
        }
        return this;
    }

    public Asn1BerValue constructed() {
        if (!this.constructed) {
            throw new Asn1DecodeException("Non-constructed value; at offset " + this.offset);
        }
        return this;
    }

    public Asn1BerValue primitive() {
        if (this.constructed) {
            throw new Asn1DecodeException("Constructed value; at offset " + this.offset);
        }
        return this;
    }

    public Asn1BerValue tag(int expectedCsTag) {
        this.tag(TagClass.CONTEXT_SPECIFIC, expectedCsTag);
        return this;
    }

    public Asn1BerValue tag(UniversalTag expectedUniversalTag) {
        this.tag(TagClass.UNIVERSAL, expectedUniversalTag.tagValue);
        return this;
    }

    public Asn1BerValue tag(TagClass expectedTagClass, int expectedTagValue) {
        if (!this.hasTag(expectedTagClass, expectedTagValue)) {
            throw new Asn1DecodeException("Unexpected tag at offset " + this.offset + "; expected: " + Asn1.tagToString(expectedTagClass, expectedTagValue) + ", was: " + Asn1.tagToString(this.tagClass, this.tagValue));
        }
        return this;
    }

    public Asn1BerValue tagIfUniversal(UniversalTag expectedUniversalTag) {
        if (this.tagClass == TagClass.UNIVERSAL) {
            this.tag(expectedUniversalTag);
        }
        return this;
    }

    public boolean hasTag(int expectedCsTag) {
        return this.hasTag(TagClass.CONTEXT_SPECIFIC, expectedCsTag);
    }

    public boolean hasTag(UniversalTag expectedUniversalTag) {
        return this.hasTag(TagClass.UNIVERSAL, expectedUniversalTag.tagValue);
    }

    public boolean hasTag(TagClass expectedTagClass, int expectedTagValue) {
        if (expectedTagClass == null || expectedTagValue < 0) {
            throw new IllegalArgumentException("Invalid expected tag class or tag value");
        }
        return this.tagClass == expectedTagClass && this.tagValue == expectedTagValue;
    }

    static int decodeIdentifier(ByteBuffer src) {
        byte octet = src.get();
        if (octet == -128) {
            throw new Asn1DecodeException("Invalid identifier encoding");
        }
        int value = octet & 0x7F;
        while (octet < 0) {
            if (value > 0xFFFFFF) {
                throw new Asn1DecodeException("Identifier value exceeds implementation limit");
            }
            octet = src.get();
            value = value << 7 | octet & 0x7F;
        }
        return value;
    }

    static int decodeTagValue(ByteBuffer src) {
        int tagValue;
        try {
            tagValue = Asn1BerValue.decodeIdentifier(src);
            if (tagValue < 31) {
                throw new Asn1DecodeException("Tag value would fit in first identifier octet");
            }
        }
        catch (Asn1DecodeException ex) {
            throw new Asn1DecodeException("Invalid BER tag encoding", ex);
        }
        return tagValue;
    }

    int decodeLength(ByteBuffer src) {
        int b = src.get();
        if (b >= 0) {
            return b;
        }
        int c = b & 0x7F;
        if (c == 0) {
            this.der = false;
            return -1;
        }
        if (c > 4) {
            throw new Asn1DecodeException(CONTENT_LENGTH_EXCEEDS_IMPLEMENTATION_LIMIT);
        }
        int len = src.get() & 0xFF;
        if (len == 0) {
            this.der = false;
        }
        while (--c > 0) {
            b = src.get() & 0xFF;
            len = len << 8 | b;
        }
        if (len < 0) {
            throw new Asn1DecodeException(CONTENT_LENGTH_EXCEEDS_IMPLEMENTATION_LIMIT);
        }
        if (len < 128) {
            this.der = false;
        }
        return len;
    }

    public boolean isDer() {
        return this.der;
    }

    public ByteBuffer content() {
        return this.contentBuffer.slice();
    }

    public ByteBuffer gatherContent() {
        ArrayList<ByteBuffer> bufferList = new ArrayList<ByteBuffer>();
        int size = this.gatherContent(bufferList);
        ByteBuffer gatherBuf = ByteBuffer.allocate(size);
        for (ByteBuffer buf : bufferList) {
            gatherBuf.put(buf);
        }
        gatherBuf.rewind();
        return gatherBuf;
    }

    int gatherContent(List<ByteBuffer> bufferList) {
        if (this.constructed) {
            int size = 0;
            for (Asn1BerValue value : this.values()) {
                value.tag(UniversalTag.OCTET_STRING);
                size += value.gatherContent(bufferList);
            }
            return size;
        }
        if (this.contentBuffer.hasRemaining()) {
            bufferList.add(this.content());
        }
        return this.contentBuffer.remaining();
    }

    public byte[] gatherOctets() {
        return this.gatherContent().array();
    }

    public byte[] octets() {
        ByteBuffer c = this.content();
        byte[] contentOctets = new byte[c.remaining()];
        c.get(contentOctets);
        return contentOctets;
    }

    public List<Asn1BerValue> values() {
        this.constructed();
        return this.contentValues;
    }

    public List<Asn1BerValue> sequence() {
        this.constructed().tagIfUniversal(UniversalTag.SEQUENCE);
        return this.contentValues;
    }

    public List<Asn1BerValue> set() {
        this.constructed().tagIfUniversal(UniversalTag.SET);
        return this.contentValues;
    }

    public Asn1BerValue maxCount(int max) {
        List<Asn1BerValue> vl = this.values();
        if (vl.size() > max) {
            throw new Asn1DecodeException("Constructed BER value contains more than " + max + " sub-values: " + vl.size() + AT_OFFSET + this.offset);
        }
        return this;
    }

    public Asn1BerValue minCount(int min) {
        List<Asn1BerValue> vl = this.values();
        if (vl.size() < min) {
            throw new Asn1DecodeException("Constructed BER value contains less than " + min + " sub-values: " + vl.size() + AT_OFFSET + this.offset);
        }
        return this;
    }

    public Asn1BerValue count(int min, int max) {
        return this.minCount(min).maxCount(max);
    }

    public Asn1BerValue count(int n) {
        return this.count(n, n);
    }

    public Asn1BerValue explicit() {
        this.count(1);
        if (this.tagClass == TagClass.UNIVERSAL) {
            throw new Asn1DecodeException("Unexpected universal explicit tag; at offset " + this.offset);
        }
        return this.contentValues.get(0);
    }

    public boolean getBoolean() {
        this.primitive().tagIfUniversal(UniversalTag.BOOLEAN);
        if (this.contentBuffer.remaining() != 1) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + UniversalTag.BOOLEAN.asn1Name() + AT_OFFSET + this.offset);
        }
        byte v = this.contentBuffer.get(this.contentBuffer.position());
        if (this.checkDer && (v + 1 & 0xFE) != 0) {
            throw new Asn1DerDecodeException(CONTENT_NOT_DER_COMPLIANT + UniversalTag.BOOLEAN.asn1Name() + AT_OFFSET + this.offset);
        }
        return v != 0;
    }

    public BigInteger getEnumerated() {
        return this.getInteger(UniversalTag.ENUMERATED);
    }

    public BigInteger getInteger() {
        return this.getInteger(UniversalTag.INTEGER);
    }

    private BigInteger getInteger(UniversalTag type) {
        this.primitive().tagIfUniversal(type);
        if (!this.contentBuffer.hasRemaining()) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + type.asn1Name() + AT_OFFSET + this.offset);
        }
        byte[] val = this.octets();
        if (val.length > 1 && (val[0] + 1 & 0xFE) == 0) {
            int i;
            byte b = val[0];
            for (i = 1; i < val.length - 1 && val[i] == b; ++i) {
            }
            if (((b ^ val[i]) & 0x80) == 0) {
                throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + type.asn1Name() + AT_OFFSET + this.offset);
            }
        }
        return new BigInteger(val);
    }

    public Date getUtcTime() {
        return this.getTime(UniversalTag.UTCTime);
    }

    public Date getGeneralizedTime() {
        return this.getTime(UniversalTag.GeneralizedTime);
    }

    private Date getTime(UniversalTag tag) {
        this.primitive().tagIfUniversal(tag);
        if (!this.contentBuffer.hasRemaining()) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + (Object)((Object)tag) + AT_OFFSET + this.offset);
        }
        String dateString = new String(this.gatherOctets(), Asn1.US_ASCII);
        try {
            return Asn1BerValue.parseTime(dateString, tag);
        }
        catch (ParseException e) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + (Object)((Object)tag) + AT_OFFSET + this.offset);
        }
    }

    private static String timezoneSubstring(String s) {
        int zIndex = s.indexOf("Z");
        int zdiffIndex = Math.max(s.indexOf("-"), s.indexOf("+"));
        if (zIndex == -1 && zdiffIndex == -1) {
            return "";
        }
        int tzIndex = Math.max(zIndex, zdiffIndex);
        return s.substring(tzIndex);
    }

    private static String fractionSubstring(String s, String tzSubs) {
        int decIndex = (s = s.replace(',', '.')).indexOf(46);
        if (decIndex == -1) {
            return "";
        }
        return s.substring(decIndex, s.length() - tzSubs.length());
    }

    private static String appendZeros(String s, int numZeros) {
        StringBuilder sb = new StringBuilder(s);
        for (int i = 0; i < numZeros; ++i) {
            sb.append("0");
        }
        return sb.toString();
    }

    static Date parseTime(String s, UniversalTag timeTag) throws ParseException {
        String dateFormat;
        String tz = Asn1BerValue.timezoneSubstring(s);
        String dec = Asn1BerValue.fractionSubstring(s, tz);
        String time = s.substring(0, s.length() - tz.length() - dec.length());
        String string = dateFormat = timeTag == UniversalTag.GeneralizedTime ? "yyyyMMddHHmmss" : "yyMMddHHmmss";
        if (dateFormat.length() - time.length() > 0) {
            time = Asn1BerValue.appendZeros(time, dateFormat.length() - time.length());
        } else if (dateFormat.length() == time.length() && timeTag == UniversalTag.GeneralizedTime && !dec.isEmpty()) {
            dec = dec.length() < 4 ? Asn1BerValue.appendZeros(dec, 4 - dec.length()) : dec.substring(0, 4);
            time = time + dec;
            dateFormat = dateFormat + ".SSS";
        }
        time = time + tz;
        if (tz.equals("Z")) {
            dateFormat = dateFormat + "'Z'";
        } else if (!tz.isEmpty()) {
            dateFormat = dateFormat + "Z";
        }
        SimpleDateFormat df = new SimpleDateFormat(dateFormat);
        if (tz.equals("Z")) {
            df.setTimeZone(new SimpleTimeZone(0, "UTC"));
        }
        return df.parse(time);
    }

    public ByteBuffer getBitStringContent() {
        int unusedBits;
        this.tagIfUniversal(UniversalTag.BIT_STRING);
        ByteBuffer content = this.gatherContent();
        if (!content.hasRemaining() || (unusedBits = content.get(0) & 0xFF) > 7) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + UniversalTag.BIT_STRING.asn1Name() + AT_OFFSET + this.offset);
        }
        if (unusedBits > 0) {
            byte lastOctet;
            boolean contentIsDer;
            int mask = (1 << unusedBits) - 1;
            int lastIndex = content.capacity() - 1;
            boolean bl = contentIsDer = mask == 0 || lastIndex > 0;
            if (lastIndex > 0 && ((lastOctet = content.get(lastIndex)) & mask) != 0) {
                content.put(lastIndex, (byte)(lastOctet & ~mask));
                contentIsDer = false;
            }
            if (this.checkDer && !contentIsDer) {
                throw new Asn1DerDecodeException(CONTENT_NOT_DER_COMPLIANT + UniversalTag.BIT_STRING.asn1Name() + AT_OFFSET + this.offset);
            }
        }
        return content;
    }

    public byte[] getBitStringOctets() {
        byte[] bytes = this.getBitStringContent().array();
        return Arrays.copyOfRange(bytes, 1, bytes.length);
    }

    public BitSet getBitString() {
        ByteBuffer content = this.getBitStringContent();
        content.position(1);
        while (content.hasRemaining()) {
            int b = Integer.reverse(content.get(content.position())) >>> 24;
            content.put((byte)b);
        }
        content.position(1);
        return BitSet.valueOf(content);
    }

    public byte[] getOctetString() {
        this.tagIfUniversal(UniversalTag.OCTET_STRING);
        return this.gatherOctets();
    }

    public void getNull() {
        this.primitive().tagIfUniversal(UniversalTag.NULL);
        if (this.contentBuffer.hasRemaining()) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + UniversalTag.NULL.asn1Name() + AT_OFFSET + this.offset);
        }
    }

    static int countOidSubIdentifiers(ByteBuffer content) {
        int subIds = 0;
        content.mark();
        while (content.hasRemaining()) {
            if (content.get() < 0) continue;
            ++subIds;
        }
        content.reset();
        return subIds;
    }

    public static String dottedString(int[] components) {
        if (components.length == 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        sb.append(components[0]);
        for (int i = 1; i < components.length; ++i) {
            sb.append('.').append(components[i]);
        }
        return sb.toString();
    }

    void decodeOidSubIdentifiers(ByteBuffer content, int[] subIds, int offset, UniversalTag ut) {
        try {
            while (content.hasRemaining()) {
                subIds[offset++] = Asn1BerValue.decodeIdentifier(content);
            }
        }
        catch (Asn1DecodeException | BufferUnderflowException ex) {
            throw new Asn1ContentDecodeException(INVALID_BER_CONTENT_ENCODING + ut.asn1Name() + AT_OFFSET + this.offset, ex);
        }
    }

    public int[] getOidComponents() {
        this.primitive().tagIfUniversal(UniversalTag.OBJECT_IDENTIFIER);
        ByteBuffer content = this.content();
        int[] subIds = new int[Asn1BerValue.countOidSubIdentifiers(content) + 1];
        int c1 = Asn1BerValue.decodeIdentifier(content);
        int c0 = c1 / 40;
        switch (c0) {
            case 0: {
                break;
            }
            case 1: {
                c1 -= 40;
                break;
            }
            default: {
                c0 = 2;
                c1 -= 80;
            }
        }
        subIds[0] = c0;
        subIds[1] = c1;
        this.decodeOidSubIdentifiers(content, subIds, 2, UniversalTag.OBJECT_IDENTIFIER);
        return subIds;
    }

    public String getOid() {
        return Asn1BerValue.dottedString(this.getOidComponents());
    }

    public int[] getRelativeOidComponents() {
        this.primitive().tagIfUniversal(UniversalTag.RELATIVE_OID);
        ByteBuffer content = this.content();
        int[] subIds = new int[Asn1BerValue.countOidSubIdentifiers(content)];
        this.decodeOidSubIdentifiers(content, subIds, 0, UniversalTag.RELATIVE_OID);
        return subIds;
    }

    public String getRelativeOid() {
        return Asn1BerValue.dottedString(this.getRelativeOidComponents());
    }

    public String getRcs() {
        UniversalTag ut = Asn1.toUniversalTag(this.tagClass, this.tagValue);
        if (ut == null) {
            throw new Asn1DecodeException("Valid Restricted Character String Universal Tag required: " + Asn1.tagToString(this.tagClass, this.tagValue));
        }
        return this.getRcs(ut);
    }

    public String getRcs(UniversalTag encoding) {
        Charset charset;
        this.tagIfUniversal(encoding);
        try {
            charset = Asn1.getRcsCharset(encoding);
        }
        catch (Asn1Exception ex) {
            throw new Asn1DecodeException(ex.getMessage() + AT_OFFSET + this.offset, ex);
        }
        return new String(this.gatherOctets(), charset);
    }

    int calcIdAndLenLength(int contentLen) {
        int idLen = 1;
        if (this.tagValue >= 31) {
            int tagBitCount = 32 - Integer.numberOfLeadingZeros(this.tagValue);
            idLen += (tagBitCount + 6) / 7;
        }
        int lenLen = 1;
        if (contentLen >= 128) {
            int lenBitCount = 32 - Integer.numberOfLeadingZeros(contentLen);
            lenLen += (lenBitCount + 7) / 8;
        }
        return idLen + lenLen;
    }

    public int contentLength() {
        if (this.constructed) {
            int len = 0;
            for (Asn1BerValue value : this.contentValues) {
                len += value.encodedLength();
            }
            return len;
        }
        return this.contentBuffer.remaining();
    }

    public int encodedLength() {
        int contentLen = this.contentLength();
        return this.calcIdAndLenLength(contentLen) + contentLen;
    }

    void encodeIdAndLen(ByteBuffer dst) {
        int id = this.tagClass.ordinal() << 6;
        if (this.constructed) {
            id |= 0x20;
        }
        if (this.tagValue < 31) {
            dst.put((byte)(id |= this.tagValue));
        } else {
            dst.put((byte)(id |= 0x1F));
            Asn1.encodeIdentifier(dst, this.tagValue);
        }
        int contentLen = this.contentLength();
        if (contentLen < 128) {
            dst.put((byte)contentLen);
        } else {
            int lenBitCount = 32 - Integer.numberOfLeadingZeros(contentLen);
            int lenLen = (lenBitCount + 7) / 8;
            dst.put((byte)(0x80 | lenLen));
            int shift = lenLen * 8;
            do {
                dst.put((byte)(contentLen >>> (shift -= 8)));
            } while (shift > 0);
        }
    }

    public static void encodeSequenceContent(ByteBuffer dst, List<Asn1BerValue> content) {
        for (Asn1BerValue value : content) {
            value.encodeDer(dst);
        }
    }

    public static void encodeSetContent(ByteBuffer dst, List<Asn1BerValue> content) {
        ArrayList<Asn1BerValue> sortedContent = new ArrayList<Asn1BerValue>(content);
        Collections.sort(sortedContent, SET_COMPARATOR);
        Asn1BerValue.encodeSequenceContent(dst, sortedContent);
    }

    public static void encodeSetOfContent(ByteBuffer dst, List<Asn1BerValue> content) {
        byte[][] sortedContent = new byte[content.size()][];
        int i = 0;
        for (Asn1BerValue value : content) {
            sortedContent[i++] = value.encodeDerOctets();
        }
        Arrays.sort(sortedContent, SET_OF_COMPARATOR);
        for (Object encodedValue : (Object)sortedContent) {
            dst.put((byte[])encodedValue);
        }
    }

    public void encodeDer(ByteBuffer dst) {
        this.encodeIdAndLen(dst);
        if (this.constructed) {
            switch (this.sortOrder) {
                case SEQUENCE: {
                    Asn1BerValue.encodeSequenceContent(dst, this.contentValues);
                    break;
                }
                case SET: {
                    Asn1BerValue.encodeSetContent(dst, this.contentValues);
                    break;
                }
                case SET_OF: {
                    Asn1BerValue.encodeSetOfContent(dst, this.contentValues);
                }
            }
        } else {
            dst.put(this.content());
        }
    }

    public ByteBuffer encodeDer() {
        ByteBuffer buf = ByteBuffer.allocate(this.encodedLength());
        this.encodeDer(buf);
        assert (!buf.hasRemaining());
        buf.rewind();
        return buf;
    }

    public byte[] encodeDerOctets() {
        return this.encodeDer().array();
    }

    static String hex(ByteBuffer buf) {
        StringBuilder sb = new StringBuilder();
        try {
            Asn1BerValue.hex(sb, buf);
        }
        catch (IOException ex) {
            throw new Error(ex);
        }
        return sb.toString();
    }

    static void hex(Appendable out, ByteBuffer buf) throws IOException {
        boolean first = true;
        while (buf.hasRemaining()) {
            if (first) {
                first = false;
            } else {
                out.append(' ');
            }
            out.append(String.format("%02X", buf.get() & 0xFF));
        }
    }

    static String quoteString(String str) {
        return str.replace("'", "\\'");
    }

    public String toString() {
        if (this.constructed) {
            return Asn1.tagToString(this.tagClass, this.tagValue) + " <constructed>";
        }
        UniversalTag ut = Asn1.toUniversalTag(this.tagClass, this.tagValue);
        if (ut != null) {
            switch (ut) {
                case BOOLEAN: {
                    return Boolean.toString(this.getBoolean()).toUpperCase();
                }
                case INTEGER: 
                case ENUMERATED: {
                    return this.getInteger(ut).toString();
                }
                case NULL: {
                    this.getNull();
                    return "NULL";
                }
                case OBJECT_IDENTIFIER: {
                    return "(" + this.getOid().replace('.', ' ') + ')';
                }
                case RELATIVE_OID: {
                    return "(" + this.getRelativeOid().replace('.', ' ') + ')';
                }
                case UTF8String: 
                case NumericString: 
                case PrintableString: 
                case TeletexString: 
                case VideotexString: 
                case IA5String: 
                case GraphicString: 
                case VisibleString: 
                case GeneralString: 
                case UniversalString: 
                case BMPString: {
                    return "'" + Asn1BerValue.quoteString(this.getRcs(ut)) + '\'';
                }
                case UTCTime: {
                    return this.getUtcTime().toString();
                }
                case GeneralizedTime: {
                    return this.getGeneralizedTime().toString();
                }
            }
        }
        return Asn1BerValue.hex(this.content());
    }

    public int hashCode() {
        int tagHash = this.tagValue * 3 + this.tagClass.ordinal();
        if (this.constructed) {
            return this.contentValues.hashCode() * 17 + tagHash;
        }
        return this.contentBuffer.hashCode() * 17 + tagHash;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Asn1BerValue)) {
            return false;
        }
        Asn1BerValue other = (Asn1BerValue)o;
        if (this.tagClass != other.tagClass || this.tagValue != other.tagValue || this.constructed != other.constructed) {
            return false;
        }
        if (this.constructed) {
            return this.contentValues.equals(other.contentValues);
        }
        return this.contentBuffer.equals(other.contentBuffer);
    }

    static class SetOfComparator
    implements Comparator<byte[]> {
        SetOfComparator() {
        }

        @Override
        public int compare(byte[] v1, byte[] v2) {
            int commLen = Math.min(v1.length, v2.length);
            for (int i = 0; i < commLen; ++i) {
                int c = (v1[i] & 0xFF) - (v2[i] & 0xFF);
                if (c == 0) continue;
                return c;
            }
            return Integer.compare(v1.length, v2.length);
        }
    }

    static class SetComparator
    implements Comparator<Asn1BerValue> {
        SetComparator() {
        }

        @Override
        public int compare(Asn1BerValue v1, Asn1BerValue v2) {
            if (v1.tagClass != v2.tagClass) {
                return v1.tagClass.ordinal() - v2.tagClass.ordinal();
            }
            return Integer.compare(v1.tagValue, v2.tagValue);
        }
    }

    static enum SortOrder {
        SEQUENCE,
        SET,
        SET_OF;

    }
}

