/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.compiler.hotspot.amd64;

import org.graalvm.compiler.asm.amd64.AMD64MacroAssembler;
import org.graalvm.compiler.core.common.NumUtil;
import org.graalvm.compiler.debug.GraalError;
import org.graalvm.compiler.hotspot.GraalHotSpotVMConfig;
import org.graalvm.compiler.options.OptionValues;

import jdk.vm.ci.code.TargetDescription;
import jdk.vm.ci.code.site.Call;
import jdk.vm.ci.code.site.Infopoint;
import jdk.vm.ci.code.site.InfopointReason;

public class AMD64HotSpotMacroAssembler extends AMD64MacroAssembler {
    private final GraalHotSpotVMConfig config;

    public AMD64HotSpotMacroAssembler(GraalHotSpotVMConfig config, TargetDescription target, OptionValues optionValues, boolean hasIntelJccErratum) {
        super(target, optionValues, hasIntelJccErratum);
        this.config = config;
    }

    @Override
    public void postCallNop(Infopoint infopoint) {
        if (config.continuationsEnabled && infopoint instanceof Call) {
            // Support for loom requires custom nops after call sites that might deopt
            Call call = (Call) infopoint;
            if (call.debugInfo != null && call.reason == InfopointReason.CALL) {
                // Expected nop pattern taken from src/hotspot/cpu/x86/macroAssembler_x86.cpp in
                // MacroAssembler::post_call_nop(). JVMCI will add a relocation during installation.
                emitByte(0x0f);
                emitByte(0x1f);
                emitByte(0x84);
                emitByte(0x00);
                emitInt(0x00);
                return;
            }
        }
        super.postCallNop(infopoint);
    }

    /**
     * Emit the expected patchable code sequence for the nmethod entry barrier. The int sized
     * payload must be naturally aligned so it can be patched atomically.
     */
    public void nmethodEntryCompare(int displacement) {
        // cmp dword ptr [r15 + <displacement>], 0x00000000
        // 41 81 7f <db> 00 00 00 00
        emitByte(0x41);
        emitByte(0x81);
        emitByte(0x7f);
        GraalError.guarantee(NumUtil.isByte(displacement), "expected byte sized displacement");
        emitByte(displacement & 0xff);
        GraalError.guarantee(position() % 4 == 0, "must be aligned");
        emitInt(0);
    }
}
