/*
 * Copyright (c) 2014, 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.hosted.option;

import org.graalvm.compiler.options.OptionDescriptor;
import org.graalvm.compiler.options.OptionGroup;
import org.graalvm.compiler.options.OptionKey;
import org.graalvm.nativeimage.ImageSingletons;

import com.oracle.svm.core.feature.InternalFeature;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.option.RuntimeOptionParser;
import com.oracle.svm.core.feature.AutomaticallyRegisteredFeature;
import com.oracle.svm.core.util.VMError;

@AutomaticallyRegisteredFeature
public class RuntimeOptionFeature implements InternalFeature {

    private RuntimeOptionParser runtimeOptionParser;

    @Override
    public void afterRegistration(AfterRegistrationAccess access) {
        runtimeOptionParser = new RuntimeOptionParser();
        ImageSingletons.add(RuntimeOptionParser.class, runtimeOptionParser);
    }

    @Override
    public void duringSetup(DuringSetupAccess config) {
        config.registerObjectReplacer(this::collectOptionKeys);
    }

    private Object collectOptionKeys(Object obj) {
        if (obj instanceof OptionKey) {
            OptionKey<?> optionKey = (OptionKey<?>) obj;
            if (!(optionKey instanceof HostedOptionKey)) {
                OptionDescriptor optionDescriptor = optionKey.getDescriptor();
                if (optionDescriptor == null) {
                    throw VMError.shouldNotReachHere("No OptionDescriptor registered for an OptionKey. Often that is the result of an incomplete build. " +
                                    "The registration code is generated by an annotation processor at build time, so a clean and full rebuild often helps to solve this problem");
                }

                OptionGroup group = optionDescriptor.getDeclaringClass().getAnnotation(OptionGroup.class);
                if (group == null || group.registerAsService()) {
                    runtimeOptionParser.addDescriptor(optionDescriptor);
                }
            }
        }
        return obj;
    }
}
