/*
 * Copyright (c) 2019, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.core.jfr;

import java.util.concurrent.TimeUnit;

import com.oracle.svm.core.Uninterruptible;

/**
 * Utility class to manage ticks for event timestamps based on an initial start point when the
 * system initializes.
 */
public final class JfrTicks {
    private static long initialTicks;

    private JfrTicks() {
    }

    public static void initialize() {
        initialTicks = System.nanoTime();
    }

    /**
     * This method returns 0 if the JFR support is either disabled or not yet initialized.
     */
    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    public static long elapsedTicks() {
        if (HasJfrSupport.get() && initialTicks > 0) {
            return System.nanoTime() - initialTicks;
        }
        return 0;
    }

    @Uninterruptible(reason = "Called from uninterruptible code.", mayBeInlined = true)
    public static long duration(long startTicks) {
        return elapsedTicks() - startTicks;
    }

    public static long getTicksFrequency() {
        return TimeUnit.SECONDS.toNanos(1);
    }

    public static long currentTimeNanos() {
        return TimeUnit.MILLISECONDS.toNanos(System.currentTimeMillis());
    }
}
