/*
 * Copyright (c) 2018, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.oracle.svm.core.jdk;

import java.util.Map;

import com.oracle.svm.core.NeverInline;
import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.annotate.Alias;
import com.oracle.svm.core.annotate.Delete;
import com.oracle.svm.core.annotate.InjectAccessors;
import com.oracle.svm.core.annotate.RecomputeFieldValue;
import com.oracle.svm.core.annotate.RecomputeFieldValue.Kind;
import com.oracle.svm.core.annotate.Substitute;
import com.oracle.svm.core.annotate.TargetClass;
import com.oracle.svm.core.heap.PhysicalMemory;
import com.oracle.svm.core.snippets.KnownIntrinsics;

@TargetClass(className = "jdk.internal.misc.VM")
public final class Target_jdk_internal_misc_VM {
    /** Ensure that we do not leak the full set of properties from the image generator. */
    @Delete //
    private static Map<String, String> savedProps;

    @Substitute
    public static String getSavedProperty(String name) {
        return SystemPropertiesSupport.singleton().getInitialProperty(name);
    }

    @Substitute
    @NeverInline("Starting a stack walk in the caller frame")
    public static ClassLoader latestUserDefinedLoader0() {
        return StackTraceUtils.latestUserDefinedClassLoader(KnownIntrinsics.readCallerStackPointer());
    }

    /*
     * Finalizers are not supported, but we still do not want to inherit any counters from the image
     * builder.
     */
    @Alias @RecomputeFieldValue(kind = Kind.Reset) //
    private static int finalRefCount;
    @Alias @RecomputeFieldValue(kind = Kind.Reset) //
    private static int peakFinalRefCount;

    @Alias @InjectAccessors(DirectMemoryAccessors.class) //
    private static long directMemory;
    @Alias @InjectAccessors(PageAlignDirectMemoryAccessors.class) //
    private static boolean pageAlignDirectMemory;
}

final class DirectMemoryAccessors {
    private static final long DIRECT_MEMORY_DURING_INITIALIZATION = 25 * 1024 * 1024;

    /**
     * This field needs to be volatile to ensure that reads emit a LOAD-LOAD barrier. Without this
     * barrier, subsequent reads could be reordered before the read of {@link #initialized},
     * allowing threads to observe an uninitialized value for {@link #directMemory}. We could
     * directly emit a LOAD-LOAD barrier instead, but it doesn't make any difference in terms of the
     * used instructions on any of the relevant CPU architectures.
     */
    private static volatile boolean initialized;
    private static long directMemory;

    static long getDirectMemory() {
        if (!initialized) {
            return tryInitialize();
        }
        return directMemory;
    }

    private static long tryInitialize() {
        /*
         * The JDK method VM.saveAndRemoveProperties looks at the system property
         * "sun.nio.MaxDirectMemorySize". However, that property is always set by the Java HotSpot
         * VM to the value of the option -XX:MaxDirectMemorySize, so we do not need to take that
         * system property into account.
         */
        long newDirectMemory = SubstrateOptions.MaxDirectMemorySize.getValue();
        if (newDirectMemory == 0) {
            /*
             * No value explicitly specified. The default in the JDK in this case is the maximum
             * heap size.
             */
            if (PhysicalMemory.isInitializationInProgress()) {
                /*
                 * When initializing PhysicalMemory, we use NIO/cgroups code that calls
                 * VM.getDirectMemory(). When this initialization is in progress, we need to prevent
                 * that Runtime.maxMemory() is called below because it would trigger a recursive
                 * initialization of PhysicalMemory. So, we return a temporary value.
                 */
                return DIRECT_MEMORY_DURING_INITIALIZATION;
            }
            newDirectMemory = Runtime.getRuntime().maxMemory();
        }

        directMemory = newDirectMemory;
        /* STORE_STORE barrier is executed as part of the volatile write. */
        initialized = true;
        return newDirectMemory;
    }
}

final class PageAlignDirectMemoryAccessors {
    /** See DirectMemoryAccessors#initialized on why this needs to be volatile. */
    private static volatile boolean initialized;
    private static boolean pageAlignDirectMemory;

    static boolean getPageAlignDirectMemory() {
        if (!initialized) {
            initialize();
        }
        return pageAlignDirectMemory;
    }

    private static void initialize() {
        pageAlignDirectMemory = Boolean.getBoolean("sun.nio.PageAlignDirectMemory");
        /* STORE_STORE barrier is executed as part of the volatile write. */
        initialized = true;
    }
}
