/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.jipher.pki.ocsp;

import com.oracle.jipher.pki.internal.AlgIdException;
import com.oracle.jipher.pki.internal.AlgorithmId;
import com.oracle.jipher.pki.internal.Debug;
import com.oracle.jipher.pki.internal.ExtensionHelper;
import com.oracle.jipher.pki.internal.Util;
import com.oracle.jipher.pki.ocsp.CertResponse;
import com.oracle.jipher.pki.ocsp.OcspException;
import com.oracle.jipher.pki.ocsp.OcspRequest;
import com.oracle.jipher.pki.ocsp.OcspResponse;
import com.oracle.jipher.tools.asn1.Asn1;
import com.oracle.jipher.tools.asn1.Asn1BerValue;
import com.oracle.jipher.tools.asn1.Asn1Exception;
import com.oracle.jipher.tools.asn1.UniversalTag;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.security.auth.x500.X500Principal;

public class OcspClient {
    private static final String OID_BASIC_RESPONSE = "1.3.6.1.5.5.7.48.1.1";
    private X509Certificate responderCert;
    private boolean allowNonceIgnore;
    private boolean allowUnsupportedCritical;
    private Debug debug = Debug.getInstance("ocsp");

    public OcspClient() {
    }

    public OcspClient(Flag ... flags) {
        for (Flag f : flags) {
            if (f == Flag.IGNORE_NONCE) {
                this.allowNonceIgnore = true;
                continue;
            }
            if (f != Flag.IGNORE_UNSUPPORTED_CRITICAL_EXT) continue;
            this.allowUnsupportedCritical = true;
        }
    }

    public OcspClient(X509Certificate responderCert, Flag ... flags) {
        this(flags);
        this.responderCert = responderCert;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public OcspResponse queryResponse(HttpURLConnection connection, OcspRequest req) throws IOException, OcspException {
        byte[] rspBytes;
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Content-Type", "application/ocsp-request");
        connection.setDoOutput(true);
        OutputStream out = null;
        InputStream is = null;
        try {
            out = connection.getOutputStream();
            out.write(req.getEncoded());
            this.debug.println(() -> "Sent request bytes to  " + connection);
            int respCode = connection.getResponseCode();
            if (respCode != 200) {
                throw new OcspException("HTTP response was " + respCode);
            }
            is = connection.getInputStream();
            rspBytes = OcspClient.readFromStream(is);
            this.debug.println(() -> "Received response from " + connection);
        }
        finally {
            if (out != null) {
                out.close();
            }
            if (is != null) {
                is.close();
            }
        }
        return this.processResponse(rspBytes, req.getNonce());
    }

    public OcspResponse processResponse(byte[] respBytes, byte[] expectedNonce) throws OcspException {
        try {
            byte[] basicResponseBytes = this.decodeBasicResponse(Asn1.decodeOne(respBytes));
            Asn1BerValue basicRsp = Asn1.decodeOne(ByteBuffer.wrap(basicResponseBytes), true);
            List<Asn1BerValue> basicRspContents = basicRsp.tag(UniversalTag.SEQUENCE).count(3, 4).sequence();
            Asn1BerValue tbsRspData = basicRspContents.get(0);
            AlgorithmId signAlg = AlgorithmId.decode(basicRspContents.get(1));
            byte[] sigBytes = basicRspContents.get(2).getBitStringOctets();
            List<X509Certificate> certs = basicRspContents.size() == 4 ? this.processCerts(basicRspContents.get(3)) : null;
            this.checkSignature(signAlg, certs, tbsRspData.encodeDerOctets(), sigBytes);
            OcspResponse ocspResp = this.processTbsResponseData(tbsRspData, expectedNonce);
            ocspResp.certs = certs;
            return ocspResp;
        }
        catch (AlgIdException | Asn1Exception e) {
            throw new OcspException("Invalid response received", e);
        }
    }

    public OcspResponse processResponse(byte[] respBytes) throws OcspException {
        return this.processResponse(respBytes, null);
    }

    private byte[] decodeBasicResponse(Asn1BerValue ocspResponse) throws OcspException {
        List<Asn1BerValue> rspContents = ocspResponse.count(1, 2).sequence();
        BigInteger status = rspContents.get(0).getEnumerated();
        if (!status.equals(BigInteger.ZERO)) {
            throw new OcspException("Responder error received: " + OcspClient.getErrorMessage(status));
        }
        List<Asn1BerValue> rspBytes = rspContents.get(1).tag(0).explicit().tag(UniversalTag.SEQUENCE).count(2).sequence();
        String oid = rspBytes.get(0).getOid();
        if (!oid.equals(OID_BASIC_RESPONSE)) {
            throw new OcspException("Unsupported response type: " + oid);
        }
        return rspBytes.get(1).getOctetString();
    }

    private OcspResponse processTbsResponseData(Asn1BerValue tbsRspData, byte[] expectedNonce) throws OcspException, AlgIdException {
        OcspResponse ocspResponse = new OcspResponse();
        List<Asn1BerValue> responseDataContents = tbsRspData.count(3, 5).tag(UniversalTag.SEQUENCE).sequence();
        Iterator<Asn1BerValue> iter = responseDataContents.iterator();
        Asn1BerValue next = iter.next();
        if (next.hasTag(0)) {
            BigInteger v = next.explicit().tag(UniversalTag.INTEGER).getInteger();
            next = iter.next();
            if (!v.equals(BigInteger.ZERO)) {
                throw new OcspException("OCSP response version (" + v + ") not supported.");
            }
        }
        this.processResponderId(ocspResponse, next);
        ocspResponse.producedAt = iter.next().tag(UniversalTag.GeneralizedTime).getGeneralizedTime();
        this.processResponses(ocspResponse, iter.next());
        List<Asn1BerValue> extns = iter.hasNext() ? iter.next().explicit().tag(UniversalTag.SEQUENCE).sequence() : null;
        this.processResponseExtns(ocspResponse, extns, expectedNonce);
        return ocspResponse;
    }

    private List<X509Certificate> processCerts(Asn1BerValue certsTagged) throws OcspException {
        ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>();
        List<Asn1BerValue> certValues = certsTagged.tag(0).explicit().tag(UniversalTag.SEQUENCE).sequence();
        for (Asn1BerValue certVal : certValues) {
            X509Certificate crt = this.readCert(certVal);
            certs.add(crt);
        }
        return certs;
    }

    private void processResponses(OcspResponse ocspResponse, Asn1BerValue responseSeq) throws AlgIdException {
        List<Asn1BerValue> responses = responseSeq.tag(UniversalTag.SEQUENCE).sequence();
        ArrayList<CertResponse> certResps = new ArrayList<CertResponse>();
        for (Asn1BerValue rsp : responses) {
            CertResponse certRsp = CertResponse.decode(rsp);
            this.debug.println(() -> "CertResponse " + (Object)((Object)certRsp.getStatus()) + ": id=(" + certRsp.certId + ")");
            certResps.add(certRsp);
        }
        ocspResponse.responses = certResps;
    }

    private void processResponderId(OcspResponse ocspResponse, Asn1BerValue responderId) throws OcspException {
        if (responderId.hasTag(1)) {
            ocspResponse.responderName = new X500Principal(responderId.explicit().encodeDerOctets());
            this.debug.println(() -> "Response from responderId: name=" + ocspResponse.responderName);
        } else if (responderId.hasTag(2)) {
            ocspResponse.responderKeyHash = responderId.explicit().getOctetString();
            this.debug.println(() -> "Response from responderId: keyHash=0x" + Util.toHex(ocspResponse.responderKeyHash));
        } else {
            throw new OcspException("Invalid responderId content in response");
        }
    }

    private void processResponseExtns(OcspResponse ocspResponse, List<Asn1BerValue> extns, byte[] expectedNonce) throws OcspException {
        HashSet<String> nonCritical = new HashSet<String>();
        HashSet<String> critical = new HashSet<String>();
        Map<String, byte[]> allExtns = ExtensionHelper.processExtns(extns, nonCritical, critical);
        if (!allExtns.isEmpty()) {
            ocspResponse.extns = allExtns;
            ocspResponse.nonCriticalExtOids = nonCritical;
            ocspResponse.criticalExtOids = critical;
        }
        this.verifyNonce(allExtns.get("1.3.6.1.5.5.7.48.1.2"), expectedNonce);
        if (ocspResponse.hasUnsupportedCriticalExtension()) {
            if (this.allowUnsupportedCritical) {
                this.debug.println("Ignoring unsupported critical extensions");
            } else {
                this.debug.println(() -> {
                    HashSet<String> crit = new HashSet<String>(ocspResponse.getCriticalExtensionOIDs());
                    crit.remove("1.3.6.1.5.5.7.48.1.2");
                    return "Unsupported critical extensions in response: " + crit;
                });
                throw new OcspException("Response contains unsupported critical extensions.");
            }
        }
    }

    private void verifyNonce(byte[] nonceExtValue, byte[] expectedNonce) throws OcspException {
        if (expectedNonce == null) {
            return;
        }
        if (this.allowNonceIgnore) {
            this.debug.println("Skipping nonce check as per IGNORE_NONCE flag");
            return;
        }
        if (nonceExtValue == null) {
            throw new OcspException("Response did not contain expected nonce.");
        }
        byte[] nonceValueDer = Asn1.decodeOne(nonceExtValue).tag(UniversalTag.OCTET_STRING).getOctetString();
        byte[] nonceBytes = Asn1.decodeOne(nonceValueDer).tag(UniversalTag.OCTET_STRING).getOctetString();
        if (!Arrays.equals(nonceBytes, expectedNonce)) {
            this.debug.println(() -> "Actual nonce (0x" + Util.toHex(nonceBytes) + ") != expected (0x" + Util.toHex(expectedNonce) + ")");
            throw new OcspException("Response did not contain expected nonce.");
        }
    }

    private void checkSignature(AlgorithmId sigAlg, List<X509Certificate> certs, byte[] tbs, byte[] signature) throws OcspException {
        if (this.responderCert == null && (certs == null || certs.isEmpty())) {
            throw new OcspException("No responder cert specified, and response did not contain certificates");
        }
        X509Certificate verCert = this.responderCert != null ? this.responderCert : (Certificate)certs.get(0);
        try {
            Signature verifier = Signature.getInstance(sigAlg.getAlg());
            verifier.initVerify(verCert);
            verifier.update(tbs);
            if (!verifier.verify(signature)) {
                throw new OcspException("Signature verification failed.");
            }
        }
        catch (GeneralSecurityException e) {
            throw new OcspException("Failed to verify signature", e);
        }
    }

    private X509Certificate readCert(Asn1BerValue certDecoded) throws OcspException {
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            return (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(certDecoded.encodeDerOctets()));
        }
        catch (CertificateException e) {
            throw new OcspException("Could not read certificate in response", e);
        }
    }

    private static String getErrorMessage(BigInteger status) {
        switch (status.intValueExact()) {
            case 1: {
                return "Malformed request";
            }
            case 2: {
                return "Internal error";
            }
            case 3: {
                return "Try later";
            }
            case 5: {
                return "Signature required";
            }
            case 6: {
                return "Request unauthorized";
            }
        }
        return "Unknown error";
    }

    private static byte[] readFromStream(InputStream is) throws IOException {
        int len;
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        byte[] buf = new byte[1024];
        while ((len = is.read(buf)) != -1) {
            bout.write(buf, 0, len);
        }
        return bout.toByteArray();
    }

    public static enum Flag {
        IGNORE_NONCE,
        IGNORE_UNSUPPORTED_CRITICAL_EXT;

    }
}

