/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
class Sunburst {

    // class fields are not well supported by Closure: https://github.com/google/closure-compiler/issues/2731

    constructor(width, height, data, valueKind, container, updateCallback) {
        this._width = width;
        this._height = height;
        this._data = data;
        this._valueKind = valueKind;
        this._container = container;
        this._updateCallback = updateCallback;

        this._title = {
            start: {
                x: 0,
                y: 0
            },
            width: width,
            height: 0.15 * height
        };
        this._padding = {
            bottom: 0.05 * this._height
        };
        this._chart = {
            start: {
                x: 0,
                y: this._title.height
            },
            width: width,
            height: height - this._title.height - this._padding.bottom,
        };
        this._chart.center = {
            x: this._chart.start.x + this._chart.width / 2,
            y: this._chart.start.y + this._chart.height / 2
        }
        this._chart.maxRadius = Math.min(this._chart.width, this._chart.height) / 2;
        this._tooltip = {
            width: 175,
            height: 75,
            elements: {
                root: `${this._container}-tooltip`,
                container: `${this._container}-tooltip-container`,
                label: `${this._container}-tooltip-label`,
                value: `${this._container}-tooltip-value`,
                percentage: `${this._container}-tooltip-percentage`,
            },
            offset: {
                x: 35,
                y: -35
            }
        };
    }

    static _setElementStyle(element, style) {
        for (const rule in style) {
            element.style[rule] = style[rule];
        }
    }

    _calculateRelativeDepth(data) {
        return data.depth - this._data.depth;
    }

    _calculateRelativeRatio(data) {
        return data.ratio / this._data.ratio;
    }

    _calculateRadius(depth) {
        const maxDepth = Math.min(this._data.height, Sunburst.MaxDepth);
        const r0 = 0.2 * this._chart.maxRadius;
        const dr = ((maxDepth + 1) * r0 - this._chart.maxRadius) / (maxDepth * (maxDepth + 1));

        return r0 * (depth + 1) - (depth * (depth + 1)) * dr;
    }

    _polarToCartesian(radius, angle) {
        angle = angle - Math.PI / 2; // starting angle offset

        return {
            x: this._chart.center.x + (radius * Math.cos(angle)),
            y: this._chart.center.y + (radius * Math.sin(angle))
        };
    }

    _describeArcPath(depth, ratio, offset) {
        const innerRadius = this._calculateRadius(depth - 1);
        const outerRadius = this._calculateRadius(depth);

        const startAngle = offset * 2 * Math.PI;
        const endAngle = ratio < 1 ? startAngle + ratio * 2 * Math.PI : 0.999999 * 2 * Math.PI; // workaround for full circle arcs

        const innerStart = this._polarToCartesian(innerRadius, endAngle);
        const innerEnd = this._polarToCartesian(innerRadius, startAngle);

        const outerStart = this._polarToCartesian(outerRadius, endAngle);
        const outerEnd = this._polarToCartesian(outerRadius, startAngle);

        const largeArcFlag = endAngle - startAngle <= Math.PI ? 0 : 1;

        return [
            `M${innerStart.x},${innerStart.y}`,
            `A${innerRadius},${innerRadius},0,${largeArcFlag},0,${innerEnd.x},${innerEnd.y}`,
            `L${outerEnd.x},${outerEnd.y}`,
            `A${outerRadius},${outerRadius},0,${largeArcFlag},1,${outerStart.x},${outerStart.y}`,
            `Z`
        ].join("");
    }

    _transformArcText(depth, ratio, offset, textLength) {
        const scale = (textLength, textWidth) => {
            const k = 0.1;
            const a = 10;

            return k * textWidth / (textLength + a) < 1 ? k * textWidth / (textLength + a) : 1;
        };
        const rotationAngle = (angle, isVertical) => {
            angle = angle >= Math.PI / 2 && angle <= Math.PI * 3 / 2 ?  angle - Math.PI : angle; // horizontal alignment
            if (isVertical) {
                angle -= Math.PI / 2;
            }

            return angle * 180 / Math.PI;
        };

        const innerRadius = depth > 0 ? this._calculateRadius(depth - 1) : 0;
        const outerRadius = depth > 0 ? this._calculateRadius(depth) : 0;
        const midRadius = (innerRadius + outerRadius) / 2;

        const startAngle = offset * 2 * Math.PI;
        const endAngle = startAngle + ratio * 2 * Math.PI;
        const midAngle = (startAngle + endAngle) / 2;

        const midCenter = this._polarToCartesian(midRadius, midAngle);

        const arcWidth = (endAngle - startAngle) * midRadius;
        const arcHeight = outerRadius - innerRadius;
        const textWidth = Math.max(arcWidth, arcHeight);

        return [
            `translate(${midCenter.x}, ${midCenter.y})`,
            `scale(${textWidth > 0 ? scale(textLength, textWidth) : 1})`,
            `rotate(${rotationAngle(midAngle, arcWidth < arcHeight)})`
        ].join("");
    }

    _calculateArcCenter(depth, ratio, offset) {
        const innerRadius = this._calculateRadius(depth - 1);
        const outerRadius = this._calculateRadius(depth);
        const startAngle = offset * 2 * Math.PI;
        let endAngle = startAngle + ratio * 2 * Math.PI;
        let midAngle = (startAngle + endAngle) / 2;

        let midRadius = (innerRadius + outerRadius) / 2;
        let midCenter = this._polarToCartesian(midRadius, midAngle);

        return midCenter;
    }

    _drawTitle() {
        const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
        Sunburst._setElementStyle(rect, Sunburst.Style.Title.Rect);
        rect.setAttribute("x", this._title.start.x);
        rect.setAttribute("y", this._title.start.y);
        rect.setAttribute("width", this._title.width);
        rect.setAttribute("height", this._title.height);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        Sunburst._setElementStyle(text, Sunburst.Style.Title.Text);
        text.setAttribute("x", this._title.width / 2);
        text.setAttribute("y", this._title.height / 2);

        if (this._data.parent != null) {
            const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
            labelTSpan.textContent = this._data.label;
            text.prepend(labelTSpan);

            let parent = this._data.parent;
            while (parent != null) {
                const parentLinkTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
                Sunburst._setElementStyle(parentLinkTSpan, Sunburst.Style.Title.Link);
                parentLinkTSpan.textContent = parent.label;

                const parentData = parent;
                parentLinkTSpan.addEventListener("click", function() {
                    this._data = parentData;
                    this.draw();
                    this._updateCallback(this._data, this._valueKind);
                }.bind(this));

                const separatorTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
                separatorTSpan.textContent = "\u2192";

                text.prepend(separatorTSpan);
                text.prepend(parentLinkTSpan);

                parent = parent.parent;
            }
        }

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        g.appendChild(rect);
        g.appendChild(text);

        return g;
    }

    _drawCircle(data) {
        const relativeDepth = this._calculateRelativeDepth(data); // should always be 0
        const relativeRatio = this._calculateRelativeRatio(data); // should always be 1
        const radius = this._calculateRadius(relativeDepth);

        const circle = document.createElementNS("http://www.w3.org/2000/svg", "circle");
        Sunburst._setElementStyle(circle, Sunburst.Style.Arc.Path(1));
        circle.setAttribute("cx", this._chart.center.x);
        circle.setAttribute("cy", this._chart.center.y);
        circle.setAttribute("r", radius);

        const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        labelTSpan.setAttribute("x", 0);
        labelTSpan.setAttribute("y", "-1.5%");
        labelTSpan.textContent = data.label;

        const percentageTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        percentageTSpan.setAttribute("x", 0);
        percentageTSpan.setAttribute("y", "1.5%");
        percentageTSpan.textContent = Utils.toPercentage(relativeRatio, 1);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        Sunburst._setElementStyle(text, Sunburst.Style.Arc.Text(1));
        text.setAttribute("transform", this._transformArcText(relativeDepth, relativeRatio, 0, data.label.length));
        text.appendChild(labelTSpan);
        text.appendChild(percentageTSpan);

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        g.appendChild(circle);
        g.appendChild(text);

        g.addEventListener("mouseenter", function() {
            this._showTooltip(this._chart.center, data);
        }.bind(this));
        g.addEventListener("mouseleave", function() {
            this._hideTooltip();
        }.bind(this));

        if (data.parent != null) {
            g.style.cursor = "pointer";
            g.addEventListener("click", function() {
                this._data = data.parent;
                this.draw();
                this._updateCallback(this._data, this._valueKind);
            }.bind(this));
        } else {
            g.style.cursor = "not-allowed";
        }

        return g;
    }

    _drawArc(data, offset) {
        const relativeDepth = this._calculateRelativeDepth(data);
        const relativeRatio = this._calculateRelativeRatio(data);

        const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
        Sunburst._setElementStyle(path, Sunburst.Style.Arc.Path(relativeRatio));
        path.setAttribute("d", this._describeArcPath(relativeDepth, relativeRatio, offset));

        const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        labelTSpan.setAttribute("x", 0);
        labelTSpan.setAttribute("y", "-1.5%");
        labelTSpan.textContent = data.label;

        const percentageTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        percentageTSpan.setAttribute("x", 0);
        percentageTSpan.setAttribute("y", "1.5%");
        percentageTSpan.textContent = Utils.toPercentage(relativeRatio, 1);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        Sunburst._setElementStyle(text, Sunburst.Style.Arc.Text(relativeRatio));
        text.setAttribute("transform", this._transformArcText(relativeDepth, relativeRatio, offset, data.label.length));
        text.appendChild(labelTSpan);
        text.appendChild(percentageTSpan);

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        g.appendChild(path);
        g.appendChild(text);

        g.addEventListener("mouseenter", function() {
            this._showTooltip(this._calculateArcCenter(relativeDepth, relativeRatio, offset), data);
        }.bind(this));
        g.addEventListener("mouseleave", function() {
            this._hideTooltip();
        }.bind(this));

        if (data.children.length > 0) {
            g.style.cursor = "pointer";
            g.addEventListener("click", function() {
                this._data = data;
                this.draw();
                this._updateCallback(this._data, this._valueKind);
            }.bind(this));
        } else {
            g.style.cursor = "not-allowed";
            g.style.opacity = 0.6;
        }

        return g;
    }

    _drawTooltip() {
        const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
        Sunburst._setElementStyle(rect, Sunburst.Style.Tooltip.Rect);
        rect.setAttribute("id", this._tooltip.elements.container);
        rect.setAttribute("height", this._tooltip.height);

        const labelTspan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        Sunburst._setElementStyle(labelTspan, Sunburst.Style.Tooltip.Label);
        labelTspan.setAttribute("id", this._tooltip.elements.label);
        labelTspan.setAttribute("dy", 0.2 * this._tooltip.height);

        const valueTspan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        Sunburst._setElementStyle(valueTspan, Sunburst.Style.Tooltip.Content);
        valueTspan.setAttribute("id", this._tooltip.elements.value);
        valueTspan.setAttribute("dy", 0.5 * this._tooltip.height);

        const percentageTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        Sunburst._setElementStyle(percentageTSpan, Sunburst.Style.Tooltip.Content);
        percentageTSpan.setAttribute("id", this._tooltip.elements.percentage);
        percentageTSpan.setAttribute("dy", 0.8 * this._tooltip.height);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        Sunburst._setElementStyle(text, Sunburst.Style.Tooltip.Text);
        text.appendChild(labelTspan);
        text.appendChild(valueTspan);
        text.appendChild(percentageTSpan);

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        Sunburst._setElementStyle(g, Sunburst.Style.Tooltip.Container);
        g.setAttribute("id", this._tooltip.elements.root);
        g.appendChild(rect);
        g.appendChild(text);

        return g;
    }

    _showTooltip(center, data) {
        const start = {
            x: center.x + this._tooltip.offset.x,
            y: center.y + this._tooltip.offset.y,
        }
        const width = data.label.length > 15 ? data.label.length * 10 : this._tooltip.width; // Workaround for very long package names

        const container = document.getElementById(this._tooltip.elements.container);
        container.setAttribute("width", width);
        container.setAttribute("x", start.x);
        container.setAttribute("y", start.y);

        const labelTSpan = document.getElementById(this._tooltip.elements.label);
        labelTSpan.setAttribute("x", start.x);
        labelTSpan.setAttribute("y", start.y);
        labelTSpan.setAttribute("dx", width / 2);
        labelTSpan.textContent = data.label;

        const valueTSpan = document.getElementById(this._tooltip.elements.value);
        valueTSpan.setAttribute("x", start.x);
        valueTSpan.setAttribute("y", start.y);
        valueTSpan.setAttribute("dx", width / 2);
        let valueText;
        switch (this._valueKind) {
            case Sunburst.ValueKind.count: {
                valueText = `# of methods: ${data.value}`;
                break;
            }
            case Sunburst.ValueKind.bytecodeSize: {
                valueText = `Bytecode size: ${Utils.toHumanReadableSize(data.value)}`;
                break;
            }
        }
        valueTSpan.textContent = valueText;

        const percentageTSpan = document.getElementById(this._tooltip.elements.percentage);
        percentageTSpan.setAttribute("x", start.x);
        percentageTSpan.setAttribute("y", start.y);
        percentageTSpan.setAttribute("dx", width / 2);
        percentageTSpan.textContent = `Percentage: ${Utils.toPercentage(this._calculateRelativeRatio(data))}`;

        document.getElementById(this._tooltip.elements.root).style.visibility = "visible";
    }

    _hideTooltip() {
        document.getElementById(this._tooltip.elements.root).style.visibility = "hidden";
    }

    draw() {
        const titleGroup = this._drawTitle();

        const chartGroup = document.createElementNS("http://www.w3.org/2000/svg", "g");
        const queue = [];
        queue.push({
            data: [this._data],
            offset: 0
        });
        while (queue.length > 0) {
            const children = queue.shift();
            if (this._calculateRelativeDepth(children.data[0]) == 0) {
                const rootData = children.data[0];
                chartGroup.appendChild(this._drawCircle(rootData));
                if (rootData.children.length > 0) {
                    queue.push({
                        data: rootData.children,
                        offset: 0
                    });
                }
            } else {
                let offset = children.offset;
                const sortedChildrenData = children.data.sort((a, b) => b.ratio - a.ratio);
                for (const childData of sortedChildrenData) {
                    chartGroup.appendChild(this._drawArc(childData, offset));
                    if (childData.children.length > 0 && this._calculateRelativeDepth(childData) < Sunburst.MaxDepth) {
                        queue.push({
                            data: childData.children,
                            offset: offset
                        });
                    }
                    offset += this._calculateRelativeRatio(childData);
                }
            }
        }

        const tooltipGroup = this._drawTooltip();

        const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
        Sunburst._setElementStyle(svg, Sunburst.Style.SVG);
        svg.setAttribute("width", this._width);
        svg.setAttribute("height", this._height);
        svg.appendChild(titleGroup);
        svg.appendChild(chartGroup);
        svg.appendChild(tooltipGroup);

        document.getElementById(this._container).replaceChildren(svg);
    }
}

Sunburst.Style = {
    SVG: {
        background: "white"
    },
    Title: {
        Rect: {
            fill: "white"
        },
        Text: {
            fontSize: "20px",
            fontWeight: "600",
            dominantBaseline: "middle",
            textAnchor: "middle"
        },
        Link: {
            fill: "var(--color-graalvm-dark-green)",
            cursor: "pointer"
        }
    },
    Arc: {
        Path: (ratio) => {
            const low = { r: 117, g: 168, b: 179 }; // #75a8b3
            const high = { r: 4, g: 73, b: 88 }; // #044958
            const fill = {
                r: low.r + ratio * (high.r - low.r),
                g: low.g + ratio * (high.g - low.g),
                b: low.b + ratio * (high.b - low.b),
            };
    
            return {
                fill: `rgb(${fill.r}, ${fill.g}, ${fill.b})`,
                stroke: "white"
            };
        },
        Text: (ratio) => {
            return {
                fill: "white",
                fontSize: ratio > 0.003 ? "14px" : 0,
                fontWeight: 600,
                dominantBaseline: "middle",
                textAnchor: "middle"
            };
        }
    },
    Tooltip: {
        Container: {
            pointerEvents: "none",
            visibility: "hidden"
        },
        Rect: {
            fill: "black",
            fillOpacity: 0.8
        },
        Text: {
            dominantBaseline: "middle",
            textAnchor: "middle"
        },
        Label: {
            fill: "#fff",
            fontSize: "15px",
            fontWeight: 600
        },
        Content: {
            fill: "#fff",
            fontSize: "13px"
        }
    }
};

Sunburst.MaxDepth = 5;

Sunburst.ValueKind = {
    count: "count",
    bytecodeSize: "bytecodeSize"
};
