/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

class MethodBreakdown {
    constructor() {}

    // class fields are not well supported by Closure: https://github.com/google/closure-compiler/issues/2731

    static getInstance() {
        if (!this._instance) {
            this._instance = new MethodBreakdown();
        }
        return this._instance;
    }

    setCountData(countData) {
        this._countData = countData;
    }

    setBytecodeSizeData(bytecodeSizeData) {
        this._bytecodeSizeData = bytecodeSizeData;
    }

    static _limitData(data, threshold) {
        if (threshold == 0) {
            return data;
        }

        const limitedData = { label: data.label, value: data.value, ratio: data.ratio, height: data.height, children: [], depth: data.depth, parent: data.parent };
        for (const child of data.children) {
            if (child.ratio >= threshold) {
                limitedData.children.push(child);
                MethodBreakdown._limitData(child, threshold);
            }
        }

        return limitedData;
    }

    _renderChart(type, origin, threshold, breakdownContainer, listContainer) {
        let data;
        let valueKind;
        switch (type) {
            case MethodBreakdown.Type.count:
                data = MethodBreakdown._limitData(this._countData[origin], threshold);
                valueKind = Sunburst.ValueKind.count;
                break;
            case MethodBreakdown.Type.bytecodeSize:
                data = MethodBreakdown._limitData(this._bytecodeSizeData[origin], threshold);
                valueKind = Sunburst.ValueKind.bytecodeSize;
                break;
        }

        const chart = new Sunburst(document.body.clientWidth, 675, data, valueKind, breakdownContainer, (data, type) => this._updateTable(data, type, listContainer));
        chart.draw();
    }

    _renderTable(type, origin, container) {
        let data;
        switch (type) {
            case MethodBreakdown.Type.count:
                data = this._countData[origin];
                break;
            case MethodBreakdown.Type.bytecodeSize:
                data = this._bytecodeSizeData[origin];
                break;
        }
        this._updateTable(data, type, container);
    }

    _updateTable(data, type, container) {
        const tableElement = document.getElementById(container);

        const valueKindElement = tableElement.querySelector("#package-list-value-kind");
        switch (type) {
            case MethodBreakdown.Type.count:
                valueKindElement.textContent = "Method count";
                break;
            case MethodBreakdown.Type.bytecodeSize:
                valueKindElement.textContent = "Bytecode size";
                break;
        }

        const tableBodyElement = tableElement.querySelector("tbody");
        tableBodyElement.innerHTML = "";

        const childrenData = data.children.sort((a, b) => b.ratio - a.ratio);
        for (const childData of childrenData) {
            const nameElement = document.createElement("td");
            nameElement.classList.add("package-name");
            let childName = childData.label;
            let parentData = childData.parent;
            while (parentData != null && parentData.parent != null) {
                childName = parentData.label + "." + childName;
                parentData = parentData.parent;
            }
            nameElement.textContent = childName;

            const valueElement = document.createElement("td");
            valueElement.style.textAlign = "right";
            valueElement.textContent = (type == MethodBreakdown.Type.bytecodeSize) ? Utils.toHumanReadableSize(childData.value) : childData.value;

            const percentageElement = document.createElement("td");
            percentageElement.style.textAlign = "right";
            percentageElement.textContent = Utils.toPercentage(childData.ratio / data.ratio, 3);

            const rowElement = document.createElement("tr");
            rowElement.appendChild(nameElement);
            rowElement.appendChild(valueElement);
            rowElement.appendChild(percentageElement);

            tableBodyElement.appendChild(rowElement);
        }
    }

    render(type, breakdownContainer, listContainer) {
        const origin = "reachable";
        const threshold = 0.001;
        this._renderChart(type, origin, threshold, breakdownContainer, listContainer);
        this._renderTable(type, origin, listContainer);
    }
}

MethodBreakdown.Type = {
    count: "count",
    bytecodeSize: "bytecodeSize"
};

MethodBreakdown.dataWorker = () => {
    const Origin = {
        reachable: "reachable",
    };

    const createTypesMap = (types) => {
        const typeMap = {};
        for (const type of types) {
            const typeData = {
                name: type.name,
                parts: type.name.split(".")
            };
            typeMap[type.id] = typeData;
        }
        return typeMap;
    };

    const getDataByOrigin = (data, method) => {
        const array = [];
        switch (method.origin) {
            case Origin.reachable: {
                array.push({
                    count: data.count.reachable,
                    bytecodeSize: data.bytecodeSize.reachable
                });
                break;
            }
        }

        return array;
    };

    const calculateRatio = (data) => {
        const ratio = (data, total) => {
            if (total == null) {
                total = data.value;
                data.ratio = 1;
            }
            for (const child of data.children) {
                child.ratio = child.value / total;
                ratio(child, total);
            }
        };
        ratio(data, null);
    }

    const calculateHeight = (data) => {
        if (data.children.length == 0) {
            data.height = 0;
        } else {
            let maxChildHeight = 0;
            for (const child of data.children) {
                const childHeight = calculateHeight(child);
                maxChildHeight = Math.max(childHeight, maxChildHeight);
            }
            data.height = maxChildHeight + 1;
        }

        return data.height;
    }

    this.onmessage = (e) => {
        const typesMap = createTypesMap(e.data.types);

        const countData = {
            reachable: { label: "Total", value: 0, children: [], depth: 0, parent: null },
        };
        const bytecodeSizeData = {
            reachable: { label: "Total", value: 0, children: [], depth: 0, parent: null },
        };
        const allData = {
            count: countData,
            bytecodeSize: bytecodeSizeData
        }
        for (const method of e.data.methods) {
            const originDataArray = getDataByOrigin(allData, method);
            for (const originData of originDataArray) {
                const declaringClassType = typesMap[method.declaringClass];

                originData.count.value += 1;
                originData.bytecodeSize.value += method.bytecodeSize;

                let children = {
                    count: originData.count.children,
                    bytecodeSize: originData.bytecodeSize.children,
                }
                let parent = {
                    count: originData.count,
                    bytecodeSize: originData.bytecodeSize,
                }
                let depth = 1;
                for (const part of declaringClassType.parts) {
                    let countChild = null;
                    for (const child of children.count) {
                        if (child.label == part) {
                            countChild = child;
                            break;
                        }
                    }
                    if (countChild == null) {
                        countChild = { label: part, value: 0, children: [], depth: depth, parent: parent.count };
                        children.count.push(countChild);
                    }
                    countChild.value += 1;

                    let bytecodeSizeChild = null;
                    for (const child of children.bytecodeSize) {
                        if (child.label == part) {
                            bytecodeSizeChild = child;
                            break;
                        }
                    }
                    if (bytecodeSizeChild == null) {
                        bytecodeSizeChild = { label: part, value: 0, children: [], depth: depth, parent: parent.bytecodeSize };
                        children.bytecodeSize.push(bytecodeSizeChild);
                    }
                    bytecodeSizeChild.value += method.bytecodeSize;

                    children = {
                        count: countChild.children,
                        bytecodeSize: bytecodeSizeChild.children
                    };
                    parent = {
                        count: countChild,
                        bytecodeSize: bytecodeSizeChild
                    };
                    depth++;
                }
            }
        }

        for (const origin in countData) {
            calculateRatio(countData[origin]);
            calculateHeight(countData[origin]);
        }

        for (const origin in bytecodeSizeData) {
            calculateRatio(bytecodeSizeData[origin]);
            calculateHeight(bytecodeSizeData[origin]);
        }

        postMessage({
            countData: countData,
            bytecodeSizeData: bytecodeSizeData
        });
    };
};
