/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
class HorizontalStackedBar {

    // class fields are not well supported by Closure: https://github.com/google/closure-compiler/issues/2731

    constructor(width, height, title, total, data, container) {
        this._width = width;
        this._height = height;
        this._total = total;
        this._data = data;
        this._container = container;

        this._title = {
            start: {
                x: 0,
                y: 0
            },
            width: width,
            height: 0.35 * height,
            text: title,
            total: total
        };
        this._padding = {
            bottom: 0.05 * this._height
        };
        this._chart = {
            start: {
                x: 0,
                y: this._title.height
            },
            width: width,
            height: height - this._title.height - this._padding.bottom,
        };
        this._tooltip = {
            width: 200,
            height: 100,
            elements: {
                root: `${this._container}-tooltip`,
                container: `${this._container}-tooltip-container`,
                label: `${this._container}-tooltip-label`,
                value: `${this._container}-tooltip-value`,
                percentage: `${this._container}-tooltip-percentage`,
                details: `${this._container}-tooltip-details`
            }
        };
        this._tooltip.offset = {
            left: {
                x: -100 - this._tooltip.width,
                y: 20 - this._tooltip.height / 2
            },
            right: {
                x: 100,
                y: 20 - this._tooltip.height / 2
            }
        };
    }

    static _setElementStyle(element, style) {
        for (const rule in style) {
            element.style[rule] = style[rule];
        }
    }

    _drawTitle() {
        const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
        HorizontalStackedBar._setElementStyle(rect, HorizontalStackedBar.Style.Title.Rect);
        rect.setAttribute("x", this._title.start.x);
        rect.setAttribute("y", this._title.start.y);
        rect.setAttribute("width", this._title.width);
        rect.setAttribute("height", this._title.height);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        HorizontalStackedBar._setElementStyle(text, HorizontalStackedBar.Style.Title.Text);
        text.setAttribute("x", 0);
        text.setAttribute("y", 0.5 * this._title.height);

        const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        labelTSpan.textContent = this._title.text;
        text.appendChild(labelTSpan);

        const totalTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        HorizontalStackedBar._setElementStyle(totalTSpan, { textAnchor: "end" });
        totalTSpan.setAttribute("x", this._title.start.x + this._title.width);

        totalTSpan.textContent = `${Utils.toHumanReadableSize(this._title.total)} in total`;
        text.appendChild(totalTSpan);

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        g.appendChild(rect);
        g.appendChild(text);

        return g;
    }

    _drawBar(data, offset) {
        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        const ratio = data.value / this._total;
        if (ratio == 0) {
            return g;
        }

        const bar = {
            start: {
                x: this._chart.start.x + offset * this._chart.width,
                y: this._chart.start.y
            },
            width: ratio * this._chart.width,
            height: this._chart.height
        };
        bar.center = {
            x: bar.start.x + bar.width / 2,
            y: bar.start.y + bar.height / 2
        };

        const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
        HorizontalStackedBar._setElementStyle(rect, HorizontalStackedBar.Style.Bar.Rect(ratio));
        rect.setAttribute("width", bar.width);
        rect.setAttribute("height", bar.height);
        rect.setAttribute("x", bar.start.x);
        rect.setAttribute("y", bar.start.y);
        rect.setAttribute("rx", ".75%");

        const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        labelTSpan.setAttribute("dy", "-5%");
        labelTSpan.textContent = data.label;

        const sublabelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        sublabelTSpan.setAttribute("x", 0);
        sublabelTSpan.setAttribute("dy", "15%");
        sublabelTSpan.textContent = `${Utils.toHumanReadableSize(data.value)} (${Utils.toPercentage(ratio, 1)})`;

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        HorizontalStackedBar._setElementStyle(text, HorizontalStackedBar.Style.Bar.Text);
        text.setAttribute("transform", `translate(${bar.center.x}, ${bar.center.y}) scale(${ratio > 0.1 ? 1 : 0.55}) rotate(${ratio > 0.1 ? 0 : -90})`);
        text.appendChild(labelTSpan);
        text.appendChild(sublabelTSpan);

        g.appendChild(rect);
        g.appendChild(text);
        g.addEventListener("mouseenter", function() {
            this._showTooltip(bar.center, data, offset <= 0.5 ? "right" : "left");
        }.bind(this));
        g.addEventListener("mouseleave", function() {
            this._hideTooltip();
        }.bind(this));

        if ("target" in data) {
            g.style.cursor = "pointer";
            g.addEventListener("click", function() {
                activateTab(data.target);
            });
        } else {
            g.style.cursor = "not-allowed";
            g.style.opacity = 0.6;
        }

        return g;
    }

    _drawTooltip() {
        const rect = document.createElementNS("http://www.w3.org/2000/svg", "rect");
        HorizontalStackedBar._setElementStyle(rect, HorizontalStackedBar.Style.Tooltip.Rect);
        rect.setAttribute("id", this._tooltip.elements.container);
        rect.setAttribute("width", this._tooltip.width);
        rect.setAttribute("height", this._tooltip.height);

        const labelTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        HorizontalStackedBar._setElementStyle(labelTSpan, HorizontalStackedBar.Style.Tooltip.Label);
        labelTSpan.setAttribute("id", this._tooltip.elements.label);
        labelTSpan.setAttribute("dx", this._tooltip.width / 2);
        labelTSpan.setAttribute("dy", 0.2 * this._tooltip.height);

        const valueTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        HorizontalStackedBar._setElementStyle(valueTSpan, HorizontalStackedBar.Style.Tooltip.Content);
        valueTSpan.setAttribute("id", this._tooltip.elements.value);
        valueTSpan.setAttribute("dx", this._tooltip.width / 2);
        valueTSpan.setAttribute("dy", 0.45 * this._tooltip.height);

        const percentageTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        HorizontalStackedBar._setElementStyle(percentageTSpan, HorizontalStackedBar.Style.Tooltip.Content);
        percentageTSpan.setAttribute("id", this._tooltip.elements.percentage);
        percentageTSpan.setAttribute("dx", this._tooltip.width / 2);
        percentageTSpan.setAttribute("dy", 0.65 * this._tooltip.height);

        const detailsTSpan = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        HorizontalStackedBar._setElementStyle(detailsTSpan, HorizontalStackedBar.Style.Tooltip.Content);
        detailsTSpan.setAttribute("id", this._tooltip.elements.details);
        detailsTSpan.setAttribute("dx", this._tooltip.width / 2);
        detailsTSpan.setAttribute("dy", 0.85 * this._tooltip.height);

        const text = document.createElementNS("http://www.w3.org/2000/svg", "text");
        HorizontalStackedBar._setElementStyle(text, HorizontalStackedBar.Style.Tooltip.Text);
        text.appendChild(labelTSpan);
        text.appendChild(valueTSpan);
        text.appendChild(percentageTSpan);
        text.appendChild(detailsTSpan);

        const g = document.createElementNS("http://www.w3.org/2000/svg", "g");
        HorizontalStackedBar._setElementStyle(g, HorizontalStackedBar.Style.Tooltip.Container);
        g.setAttribute("id", this._tooltip.elements.root);
        g.appendChild(rect);
        g.appendChild(text);

        return g;
    }

    _showTooltip(center, data, orientation) {
        const start = {
            x: center.x + this._tooltip.offset[orientation].x,
            y: center.y + this._tooltip.offset[orientation].y
        }

        const container = document.getElementById(this._tooltip.elements.container);
        container.setAttribute("x", start.x);
        container.setAttribute("y", start.y);

        const labelTSpan = document.getElementById(this._tooltip.elements.label);
        labelTSpan.setAttribute("x", start.x);
        labelTSpan.setAttribute("y", start.y);
        labelTSpan.textContent = data.label;

        const valueTSpan = document.getElementById(this._tooltip.elements.value);
        valueTSpan.setAttribute("x", start.x);
        valueTSpan.setAttribute("y", start.y);
        valueTSpan.textContent = `Size: ${Utils.toHumanReadableSize(data.value)}`;

        const percentageTSpan = document.getElementById(this._tooltip.elements.percentage);
        percentageTSpan.setAttribute("x", start.x);
        percentageTSpan.setAttribute("y", start.y);
        percentageTSpan.textContent = `Percentage: ${Utils.toPercentage(data.value / this._total)}`;

        const detailsTSpan = document.getElementById(this._tooltip.elements.details);
        detailsTSpan.setAttribute("x", start.x);
        detailsTSpan.setAttribute("y", start.y);
        detailsTSpan.textContent = "details" in data ? data.details : "";

        document.getElementById(this._tooltip.elements.root).style.visibility = "visible";
    }

    _hideTooltip() {
        document.getElementById(this._tooltip.elements.root).style.visibility = "hidden";
    }

    draw() {
        const titleGroup = this._drawTitle();

        const chartGroup = document.createElementNS("http://www.w3.org/2000/svg", "g");
        let offset = 0;
        for (let item of this._data) {
            chartGroup.appendChild(this._drawBar(item, offset));
            offset += item.value / this._total;
        }

        const tooltipGroup = this._drawTooltip();

        const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
        HorizontalStackedBar._setElementStyle(svg, HorizontalStackedBar.Style.SVG);
        svg.setAttribute("width", this._width);
        svg.setAttribute("height", this._height);
        svg.appendChild(titleGroup);
        svg.appendChild(chartGroup);
        svg.appendChild(tooltipGroup);

        document.getElementById(this._container).replaceChildren(svg);
    }
}

HorizontalStackedBar.Style = {
    SVG: {
        background: "white"
    },
    Title: {
        Rect: {
            fill: "white"
        },
        Text: {
            fontSize: "24px",
            dominantBaseline: "middle",
            textAnchor: "start"
        }
    },
    Bar: {
        Rect: (ratio) => {
            const low = { r: 117, g: 168, b: 179 }; // #75a8b3
            const high = { r: 4, g: 73, b: 88 }; // #044958
            const fill = {
                r: low.r + ratio * (high.r - low.r),
                g: low.g + ratio * (high.g - low.g),
                b: low.b + ratio * (high.b - low.b),
            };
    
            return {
                fill: `rgb(${fill.r}, ${fill.g}, ${fill.b})`,
                stroke: "white",
                strokeWidth: "5px"
            };
        },
        Text: {
            fill: "white",
            fontSize: "20px",
            fontWeight: 600,
            dominantBaseline: "middle",
            textAnchor: "middle"
        }
    },
    Tooltip: {
        Container: {
            pointerEvents: "none",
            visibility: "hidden"
        },
        Rect: {
            fill: "black",
            fillOpacity: 0.8
        },
        Text: {
            dominantBaseline: "middle",
            textAnchor: "middle"
        },
        Label: {
            fill: "#fff",
            fontSize: "15px",
            fontWeight: 600
        },
        Content: {
            fill: "#fff",
            fontSize: "13px"
        }
    }
};
