/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

class BuildOutput {
    constructor() {}

    // class fields are not well supported by Closure: https://github.com/google/closure-compiler/issues/2731

    static getInstance() {
        if (!this._instance) {
            this._instance = new BuildOutput();
        }
        return this._instance;
    }

    getData() {
        return this._data;
    }

    setData(data) {
        this._data = data;
    }

    render(detailsContainer) {
        this.renderEnvironment();
        this.renderAnalysisResults();
        this.renderImageDetails(detailsContainer);
        this.renderResourceUsage();
    };

    renderEnvironment() {
        for (const key of ['java_version', 'vendor_version', 'c_compiler', 'garbage_collector']) {
            const elem = document.getElementById(`build-output-environment-${key.replace('_', '-')}`);
            elem.innerHTML = key.includes('version') ? `<span class="version">${this._data.general_info[key]}</span>` : this._data.general_info[key];
        }

        const graalCompiler = this._data.general_info.graal_compiler
        document.getElementById('build-output-environment-graal-compiler-optimization-level').innerText = graalCompiler.optimization_level
        document.getElementById('build-output-environment-graal-compiler-march').innerText = graalCompiler.march
        document.getElementById('build-output-environment-graal-compiler-pgo').innerText = graalCompiler.pgo;
    };

    renderAnalysisResults() {
        const analysisResults = this._data.analysis_results;
        const totalTypes = analysisResults.classes.total;
        const totalFields = analysisResults.fields.total;
        const totalMethods = analysisResults.methods.total;
        document.getElementById('build-output-analysis-results-total-types-count').innerText = totalTypes
        document.getElementById('build-output-analysis-results-total-fields-count').innerText = totalFields
        document.getElementById('build-output-analysis-results-total-methods-count').innerText = totalMethods

        for (const key of ['reachable', 'reflection', 'jni']) {
            const typesCountElem = document.getElementById(`build-output-analysis-results-${key}-types-count`);
            typesCountElem.innerText = Utils.printIfExists(analysisResults.classes[key]);
            const typesPercentageElem = document.getElementById(`build-output-analysis-results-${key}-types-percentage`);
            typesPercentageElem.innerText = Utils.printIfExists(analysisResults.classes[key], Utils.toPercentage(analysisResults.classes[key] / totalTypes));

            const fieldCountElem = document.getElementById(`build-output-analysis-results-${key}-fields-count`);
            fieldCountElem.innerText = Utils.printIfExists(analysisResults.fields[key]);
            const fieldsPercentageElem = document.getElementById(`build-output-analysis-results-${key}-fields-percentage`);
            fieldsPercentageElem.innerText = Utils.printIfExists(analysisResults.fields[key], Utils.toPercentage(analysisResults.fields[key] / totalFields));

            const methodsCountElem = document.getElementById(`build-output-analysis-results-${key}-methods-count`);
            methodsCountElem.innerText = Utils.printIfExists(analysisResults.methods[key]);
            const methodsPercentageElem = document.getElementById(`build-output-analysis-results-${key}-methods-percentage`);
            methodsPercentageElem.innerText = Utils.printIfExists(analysisResults.methods[key], Utils.toPercentage(analysisResults.methods[key] / totalMethods));
        }
    };

    renderImageDetails(container) {
        const details = this._data.image_details;
        const data = [];

        data.push({
            label: "Code area",
            value: details.code_area.bytes,
            details: `${details.code_area.compilation_units} compilation units`,
            target: "#code-area-tab"
        });

        data.push({
            label: "Image heap",
            value: details.image_heap.bytes,
            details: `${Utils.toHumanReadableSize(details.image_heap.resources.bytes)} for ${details.image_heap.resources.count} resources`,
            target: "#image-heap-tab"
        });

        let debugInfoBytes = 0;
        if (details.debug_info) {
            debugInfoBytes = details.debug_info.bytes;
            data.push({
                label: "Debug info",
                value: debugInfoBytes
            });
        }

        const otherBytes = details.total_bytes - details.code_area.bytes - details.image_heap.bytes - debugInfoBytes;
        data.push({
            label: "Other data",
            value: otherBytes
        });

        const chart = new HorizontalStackedBar(document.body.clientWidth, 200, "Image Details", details.total_bytes, data, container);
        chart.draw();
    };

    renderResourceUsage() {
        const resourceUsage = this._data.resource_usage;

        const totalSecsElem = document.getElementById('build-output-resource-usage-total-secs');
        totalSecsElem.innerText = `${resourceUsage.total_secs.toFixed(2)}s`;

        const gcElem = document.getElementById('build-output-resource-usage-gc');
        gcElem.innerText = `${resourceUsage.garbage_collection.total_secs.toFixed(2)}s in ${resourceUsage.garbage_collection.count} GCs`

        const peakRssElem = document.getElementById('build-output-resource-usage-peak-rss');
        peakRssElem.innerText = `${Utils.toHumanReadableSize(resourceUsage.memory.peak_rss_bytes)} (${Utils.toPercentage(resourceUsage.memory.peak_rss_bytes / resourceUsage.memory.system_total)} of ${Utils.toHumanReadableSize(resourceUsage.memory.system_total)} system memory)`

        const cpuLoadElem = document.getElementById('build-output-resource-usage-cpu-load');
        cpuLoadElem.innerText = `${resourceUsage.cpu.load.toFixed(3)} (${Utils.toPercentage(resourceUsage.cpu.load / resourceUsage.cpu.total_cores)} of ${resourceUsage.cpu.total_cores} cores)`
    };
}
