/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.component.installer.model;

import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.graalvm.component.installer.ComponentCollection;
import org.graalvm.component.installer.FailedOperationException;
import org.graalvm.component.installer.Feedback;
import org.graalvm.component.installer.SystemUtils;
import org.graalvm.component.installer.Version;
import org.graalvm.component.installer.model.ComponentInfo;
import org.graalvm.component.installer.model.ManagementStorage;

public final class ComponentRegistry
implements ComponentCollection {
    private final ManagementStorage storage;
    private final Feedback env;
    private boolean allLoaded;
    private Map<String, Collection<String>> fileIndex;
    private Map<String, ComponentInfo> components = new HashMap<String, ComponentInfo>();
    private Map<String, String> graalAttributes;
    private Map<String, Collection<String>> replacedFiles;
    private Set<String> componentDirectories;
    private boolean replaceFilesChanged;
    private boolean allowDistUpdate;
    private static final Pattern JAVA_VERSION_PATTERN = Pattern.compile("((?:1\\.)?[0-9]+)([._].*)?");
    private String overrideEdition;
    private Version graalVer;

    public ComponentRegistry(Feedback env, ManagementStorage storage) {
        this.storage = storage;
        this.env = env;
    }

    public boolean compatibleVersion(Version v) {
        Version gv = this.getGraalVersion();
        if (this.allowDistUpdate) {
            return gv.updatable().equals(v.updatable());
        }
        return gv.onlyVersion().equals(v.installVersion());
    }

    @Override
    public boolean isAllowDistUpdate() {
        return this.allowDistUpdate;
    }

    @Override
    public void setAllowDistUpdate(boolean allowDistUpdate) {
        this.allowDistUpdate = allowDistUpdate;
    }

    @Override
    public ComponentInfo findComponentMatch(String idspec, Version.Match vm, boolean exact) {
        Version.Match[] vmatch = new Version.Match[1];
        String id = Version.idAndVersion(idspec, vmatch);
        if (!this.allLoaded) {
            return this.loadSingleComponent(id, false, false, exact);
        }
        ComponentInfo ci = this.components.get(id);
        if (ci != null) {
            return ci;
        }
        String fullId = exact ? id : this.findAbbreviatedId(id);
        return fullId == null ? null : this.components.get(fullId);
    }

    private String findAbbreviatedId(String id) {
        String candidate = null;
        String lcid = id.toLowerCase(Locale.ENGLISH);
        String end = "." + lcid;
        Collection<String> ids = this.getComponentIDs();
        String ambiguous = null;
        for (String s : ids) {
            String lcs = s.toLowerCase(Locale.ENGLISH);
            if (lcs.equals(lcid)) {
                return s;
            }
            if (!lcs.endsWith(end)) continue;
            if (candidate != null) {
                ambiguous = s;
                continue;
            }
            candidate = s;
        }
        if (ambiguous != null) {
            throw this.env.failure("COMPONENT_AmbiguousIdFound", null, candidate, ambiguous);
        }
        return candidate;
    }

    public void setOverrideEdition(String overrideEdition) {
        this.overrideEdition = overrideEdition;
    }

    public Map<String, String> getGraalCapabilities() {
        if (this.graalAttributes != null) {
            return this.graalAttributes;
        }
        HashMap<String, String> m = new HashMap<String, String>(this.storage.loadGraalVersionInfo());
        String v = (String)m.get("java_version");
        if (v != null) {
            int mv;
            Matcher rm = JAVA_VERSION_PATTERN.matcher(v);
            if (rm.matches()) {
                v = rm.group(1);
            }
            if ((mv = SystemUtils.interpretJavaMajorVersion(v)) < 1) {
                m.remove("java_version");
            } else {
                m.put("java_version", "" + mv);
            }
            this.graalAttributes = m;
        }
        if ((v = SystemUtils.normalizeArchitecture((String)m.get("os_name"), (String)m.get("os_arch"))) != null) {
            m.put("os_arch", v);
        }
        if ((v = SystemUtils.normalizeOSName((String)m.get("os_name"), (String)m.get("os_arch"))) != null) {
            m.put("os_name", v);
        }
        if (this.overrideEdition != null) {
            this.graalAttributes.put("edition", this.overrideEdition);
        }
        return this.graalAttributes;
    }

    @Override
    public Collection<String> getComponentIDs() {
        if (!this.allLoaded) {
            try {
                return this.storage.listComponentIDs();
            }
            catch (IOException ex) {
                throw this.env.failure("REGISTRY_ReadingComponentList", ex, ex.getLocalizedMessage());
            }
        }
        return Collections.unmodifiableCollection(this.components.keySet());
    }

    public void removeComponent(ComponentInfo info) throws IOException {
        this.replaceFilesChanged = false;
        this.buildFileIndex();
        String id = info.getId();
        for (String p : info.getPaths()) {
            Collection<String> compIds = this.fileIndex.get(p);
            if (compIds == null || !compIds.remove(id)) continue;
            this.replaceFilesChanged = true;
            if (!compIds.isEmpty()) continue;
            this.fileIndex.remove(p);
            this.componentDirectories.remove(p);
        }
        if (this.allLoaded) {
            this.components.remove(id);
        }
        this.storage.deleteComponent(id);
        this.updateReplacedFiles();
    }

    public void addComponent(ComponentInfo info) throws IOException {
        this.replaceFilesChanged = false;
        this.buildFileIndex();
        String id = info.getId();
        for (String p : info.getPaths()) {
            Collection compIds = this.fileIndex.computeIfAbsent(p, k -> new HashSet());
            this.replaceFilesChanged |= !compIds.isEmpty();
            compIds.add(id);
            if (!p.endsWith("/")) continue;
            this.componentDirectories.add(p);
        }
        if (this.allLoaded) {
            this.addComponentToCache(info);
        }
        this.storage.saveComponent(info);
        this.updateReplacedFiles();
    }

    private void addComponentToCache(ComponentInfo info) {
        String id = info.getId();
        ComponentInfo old = this.components.put(id, info);
        if (old != null) {
            throw new IllegalStateException("Replacing existing component");
        }
    }

    private void computeReplacedFiles() {
        LinkedHashMap<String, Collection<String>> shared = new LinkedHashMap<String, Collection<String>>();
        ArrayList<String> keys = new ArrayList<String>(this.fileIndex.keySet());
        Collections.sort(keys);
        for (String p : keys) {
            Collection<String> compIds = this.fileIndex.get(p);
            if (compIds == null || compIds.size() <= 1) continue;
            shared.put(p, compIds);
        }
        this.replacedFiles = shared;
    }

    private void updateReplacedFiles() throws IOException {
        if (!this.replaceFilesChanged) {
            return;
        }
        this.computeReplacedFiles();
        this.storage.updateReplacedFiles(this.replacedFiles);
    }

    public List<String> getPreservedFiles(ComponentInfo info) {
        this.buildFileIndex();
        ArrayList<String> result = new ArrayList<String>();
        for (String p : info.getPaths()) {
            Collection<String> compIds = this.fileIndex.get(p);
            if (compIds == null || compIds.size() <= 1) continue;
            result.add(p);
        }
        result.add("lib/installer/components");
        return result;
    }

    public List<String> getOwners(String p) {
        this.buildFileIndex();
        Collection<String> compIds = this.fileIndex.get(p);
        if (compIds == null) {
            return Collections.emptyList();
        }
        ArrayList<String> ret = new ArrayList<String>(compIds);
        Collections.sort(ret);
        return ret;
    }

    public boolean isReplacedFilesChanged() {
        return this.replaceFilesChanged;
    }

    private void loadAllComponents() {
        if (this.allLoaded) {
            return;
        }
        Object currentId = null;
        try {
            Set<String> ids = this.storage.listComponentIDs();
            this.components = new HashMap<String, ComponentInfo>();
            for (String id : ids) {
                this.loadSingleComponent(id, true);
            }
            this.allLoaded = true;
        }
        catch (IOException ex) {
            throw this.env.failure("REGISTRY_ReadingComponentMetadata", ex, currentId, ex.getLocalizedMessage());
        }
    }

    public ComponentInfo loadSingleComponent(String id, boolean filelist) {
        return this.loadSingleComponent(id, filelist, false, false);
    }

    @Override
    public Collection<ComponentInfo> loadComponents(String id, Version.Match selector, boolean filelist) {
        ComponentInfo ci = this.loadSingleComponent(id, filelist);
        return ci == null ? null : Collections.singletonList(ci);
    }

    ComponentInfo loadSingleComponent(String id, boolean filelist, boolean notFoundFailure, boolean exact) {
        String fid;
        String string = fid = exact ? id : this.findAbbreviatedId(id);
        if (fid == null) {
            if (notFoundFailure) {
                throw this.env.failure("REMOTE_UnknownComponentId", null, id);
            }
            return null;
        }
        ComponentInfo info = this.components.get(fid);
        if (info != null) {
            return info;
        }
        String cid = id;
        try {
            Set<ComponentInfo> infos = this.storage.loadComponentMetadata(fid);
            if (infos == null || infos.isEmpty()) {
                if (notFoundFailure) {
                    throw this.env.failure("REMOTE_UnknownComponentId", null, id);
                }
                return null;
            }
            if (infos.size() != 1) {
                throw new IllegalArgumentException("Wrong storage");
            }
            info = (ComponentInfo)infos.iterator().next();
            cid = info.getId();
            if (filelist) {
                this.storage.loadComponentFiles(info);
                this.components.put(cid, info);
            }
        }
        catch (NoSuchFileException ex) {
            return null;
        }
        catch (IOException ex) {
            throw this.env.failure("REGISTRY_ReadingComponentMetadata", ex, id, ex.getLocalizedMessage(), fid);
        }
        return info;
    }

    private void buildFileIndex() {
        if (this.fileIndex != null) {
            return;
        }
        this.loadAllComponents();
        this.componentDirectories = new HashSet<String>();
        this.fileIndex = new HashMap<String, Collection<String>>();
        for (ComponentInfo nfo : this.components.values()) {
            for (String path : nfo.getPaths()) {
                if (path.endsWith("/")) {
                    this.componentDirectories.add(path);
                }
                this.fileIndex.computeIfAbsent(path, k -> new ArrayList()).add(nfo.getId());
            }
        }
        this.computeReplacedFiles();
    }

    public List<String> getReplacedFiles(String path) {
        this.buildFileIndex();
        Collection<String> l = this.replacedFiles.get(path);
        if (l == null) {
            return Collections.emptyList();
        }
        ArrayList<String> ret = new ArrayList<String>(l);
        Collections.sort(ret);
        return ret;
    }

    public Set<String> getComponentDirectories() {
        this.buildFileIndex();
        return this.componentDirectories;
    }

    public String localizeCapabilityName(String s) {
        String dispCapName;
        String capName = "INSTALL_Capability_" + s.toLowerCase();
        try {
            dispCapName = this.env.l10n(capName, new Object[0]);
        }
        catch (MissingResourceException ex) {
            dispCapName = s;
        }
        return dispCapName;
    }

    @Override
    public String shortenComponentId(ComponentInfo info) {
        String id = info.getId();
        if (id.startsWith("org.graalvm")) {
            int l = "org.graalvm".length();
            if (id.length() == l) {
                return "graalvm";
            }
            if (id.charAt(l) != '.' && id.length() > l + 1) {
                return id;
            }
            String shortId = id.substring(l + 1);
            try {
                ComponentInfo reg = this.findComponent(shortId);
                if (reg == null || reg.getId().equals(id)) {
                    return shortId;
                }
            }
            catch (FailedOperationException failedOperationException) {
                // empty catch block
            }
        }
        return id;
    }

    public Date isLicenseAccepted(ComponentInfo info, String id) {
        return this.storage.licenseAccepted(info, id);
    }

    public void acceptLicense(ComponentInfo info, String id, String text) {
        this.acceptLicense(info, id, text, null);
    }

    public void acceptLicense(ComponentInfo info, String id, String text, Date d) {
        try {
            this.storage.recordLicenseAccepted(info, id, text, d);
        }
        catch (IOException ex) {
            this.env.error("ERROR_RecordLicenseAccepted", ex, ex.getLocalizedMessage());
        }
    }

    public Version getGraalVersion() {
        if (this.graalVer == null) {
            this.graalVer = Version.fromString(this.getGraalCapabilities().get("graalvm_version"));
        }
        return this.graalVer;
    }

    public Map<String, Collection<String>> getAcceptedLicenses() {
        return this.storage.findAcceptedLicenses();
    }

    public String licenseText(String licId) {
        return this.storage.licenseText(licId);
    }

    public boolean isMacOsX() {
        return this.storage.loadGraalVersionInfo().get("os_name").toLowerCase().contains("macos");
    }

    public void verifyAdministratorAccess() throws IOException {
        this.storage.saveComponent(null);
    }

    public Set<ComponentInfo> findDependentComponents(ComponentInfo startFrom, boolean recursive) {
        if (startFrom == null) {
            return Collections.emptySet();
        }
        ArrayDeque<String> toSearch = new ArrayDeque<String>();
        toSearch.add(startFrom.getId());
        HashSet<ComponentInfo> result = new HashSet<ComponentInfo>();
        while (!toSearch.isEmpty()) {
            String id = (String)toSearch.poll();
            for (String cid : this.getComponentIDs()) {
                ComponentInfo ci = this.loadSingleComponent(cid, false, false, true);
                if (!ci.getDependencies().contains(id)) continue;
                result.add(ci);
                if (!recursive) continue;
                toSearch.add(ci.getId());
            }
        }
        return result;
    }

    public String getJavaVersion() {
        return this.getGraalCapabilities().get("java_version");
    }

    public ManagementStorage getManagementStorage() {
        return this.storage;
    }

    public int hashCode() {
        int hash = 7;
        hash = 97 * hash + Objects.hashCode(this.storage);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        ComponentRegistry other = (ComponentRegistry)obj;
        return Objects.equals(this.storage, other.storage);
    }
}

