/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.hosted.code;

import com.oracle.svm.core.AlwaysInline;
import com.oracle.svm.core.NeverInline;
import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.Uninterruptible;
import com.oracle.svm.core.feature.AutomaticallyRegisteredImageSingleton;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.os.RawFileOperationSupport;
import com.oracle.svm.core.util.VMError;
import com.oracle.svm.hosted.code.CompilationGraph;
import com.oracle.svm.hosted.meta.HostedMethod;
import java.lang.reflect.AnnotatedElement;
import java.util.Collection;
import java.util.Set;
import java.util.TreeSet;
import jdk.vm.ci.meta.ResolvedJavaMethod;
import org.graalvm.compiler.graph.Node;
import org.graalvm.compiler.nodes.StructuredGraph;
import org.graalvm.compiler.nodes.java.AbstractNewObjectNode;
import org.graalvm.compiler.nodes.java.MonitorEnterNode;
import org.graalvm.compiler.nodes.java.NewMultiArrayNode;
import org.graalvm.compiler.nodes.virtual.CommitAllocationNode;
import org.graalvm.compiler.options.OptionsParser;
import org.graalvm.nativeimage.AnnotationAccess;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.Platform;

@AutomaticallyRegisteredImageSingleton
public final class UninterruptibleAnnotationChecker {
    private final Set<String> violations = new TreeSet<String>();

    private static UninterruptibleAnnotationChecker singleton() {
        return (UninterruptibleAnnotationChecker)ImageSingletons.lookup(UninterruptibleAnnotationChecker.class);
    }

    UninterruptibleAnnotationChecker() {
    }

    public static void checkAfterParsing(ResolvedJavaMethod method, StructuredGraph graph) {
        if (Uninterruptible.Utils.isUninterruptible((AnnotatedElement)method) && graph != null) {
            UninterruptibleAnnotationChecker.singleton().checkNoAllocation(method, graph);
            UninterruptibleAnnotationChecker.singleton().checkNoSynchronization(method, graph);
        }
    }

    public static void checkBeforeCompilation(Collection<HostedMethod> methods) {
        if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
            System.out.println("/* DOT */ digraph uninterruptible {");
        }
        UninterruptibleAnnotationChecker c = UninterruptibleAnnotationChecker.singleton();
        for (HostedMethod method : methods) {
            Uninterruptible annotation = Uninterruptible.Utils.getAnnotation(method);
            CompilationGraph graph = method.compilationInfo.getCompilationGraph();
            c.checkSpecifiedOptions(method, annotation);
            c.checkOverrides(method, annotation);
            c.checkCallees(method, annotation, graph);
            c.checkCallers(method, annotation, graph);
        }
        if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
            System.out.println("/* DOT */ }");
        }
        c.reportViolations();
    }

    private void reportViolations() {
        if (!this.violations.isEmpty()) {
            String message = "Found " + this.violations.size() + " violations of @Uninterruptible usage:";
            for (String violation : this.violations) {
                message = message + System.lineSeparator() + "- " + violation;
            }
            throw VMError.shouldNotReachHere("%s", message);
        }
    }

    private void checkSpecifiedOptions(HostedMethod method, Uninterruptible annotation) {
        if (annotation == null || !UninterruptibleAnnotationChecker.useStrictChecking()) {
            return;
        }
        if (annotation.reason().equals("Called from uninterruptible code.")) {
            if (!annotation.mayBeInlined() && !AnnotationAccess.isAnnotationPresent((AnnotatedElement)method, NeverInline.class)) {
                this.violations.add("Method " + method.format("%H.%n(%p)") + " uses an unspecific reason but prevents inlining into interruptible code. If the method has an inherent reason for being uninterruptible, besides being called from uninterruptible code, then please improve the reason. Otherwise, allow inlining into interruptible callers via 'mayBeInlined = true'.");
            }
            if (annotation.callerMustBe()) {
                this.violations.add("Method " + method.format("%H.%n(%p)") + " uses an unspecific reason but is annotated with 'callerMustBe = true'. Please document in the reason why the callers need to be uninterruptible.");
            }
        } else if (UninterruptibleAnnotationChecker.isSimilarToUnspecificReason(annotation.reason())) {
            this.violations.add("Method " + method.format("%H.%n(%p)") + " uses a reason that is similar to the unspecific reason 'Called from uninterruptible code.'. If the method has an inherent reason for being uninterruptible, besides being called from uninterruptible code, then please improve the reason. Otherwise, use exactly the reason from above.");
        }
        if (annotation.mayBeInlined()) {
            if (!annotation.reason().equals("Called from uninterruptible code.") && !AnnotationAccess.isAnnotationPresent((AnnotatedElement)method, AlwaysInline.class)) {
                this.violations.add("Method " + method.format("%H.%n(%p)") + " is annotated with @Uninterruptible('mayBeInlined = true') which allows the method to be inlined into interruptible code. If the method has an inherent reason for being uninterruptible, besides being called from uninterruptible code, then please remove 'mayBeInlined = true'. Otherwise, use the following reason: 'Called from uninterruptible code.'");
            }
            if (AnnotationAccess.isAnnotationPresent((AnnotatedElement)method, NeverInline.class)) {
                this.violations.add("Method " + method.format("%H.%n(%p)") + " is annotated with conflicting annotations: @Uninterruptible('mayBeInlined = true') and @NeverInline");
            }
            if (annotation.callerMustBe()) {
                this.violations.add("Method " + method.format("%H.%n(%p)") + " is annotated with conflicting options: 'mayBeInlined = true' and 'callerMustBe = true'. If the callers of the method need to be uninterruptible, then it should not be allowed to inline the method into interruptible code.");
            }
        }
        if (!annotation.mayBeInlined() && !annotation.callerMustBe() && AnnotationAccess.isAnnotationPresent((AnnotatedElement)method, AlwaysInline.class)) {
            this.violations.add("Method " + method.format("%H.%n(%p)") + " is annotated with @Uninterruptible and @AlwaysInline. If the method may be inlined into interruptible code, please specify 'mayBeInlined = true'. Otherwise, specify 'callerMustBe = true'.");
        }
    }

    private static boolean isSimilarToUnspecificReason(String reason) {
        return (double)OptionsParser.stringSimilarity((String)"Called from uninterruptible code.", (String)reason) > 0.75;
    }

    private static boolean useStrictChecking() {
        if (SubstrateOptions.AllowVMInternalThreads.getValue().booleanValue()) {
            return true;
        }
        return RawFileOperationSupport.isPresent() && !Platform.includedIn(Platform.LINUX.class);
    }

    private void checkOverrides(HostedMethod method, Uninterruptible methodAnnotation) {
        if (methodAnnotation == null) {
            return;
        }
        for (HostedMethod impl : method.getImplementations()) {
            Uninterruptible implAnnotation = Uninterruptible.Utils.getAnnotation(impl);
            if (implAnnotation != null) {
                if (methodAnnotation.callerMustBe() != implAnnotation.callerMustBe()) {
                    this.violations.add("callerMustBe: " + method.format("%H.%n(%p):%r") + " != " + impl.format("%H.%n(%p):%r"));
                }
                if (methodAnnotation.calleeMustBe() == implAnnotation.calleeMustBe()) continue;
                this.violations.add("calleeMustBe: " + method.format("%H.%n(%p):%r") + " != " + impl.format("%H.%n(%p):%r"));
                continue;
            }
            this.violations.add("method " + method.format("%H.%n(%p):%r") + " is annotated but " + impl.format("%H.%n(%p):%r is not"));
        }
    }

    private void checkCallees(HostedMethod caller, Uninterruptible callerAnnotation, CompilationGraph graph) {
        if (callerAnnotation == null || graph == null) {
            return;
        }
        for (CompilationGraph.InvokeInfo invoke : graph.getInvokeInfos()) {
            Uninterruptible directCallerAnnotation;
            HostedMethod callee = invoke.getTargetMethod();
            if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
                UninterruptibleAnnotationChecker.printDotGraphEdge(caller, callee);
            }
            if ((directCallerAnnotation = Uninterruptible.Utils.getAnnotation(invoke.getDirectCaller())) == null) {
                this.violations.add("Unannotated callee: " + invoke.getDirectCaller().format("%H.%n(%p):%r") + " inlined into annotated caller " + caller.format("%H.%n(%p):%r") + System.lineSeparator() + String.valueOf(invoke.getNodeSourcePosition()));
                continue;
            }
            if (directCallerAnnotation.calleeMustBe()) {
                if (Uninterruptible.Utils.isUninterruptible(callee)) continue;
                this.violations.add("Unannotated callee: " + callee.format("%H.%n(%p):%r") + " called by annotated caller " + caller.format("%H.%n(%p):%r") + System.lineSeparator() + String.valueOf(invoke.getNodeSourcePosition()));
                continue;
            }
            if (!callee.isSynthetic()) continue;
            this.violations.add("Synthetic method " + callee.format("%H.%n(%p):%r") + " cannot be called directly from " + caller.format("%H.%n(%p):%r") + System.lineSeparator() + String.valueOf(invoke.getNodeSourcePosition()) + " because the caller is annotated with '@Uninterruptible(calleeMustBe = false)'.");
        }
    }

    private void checkCallers(HostedMethod caller, Uninterruptible callerAnnotation, CompilationGraph graph) {
        if (callerAnnotation != null || graph == null) {
            return;
        }
        for (CompilationGraph.InvokeInfo invoke : graph.getInvokeInfos()) {
            HostedMethod callee = invoke.getTargetMethod();
            if (!UninterruptibleAnnotationChecker.isCallerMustBe(callee)) continue;
            this.violations.add("Unannotated caller: " + caller.format("%H.%n(%p)") + " calls annotated callee " + callee.format("%H.%n(%p)"));
        }
    }

    private void checkNoAllocation(ResolvedJavaMethod method, StructuredGraph graph) {
        for (Node node : graph.getNodes()) {
            if (!UninterruptibleAnnotationChecker.isAllocationNode(node)) continue;
            this.violations.add("Uninterruptible method " + method.format("%H.%n(%p)") + " is not allowed to allocate.");
        }
    }

    private void checkNoSynchronization(ResolvedJavaMethod method, StructuredGraph graph) {
        for (Node node : graph.getNodes()) {
            if (!(node instanceof MonitorEnterNode)) continue;
            this.violations.add("Uninterruptible method " + method.format("%H.%n(%p)") + " is not allowed to use 'synchronized'.");
        }
    }

    public static boolean isAllocationNode(Node node) {
        return node instanceof CommitAllocationNode || node instanceof AbstractNewObjectNode || node instanceof NewMultiArrayNode;
    }

    private static boolean isCallerMustBe(HostedMethod method) {
        Uninterruptible uninterruptibleAnnotation = Uninterruptible.Utils.getAnnotation(method);
        return uninterruptibleAnnotation != null && uninterruptibleAnnotation.callerMustBe();
    }

    private static boolean isCalleeMustBe(HostedMethod method) {
        Uninterruptible uninterruptibleAnnotation = Uninterruptible.Utils.getAnnotation(method);
        return uninterruptibleAnnotation != null && uninterruptibleAnnotation.calleeMustBe();
    }

    private static void printDotGraphEdge(HostedMethod caller, HostedMethod callee) {
        String calleeColor;
        String callerColor = " [color=black]";
        if (Uninterruptible.Utils.isUninterruptible(caller)) {
            callerColor = " [color=blue]";
            if (!UninterruptibleAnnotationChecker.isCalleeMustBe(caller)) {
                callerColor = " [color=orange]";
            }
        }
        if (Uninterruptible.Utils.isUninterruptible(callee)) {
            calleeColor = " [color=blue]";
            if (!UninterruptibleAnnotationChecker.isCalleeMustBe(callee)) {
                calleeColor = " [color=purple]";
            }
        } else {
            calleeColor = " [color=red]";
        }
        System.out.println("/* DOT */    " + caller.format("<%h.%n>") + callerColor);
        System.out.println("/* DOT */    " + callee.format("<%h.%n>") + calleeColor);
        System.out.println("/* DOT */    " + caller.format("<%h.%n>") + " -> " + callee.format("<%h.%n>") + calleeColor);
    }

    public static class Options {
        public static final HostedOptionKey<Boolean> PrintUninterruptibleCalleeDOTGraph = new HostedOptionKey<Boolean>(false);
    }
}

