package oracle.i18n.demo.obe.gdkshop;

import oracle.i18n.demo.obe.gdkshop.bean.Customer;
import oracle.i18n.demo.obe.gdkshop.bean.Order;
import oracle.i18n.demo.obe.gdkshop.bean.OrderItem;

import oracle.jdbc.OracleConnection;
import oracle.jdbc.OracleResultSet;

import oracle.sql.TIMESTAMPLTZ;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import javax.servlet.http.HttpServletRequest;
import oracle.i18n.servlet.Localizer;

public class ShopAction
{
  public static String PARAM_COMMAND = "command";
  public static String PARAM_ORDER_ID = "orderid";
  public static String PARAM_SORT_DIR = "sortdir";

  private static String m_jdbc_str = null;

  static
  {
    try
    {
      DriverManager.registerDriver(new oracle.jdbc.driver.OracleDriver());
    }
    catch (Exception e)
    {
      throw new IllegalStateException(e.getMessage());
    }
  }

  public static synchronized void init(HttpServletRequest request)
  {
    m_jdbc_str = (String)request.getSession().getServletContext().
      getInitParameter("gdkdemo.jdbccon");
  }

  /**
   * Returns a command from the HTTP request object
   *
   * @param request a HTTP request object
   *
   * @return the command <code>String</code>.
   */
  public static String getCommand(HttpServletRequest request)
  {
    String command = request.getParameter(PARAM_COMMAND);
    if (command != null)
      return command;
    return "";
  }

  /**
   * Returns a order id from the HTTP request object
   *
   * @param request a HTTP request object
   *
   * @return the id <code>String</code>.
   */
  public static long getOrderId(HttpServletRequest request)
  {
    String orderid = request.getParameter(PARAM_ORDER_ID);
    if (orderid != null)
    {
      return new Long(orderid).longValue();
    }
    return 0;
  }

  /**
     */
  public static Comparator getSortDir(HttpServletRequest request)
  {
    String sortdir = request.getParameter(PARAM_SORT_DIR);
    if ("desc".equals(sortdir))
    {
      return OrderItem.sortProdNameDesc;
    }
    return OrderItem.sortProdNameAsc;
  }

  /** 
   * Returns all customers that have order history.
   *
   * @return a list of <code>Customer</code> objects
   *
   * @throws Exception if any error occurs
   */
  public static List /*Customer */ getCustomers() throws Exception
  {
    Connection cn = getConnection();
    PreparedStatement ps = null;
    ResultSet rs = null;
    ps = cn.prepareStatement(
        "SELECT c.cust_first_name, c.cust_last_name, c.nls_language, "
        + "c.nls_territory, c.cust_email, c.credit_limit * ex.rates, e.reg_date, TO_CHAR(e.reg_date) "
        + " FROM customers c, ext_user_profile e, exchange_rates ex "
        + "WHERE c.customer_id = e.ext_user_profile "
        + "AND ex.nls_territory = c.nls_territory AND c.customer_id in "
        + "(SELECT DISTINCT(cc.customer_id) FROM customers cc, orders o "
        + " WHERE o.customer_id = cc.customer_id) ORDER BY c.nls_language,"
        + "c.cust_email"
      );
    rs = ps.executeQuery();
    List userLst = new ArrayList();
    while (rs.next())
    {
      Customer c = new Customer();
      c.custFirstName = rs.getString(1);
      c.custLastName = rs.getString(2);
      c.nlsLanguage = rs.getString(3);
      c.nlsTerritory = rs.getString(4);
      c.custEMail = rs.getString(5);
      c.creditLimit = rs.getDouble(6);
      c.registeredDate = rs.getDate(7);
      c.registeredDateText = rs.getString(8);
      userLst.add(c);
    }
    closeAllObjects(cn, ps, rs);
    return userLst;
  }

  /**
   * Returns all customers that have order history.
   *
   * @return a list of <code>Customer</code> objects
   *
   * @throws Exception if any error occurs
   */
  public static List /*Customer */ get4Customers() throws Exception
  {
    Connection cn = getConnection();
    PreparedStatement ps = null;
    ResultSet rs = null;
    ps = cn.prepareStatement(
        "SELECT c.cust_first_name, c.cust_last_name, c.nls_language, "
        + "c.nls_territory, c.cust_email, c.credit_limit * ex.rates, e.reg_date, TO_CHAR(e.reg_date) "
        + "FROM customers c, ext_user_profile e, exchange_rates ex "
        + "WHERE c.customer_id = e.customer_id AND "
        + "c.customer_id >= 9000 AND ex.nls_territory = c.nls_territory "
        + "ORDER BY c.cust_first_name"
      );
    rs = ps.executeQuery();
    List userLst = new ArrayList();
    while (rs.next())
    {
      Customer c = new Customer();
      c.custFirstName = rs.getString(1);
      c.custLastName = rs.getString(2);
      c.nlsLanguage = rs.getString(3);
      c.nlsTerritory = rs.getString(4);
      c.custEMail = rs.getString(5);
      c.creditLimit = rs.getDouble(6);
      c.registeredDate = rs.getDate(7);
      c.registeredDateText = rs.getString(8);
      userLst.add(c);
    }
    closeAllObjects(cn, ps, rs);
    return userLst;
  }

  /**
   * Returns a customer given user id
   *
   * @param userid the user email address
   *
   * @return a <code>Customer</code> object
   *
   * @throws Exception if any error occurs
   */
  public static Customer getCustomer(String userid) throws Exception
  {
    Connection cn = getConnection();
    PreparedStatement ps = null;
    ResultSet rs = null;
    ps = cn.prepareStatement(
        "SELECT c.cust_first_name, c.cust_last_name, c.nls_language, "
        + "c.nls_territory, c.cust_email, c.credit_limit * ex.rates, e.reg_date, TO_CHAR(e.reg_date) "
        + "FROM customers c, ext_user_profile e, exchange_rates ex "
        + "WHERE c.customer_id = e.customer_id "
        + "AND ex.nls_territory = c.nls_territory "
        + "AND LOWER(c.cust_email) = LOWER(?)"
      );
    ps.setString(1, userid);
    rs = (OracleResultSet) ps.executeQuery();
    Customer c = null;
    while (rs.next())
    {
      c = new Customer();
      c.custFirstName = rs.getString(1);
      c.custLastName = rs.getString(2);
      c.nlsLanguage = rs.getString(3);
      c.nlsTerritory = rs.getString(4);
      c.custEMail = rs.getString(5);
      c.creditLimit = rs.getDouble(6);
      c.registeredDate = rs.getDate(7);
      c.registeredDateText = rs.getString(8);
    }
    closeAllObjects(cn, ps, rs);
    return c;
  }

  /**
   * Returns the list of orders associated with the given user
   *
   * @param userid the user id
   *
   * @return the list of <code>Order</code> objects
   *
   * @throws Exception if any error occurs
   */
  public static List /* Order */ getOrders(String userid, Localizer localizer)
    throws Exception
  {
    Connection cn = getConnection();
    TimeZone tz = TimeZone.getDefault();
    ((OracleConnection) cn).setSessionTimeZone(tz.getID());
    PreparedStatement ps = null;
    OracleResultSet rs = null;
    ps = cn.prepareStatement(
        "SELECT o.order_id, o.order_date, o.order_mode, o.customer_id, "
        + "o.order_status, o.order_total * e.rates, "
        + "TO_CHAR(o.order_date, 'DD-MON-RR HH12:MI:SS AM') "
        + "FROM orders o, customers c, exchange_rates e "
        + "WHERE o.customer_id = c.customer_id AND "
        + "e.nls_territory = ? AND "
        + "LOWER(c.cust_email) = LOWER(?) ORDER BY order_date DESC"
      );
      ps.setString(1, localizer.getOraTerritory().toUpperCase(Locale.US));
    ps.setString(2, userid);
    rs = (OracleResultSet) ps.executeQuery();
    List orderLst = new ArrayList();
    while (rs.next())
    {
      Order od = new Order();
      od.orderId = rs.getLong(1);
      TIMESTAMPLTZ tsltz = rs.getTIMESTAMPLTZ(2);
      od.orderDate = (Date) tsltz.timestampValue(cn, new GregorianCalendar(tz));
      od.orderMode = rs.getString(3);
      od.customerId = rs.getLong(4);
      od.orderStatus = rs.getInt(5);
      od.orderTotal = rs.getDouble(6);
      od.orderDateText = rs.getString(7);
      orderLst.add(od);
    }
    closeAllObjects(cn, ps, rs);
    return orderLst;
  }

  /**
   * Returns a order object associated with the order id
   *
   * @param orderid the order id
   *
   * @return a <code>Order</code> object
   *
   * @throws Exception if any error occurs
   */
  public static Order getOrder(long orderid, Localizer localizer) throws Exception
  {
    Connection cn = getConnection();
    TimeZone tz = TimeZone.getDefault();
    ((OracleConnection) cn).setSessionTimeZone(tz.getID());
    PreparedStatement ps = null;
    OracleResultSet rs = null;
    ps = cn.prepareStatement(
        "SELECT o.order_id, o.order_date, o.order_mode, o.customer_id, "
        + "o.order_status, o.order_total * e.rates, "
        + "TO_CHAR(o.order_date, 'DD-MON-RR HH12:MI:SS AM') "
        + "FROM orders o, exchange_rates e "
        + "WHERE o.order_id = ? AND e.nls_territory = ?"
      );
    ps.setLong(1, orderid);
    ps.setString(2, localizer.getOraTerritory().toUpperCase(Locale.US));
    rs = (OracleResultSet) ps.executeQuery();
    List orderLst = new ArrayList();
    Order od = null;
    while (rs.next())
    {
      od = new Order();
      od.orderId = rs.getLong(1);
      TIMESTAMPLTZ tsltz = rs.getTIMESTAMPLTZ(2);
      od.orderDate = (Date) tsltz.timestampValue(cn, new GregorianCalendar(tz));
      od.orderMode = rs.getString(3);
      od.customerId = rs.getLong(4);
      od.orderStatus = rs.getInt(5);
      od.orderTotal = rs.getDouble(6);
      od.orderDateText = rs.getString(7);
    }
    closeAllObjects(cn, ps, rs);
    return od;
  }

  /**
   * Returns the list of order items associated with the order id
   *
   * @param orderId the order id
   * @param sortComp sorting <code>Comparator</code> object
   * @param localizer the <code>Localizer</code> object
   *
   * @return the list of <code>OrderItem</code> objects
   *
   * @throws Exception if any error occurs
   */
  public static List /* OrderItem */ getOrderItems(long orderId,
    Comparator sortComp, Localizer localizer
  ) throws Exception
  {
    Connection cn = getConnection();
    PreparedStatement ps = null;
    ResultSet rs = null;
    ps = cn.prepareStatement("SELECT oi.order_id, oi.product_id, "
        + "pd.translated_name, pd.translated_description, "
        + "oi.unit_price * e.rates, oi.quantity FROM order_items oi, "
        + "product_information p, product_descriptions pd, exchange_rates e "
        + "WHERE oi.order_id = ? AND p.product_id = oi.product_id AND "
        + "pd.product_id = p.product_id AND pd.language_id = ? AND "
        + "e.nls_territory = ?"
      );
    ps.setLong(1, orderId);
    String lang = localizer.getOraShortLanguage().toUpperCase(Locale.US);
    String terr = localizer.getOraTerritory().toUpperCase(Locale.US);
    ps.setString(2, lang);
    ps.setString(3, terr);
    rs = (OracleResultSet) ps.executeQuery();
    List orderItemLst = new ArrayList();
    while (rs.next())
    {
      OrderItem odit = new OrderItem();
      odit.orderId = rs.getLong(1);
      odit.productId = rs.getLong(2);
      odit.productName = rs.getString(3);
      odit.productDescription = rs.getString(4);
      odit.unitPrice = rs.getDouble(5);
      odit.quantity = rs.getInt(6);
      orderItemLst.add(odit);
    }
    closeAllObjects(cn, ps, rs);
    Collections.sort(orderItemLst, sortComp);
    return orderItemLst;
  }

  /**
   * Returns the connection object
   *
   * @return Connection object
   *
   * @throws SQLException if any SQL error occurs
   */
  public static Connection getConnection() throws SQLException
  {
    Connection con = DriverManager.getConnection(m_jdbc_str, "oe", "oe");
    return con;
  }

  /**
   * Closes all database objects
   *
   * @param cn Connection object
   * @param ps PreparedStatement object
   * @param rs ResultSet object
   *
   * @throws IllegalStateException if any error occurs
   */
  public static void closeAllObjects(Connection cn, PreparedStatement ps,
    ResultSet rs
  )
  {
    try
    {
      if (rs != null)
      {
        rs.close();
      }
      if (ps != null)
      {
        ps.close();
      }
      if (cn != null)
      {
        cn.close();
      }
    }
    catch (SQLException e)
    {
      throw new IllegalStateException(e.getMessage());
    }
  }
}
