Rem quickstart_relational.sql
Rem
Rem Copyright (c) 2013, 2014, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem    NAME
Rem      quickstart_relational.sql - Image PL/SQL Package Quickstart Script
Rem
Rem    DESCRIPTION
Rem      SQL script for the Oracle Multimedia Image PL/SQL
Rem      Package Quick Start Guide.      
Rem
Rem    NOTES
Rem      This script should be run from a user that has create any table 
Rem      and create any procedure privileges.

set serveroutput on;

create table image_blob_table (id            number primary key, 
                               width         integer,
                               height        integer,
                               contentLength integer, 
                               mimeType      varchar2(20),
                               image_blob    BLOB)
lob(image_blob) store as securefile;

-- imports the file named <filename> in IMAGEDIR into image_blob_table in a new 
-- row. The new row has the id identifier set to <dest_id>
create or replace procedure image_blob_import(dest_id number, filename varchar2) is
  img_blob BLOB;
begin
   delete from image_blob_table where id = dest_id;
   insert into image_blob_table (id, image_blob) values (dest_id, empty_blob())
     returning image_blob into img_blob;

   ORDSYS.ORD_IMAGE.importFrom(img_blob, 'file', 'IMAGEDIR', filename);
   update image_blob_table set image_blob=img_blob where id=dest_id;
end;
/

call image_blob_import(1,'flowers.jpg');
call image_blob_import(2,'goats.gif');


------------------------------------------------
-- Populating height, width, contentLength and
-- mimeType columns in image_blob_table
------------------------------------------------
-- sets the height, width, contentLength and mimeType columns for all rows
-- in the image_blob_table that have null mimetypes using the 
-- ORDSYS.ORD_IMAGE.getProperties() procedure
create or replace procedure image_blob_getproperties is
  img_mimeType             varchar2(32);
  img_width                integer;
  img_height               integer;
  img_contentLength        integer;
  unused_fileFormat        varchar2(32);
  unused_contentFormat     varchar2(32);
  unused_compressionFormat varchar2(32);
begin
 for rec in (select id, image_blob from image_blob_table where mimeType is null) loop
     ORDSYS.ORD_IMAGE.getProperties(rec.image_blob, 
                                   img_mimeType,
                                   img_width, 
                                   img_height, 
                                   unused_fileFormat, 
                                   unused_contentFormat, 
                                   unused_compressionFormat,
                                   img_contentLength);
     update image_blob_table 
       set width=img_width, 
           height=img_height,
           contentLength=img_contentLength,
           mimeType = img_mimeType
       where id=rec.id;
  end loop;
  commit;
end;
/

call image_blob_getProperties();

------------------------------------------------
-- Selecting and Viewing Image Properties
------------------------------------------------
select id, height, width, mimeType, contentLength from image_blob_table;

------------------------------------------------
-- Creating thumbnails
------------------------------------------------
-- Adds a new row to image_blob_table with id=<dest_id> which includes a
-- new image in the image_blob column. The new image is created using the
-- image from the source row with id=<source_id> and using the
-- ORDSYS.ORD_IMAGE.thumbnail() procedure. The destination image will
-- contain the output blob will contain an image with a predefined 
-- size of 80x80 at most.
create or replace procedure image_blob_thumbnail(source_id number, dest_id number) is
  src_blob BLOB;
  dst_blob BLOB;
begin
  delete from image_blob_table where id = dest_id;
  insert into image_blob_table(id, image_blob) 
    values (dest_id, empty_blob());
  select image_blob into src_blob from image_blob_table
   where id = source_id;
  select image_blob into dst_blob from image_blob_table
    where id = dest_id  for update;
  -- Process the data
  ORDSYS.ORD_IMAGE.thumbnail(src_blob,dst_blob);
  update image_blob_table set image_blob = dst_blob where id = dest_id;
end;
/

-- Convert flowers.jpg to a thumbnail image with a predefined size of 80x80
-- at most
call image_blob_thumbnail(1,3);

------------------------------------------------
-- Changing Formats
------------------------------------------------
-- Adds a a new row to image_blob_table with id=<dest_id> which includes a
-- new image in the image_blob column. The new image is created using the
-- image from the source row with id=<source_id> and using 
-- the ORDSYS.ORD_IMAGE.convert() procedure. The destination image will 
-- be encoded with the file format specified by the fileformat parameter.
create or replace procedure image_blob_convert(source_id number, dest_id number, fileformat varchar2) is
  src_blob BLOB;
  dst_blob BLOB;
begin
  delete from image_blob_table where id = dest_id;
  insert into image_blob_table(id, image_blob)
    values (dest_id, empty_blob());
  select image_blob into src_blob from image_blob_table
   where id = source_id;
  select image_blob into dst_blob from image_blob_table
    where id = dest_id  for update;
   ORDSYS.ORD_IMAGE.convert(src_blob,fileformat,dst_blob);
  update image_blob_table set image_blob = dst_blob where id = dest_id;
end;
/
show errors;


--Convert flowers.jpg to flowers.png and store the resulting "image in
--a row of the image_blob_table with id=4"
call image_blob_convert(1,4,'PNGF');

-- Adds a new row to image_blob_table with id=<dest_id> which includes a
-- new image in the image_blob column.  The new image is created using the
-- image from the source row with id=<source_id> and processing it using
-- the ORDSYS.ORD_IMAGE.processCopy() procedure and passing the command
-- string specified by <verb>.
-- NOTE: assumes <source_id> not equal to <dest_id>
create or replace procedure image_blob_processCopy(source_id number, dest_id number, verb varchar2) is
    src_blob BLOB;
    dst_blob BLOB;
begin
  delete from image_blob_table where id = dest_id;
  insert into image_blob_table (id, image_blob) 
    values (dest_id, empty_blob());

  select image_blob into src_blob 
    from image_blob_table 
    where id = source_id;
  select image_blob into dst_blob
    from image_blob_table 
    where id = dest_id for update;

  ORDSYS.ORD_IMAGE.processCopy(src_blob, verb, dst_blob);
  update image_blob_table set image_blob = dst_blob where id = dest_id;
end;
/

-- Scale flowers.jpg to 10% into row with id=5
call image_blob_processcopy(1,5,'scale=.1');

-- convert goats.gif to a grayscale jpeg thumbnail with a maximum size
-- of 75x100 pixels and store the results into the row with id=6
call image_blob_processcopy(2,6,'fileformat=jfif contentformat=8bitgray maxscale=75 100');

-- update the metadata for the newly created image rows
call image_blob_getProperties();

-- admire our handiwork
select id, height, width, mimeType, contentLength from image_blob_table;

------------------------------------------------
-- Applying watermark to a BLOB
------------------------------------------------
-- Adds a new row to image_blob_table with id=<dest_id> which includes a
-- new image in the image_blob column.  The new image is created using the
-- image from the source row with id=<source_id> and applying a Text watermark
-- to the image using ORDSYS.ORD_IMAGE.applyWatermark()

create or replace procedure image_blob_addwatermark(source_id number, dest_id number) is
  src_blob BLOB;
  added_text varchar2(200);
  dest_blob BLOB;
  prop ordsys.ord_str_list;
  logging VARCHAR2(2000);
begin
delete from image_blob_table where id = dest_id;
insert into image_blob_table (id, image_blob)
values (dest_id, empty_blob());

select image_blob into src_blob from image_blob_table
where id = source_id;
select image_blob into dest_blob from image_blob_table
where id = dest_id for update;
  
  added_text := 'Oracle Multimedia  2013';
  -- specify properties of text watermark
  prop := ordsys.ord_str_list(
    'font_name=Times New Roman',
    'font_style=bold',
    'font_size=50',
    'text_color=red',
    'position_x=100',
    'position_y=100',
    'transparency=0.6');
 
 -- add text watermark to source image and generate new destination image
  ORDSYS.ORD_IMAGE.applyWatermark(src_blob, added_text, dest_blob, logging, prop);
  update image_blob_table set image_blob = dest_blob where id = dest_id;
  commit;
end;
/

call image_blob_addwatermark(1,7);

------------------------------------------------
-- Exporting Images with ORDSYS.ORD_IMAGE.export()
------------------------------------------------
-- Writes the image data from the BLOB with id=<source_id> in image_blob_table
-- to the file named <filename> in the IMAGEDIR directory
create or replace procedure image_blob_export(source_id number, filename varchar2) as
  img_blob BLOB;
begin
  select image_blob into img_blob from image_blob_table where id = source_id;
  ORDSYS.ORD_IMAGE.export(img_blob, 'IMAGEDIR', filename);
end;
/

call image_blob_export(3, 'flowers_thumbnail.jpg');
call image_blob_export(6, 'goats_fixedscale.jpg');

------------------------------------------------------
-- Adding Metadata with ORDSYS.ORD_IMAGE.putMetadata()
------------------------------------------------------
-- Embeds XMP metadata into a blob image stored at id=<source_id> and
-- saves the resulting blob at id=<dest_id>

create or replace procedure image_blob_addmetadata(source_id number, dest_id number) is
  src_blob BLOB;
  dst_blob BLOB;
  xmlData XMLType;
begin
  select image_blob into src_blob from image_blob_table where id = source_id;
  delete from image_blob_table where id = dest_id;
  insert into image_blob_table(id, image_blob) values(dest_id, empty_blob());
  select image_blob into dst_blob from image_blob_table where id = dest_id for update;

  -- Create a valid XML packet to embed in the image
  xmlData := xmltype(
  '<xmpMetadata xmlns="http://xmlns.oracle.com/ord/meta/xmp">' ||
  '<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"' ||
  ' xmlns:dc="http://purl.org/dc/elements/1.1/">' ||
  '<dc:rights>' ||
  ' <rdf:Alt>' ||
  ' <rdf:li xml:lang="en-us">' ||
  ' Oracle Corporation' ||
  ' </rdf:li>' ||
  ' </rdf:Alt>'||
  '</dc:rights>' ||
  '</rdf:RDF>' ||
  '</xmpMetadata>', 'http://xmlns.oracle.com/ord/meta/xmp');

  -- Insert the metadata and update the image
  ORDSYS.ORD_IMAGE.putMetadata(src_blob, dst_blob, xmlData, 'xmp', 'utf-8');
  update image_blob_table
       set image_blob = dst_blob where id = dest_id;
  commit;
end;
/


-- Embed metadata into flowers.jpg
call image_blob_addmetadata(1,8);

-- In order to verify that we have added metadata to the blob image, we retrieve
-- it from the database using ORDSYS.ORD_IMAGE.getMetadata().

create or replace procedure image_blob_extractmetadata(src_id number) as 
  metav XMLSequenceType;
  cursor xmlToString(x XMLSequenceType) is
  select value(list_of_values).getstringval() metadata from table(x) list_of_values;
  tmp varchar(4000);
  dest_blob BLOB;
begin
  select image_blob into dest_blob from image_blob_table where id = src_id;

  metav := ORDSYS.ORD_IMAGE.getMetadata(dest_blob,'ALL');
  open xmlToString(metav);
  loop
    fetch xmlToString into tmp;
    dbms_output.put_line('xmlExtracted: ' || tmp);
    exit when xmlToString%NOTFOUND;
  end loop;
  close xmlToString;
end;
/


-- Display extracted metadata for flowers.jpg
call image_blob_extractmetadata(8); 


