
/* Copyright (c) 2013, 2014, Oracle and/or its affiliates. 
All rights reserved.*/

/*
   DESCRIPTION
    Example code to demonstrate the use of Oracle Multimedia image 
    PL/SQL package and procedures from a java program including thumbnail,
    convert, createWatermark, etc.

   NOTES
    Usage: quickstart_image [connect string] [id]
		[source filename] [id] [destination filename]
 */

import java.io.Console;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.pool.OracleDataSource;
import oracle.jdbc.OracleConnection;
import oracle.jdbc.OracleTypes;
import oracle.sql.BLOB;

public class quickstart_image
{
	protected OracleConnection connection;

	public static void main(String[] args) 
	{
		try
		{
			if(args.length != 5)
			{
				System.err.println( 
					"Usage: quickstart_image [connect string] [id]"
					+" [source filename] [id] [destination filename]");
				System.exit(1);
			}

			//Retrieve the arguments.
			final String connectURI = args[0];
			final Integer id= Integer.parseInt(args[1]); 
			final String srcFile = args[2];
			Integer dst_id = Integer.parseInt(args[3]);
			final String dstFile = args[4];

			//Instantiates a quickstart_image object.
			quickstart_image quickstart = new quickstart_image();
			
			//Create the connection.
			quickstart.getConnection(connectURI);

			//Write data from a local file into a BLOB in the database.
			quickstart.writeImageToDatabase(id, srcFile);

			System.out.println("Original image properties");
			HashMap<String, Object> attributesMap= quickstart.getProperties_example_j(id);
			//Iterate over the HashMap.
			for (Map.Entry<String, Object> entry : attributesMap.entrySet()) { 
				System.out.println(entry.getKey() + " = " + entry.getValue());
			}

			//Create a thumbnail.
			quickstart.thumbnail_example_j(id, dst_id);

			//Print properties to show results.
			System.out.println("\nThumbnail properties:");
			attributesMap= quickstart.getProperties_example_j(dst_id);
			for (Map.Entry<String, Object> entry : attributesMap.entrySet()) {
				System.out.println(entry.getKey() + " = " + entry.getValue());
			}	

			//Change the file format to PNG.
			quickstart.convert_example_j(id,++dst_id, "PNGF"); 

			System.out.println("\nConverted image properties:");
			attributesMap= quickstart.getProperties_example_j(dst_id);
			for (Map.Entry<String, Object> entry : attributesMap.entrySet()) {
				System.out.println(entry.getKey() + " = " + entry.getValue());
			}

			//Crop the original image to the specified rectangle (200x300)
			//from the top-left corner (0,0).
			quickstart.crop_example_j(id, ++dst_id, 0, 0, 200, 300);

			System.out.println("\nCropped image properties");
			attributesMap= quickstart.getProperties_example_j(dst_id);
			for (Map.Entry<String, Object> entry : attributesMap.entrySet()) {
				System.out.println(entry.getKey() + " = " + entry.getValue());
			}

			//Rotate the image 180 degrees.
			quickstart.rotate_example_j(id, ++dst_id, 180);

			//Use the processCopy method to create a new image using two 
			//image processing operations.
			quickstart.processCopy_example_j(id, ++dst_id,
					"fileformat = jfif fixedscale= 75 100");

			System.out.println("\nNew processed image properties");
			attributesMap= quickstart.getProperties_example_j(dst_id);
			for (Map.Entry<String, Object> entry : attributesMap.entrySet()) {
				System.out.println(entry.getKey() + " = " + entry.getValue());
			}

			//Apply text watermarks("ORACLE") onto the original image
			quickstart.applyWatermark_example_j(id,++dst_id,"ORACLE");

			//Retrieve the image we just created to create a local file.
			quickstart.readImageFromDatabase(dst_id, dstFile);

			//Embed XMP metadata into an image.
			final String XMPmetadata =
				"<xmpMetadata xmlns=\"http://xmlns.oracle.com/ord/meta/xmp\"> "
				+ "<rdf:RDF xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\""
				+ " xmlns:dc=\"http://purl.org/dc/elements/1.1/\"> "
				+ "<dc:rights> "
				+ "<rdf:Alt> "
				+ "<rdf:li xml:lang=\"en-us\"> "
				+ "Oracle Corporation 2014"
				+ "</rdf:li>"
				+ "</rdf:Alt>"
				+ "</dc:rights>"
				+ "</rdf:RDF>"
				+ "</xmpMetadata>";
			quickstart.putMetadata_example_j(id, ++dst_id, XMPmetadata);

			//Retrieve the metadata from the image we just created.
			String metadata  = quickstart.getMetadata_example_j(dst_id);
			System.out.println(metadata);

			//Commit and close the connection to the DB.
			quickstart.closeConnection();

		} catch (NumberFormatException e) {
			System.err.println(e.getLocalizedMessage()
					+ " do not have the appropriate numeric format.");
			System.exit(1);
		} 
		catch (SQLException e) 
		{
			e.printStackTrace();
		} 
		catch (IOException e) 
		{
			e.printStackTrace();
		}
	}
	
	/**
	 * Name: getConnection
	 * Description: Open the database connection using the provided connection 
	 * String.
	 *
	 * @param 	ConnectURI: the URL from which connections have to be obtained.
	 */
	public void getConnection(String connectURI ) 
	{
		final OracleDataSource ods;
		String username;
		String password;

		try {
			Console cons= System.console();
			username = new String(cons.readLine("%s", "Username:"));
			password = new String(cons.readPassword("%s", "Password:"));

			ods = new OracleDataSource();
			ods.setURL(connectURI);
			ods.setUser(username);
			ods.setPassword(password);

			connection = (OracleConnection)ods.getConnection();

			//Set the autocommit to false.
			connection.setAutoCommit(false);
		} catch (SQLException e) {
			System.err.println("Unable to connect to the database.");
			System.err.println(e.getLocalizedMessage());
			System.exit(1);
		} 
	}

	/**
	 * Name: closeConnection
	 * Description: Commit and close the database connection.
	 * 
	 * @throws SQLException if a database access error occurs.
	 */
	public void closeConnection() throws SQLException
	{
		connection.commit();
		connection.close();
	}

	/**
	 * Name: writeImageToDatabase
	 * Description: Read the image data from the provided source file to store 
	 * it in a blob into the specified row.  
	 * 
	 * @param 	id: the row key at which the specified image is to be inserted.
	 * @param 	fileName: the path of the image to be read.
	 * @throws 	SQLException: if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 * @throws 	IOException if an I/O error occurs.
	 */
	public void writeImageToDatabase(int id, String fileName ) 
			throws SQLException, IOException   
			{
		//Define the PL/SQL block to insert the new row. 
		final String INSERT_BLOB = "DECLARE "
				+ "	src_id		NUMBER; "
				+ "BEGIN "
				+ "	src_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id=src_id; "
				+ "	INSERT INTO image_blob_table t (id, image_blob) "
				+ "		VALUES(src_id, empty_blob()) "
				+ " 		RETURNING t.image_blob INTO ?; "
				+ "END;";

		try {
			//Create the statement object.
			final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(INSERT_BLOB);

			//Binding the variables to the statement.
			pstmt.setInt(1, id); //ID
			pstmt.registerOutParameter(2, OracleTypes.BLOB);
			pstmt.execute(); //Execute the PL/SQL statement.

			//Get the BLOB locator from the table.
			BLOB blob = pstmt.getBLOB(2);

			File binaryFile = new File(fileName);
			FileInputStream instream = new FileInputStream(binaryFile);

			//Retrieve the ideal buffer size to use in writing to the BLOB.
			int size = 1024*1024; // 1MB.
			byte[] buffer = new byte[size];
			int read = -1;
			long position =1;

			//Read the file to the byte array buffer, then write it to the BLOB.
			while ((read = instream.read(buffer)) != -1)
			{
				blob.setBytes(position,buffer,0,read);
				position+=read;
			}

			instream.close();
			connection.commit();

		} catch (FileNotFoundException e) {
			throw new FileNotFoundException("File " + fileName +" not Found.");
		} catch (IOException e) {
			throw new IOException("Error while reading " + fileName);
		}
			} 

	/**
	 * Name: getProperties_example_j
	 * Description: Reads the image BLOB data to get the values of the media 
	 * attributes for supported formats and return a hashMap object containing 
	 * the properties (MIME types, width, height, file format, content format,
	 * compression format, and content length).
	 * 
	 * @param 	id: the row key from which the original image is fetched.
	 * @return 	a hashMap object containing the image properties.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter
	 * 			marker in the SQL statement.
	 */
	public HashMap<String, Object> getProperties_example_j(int id ) 
			throws SQLException
			{
		//Define the PL/SQL block to extract the properties.
		final String getPropertiesStmt = "DECLARE "
				+ "	src			BLOB; "
				+ "	img_mimeType		VARCHAR2(32); "
				+ "	img_width		INTEGER; "
				+ "	img_height		INTEGER; "
				+ "	img_contentLength	INTEGER; "
				+ "	img_fileFormat		VARCHAR2(32); "
				+ "	img_contentFormat	VARCHAR2(32); "
				+ "	img_compressionFormat	VARCHAR2(32); "
				+ "BEGIN "
				+ "	SELECT image_blob INTO src FROM image_blob_table"
				+ "		WHERE id=?; "
				+ "	ORDSYS.ORD_IMAGE.getProperties(src, "
				+ "			img_mimeType, "
				+ "			img_width, "
				+ "			img_height, "
				+ "			img_fileFormat, "
				+ "			img_compressionFormat, " 
				+ "			img_contentFormat, "
				+ "			img_contentLength); "
				+ "	? := img_mimeType; "
				+ "	? := img_width; "
				+ "	? := img_height; "
				+ "	? := img_contentLength; "
				+ "	? := img_fileFormat; "
				+ "	? := img_contentFormat;"
				+ "	? := img_compressionFormat; "
				+ "END;";

		//Create the statement object.
		final OracleCallableStatement pstmt = 
			(OracleCallableStatement)connection.prepareCall(getPropertiesStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, id);
		pstmt.registerOutParameter(2, OracleTypes.VARCHAR);
		pstmt.registerOutParameter(3, OracleTypes.INTEGER);
		pstmt.registerOutParameter(4, OracleTypes.INTEGER);
		pstmt.registerOutParameter(5, OracleTypes.INTEGER);
		pstmt.registerOutParameter(6, OracleTypes.VARCHAR);
		pstmt.registerOutParameter(7, OracleTypes.VARCHAR);
		pstmt.registerOutParameter(8, OracleTypes.VARCHAR);

		//Execute the statement.
		pstmt.execute();

		//Create a HashMap object and populate it with the properties.
		HashMap<String, Object> map = new HashMap<String, Object>();
		map.put("mimeType", pstmt.getString(2));
		map.put("width", pstmt.getInt(3) );
		map.put("height", pstmt.getInt(4));
		map.put("contentLength", pstmt.getInt(5));
		map.put("fileFormat", pstmt.getString(6));
		map.put("contentFormat", pstmt.getString(7));
		map.put("compressionFormat", pstmt.getString(8));

		return map;
			}

	/**
	 * Name: thumbnail_example_j
	 * Description: Creates a derivative image from a source image stored in 
	 * the specified row by creating an 80x80 thumbnail image and writing the 
	 * resulting image into a new row.
	 * Aspect ratio is maintained. The original source image is not modified.
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @param 	dst_id: the row key at which the thumbnail image is stored.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter 
	 * 			marker in the SQL statement.
	 */
	public void thumbnail_example_j(int src_id, int dst_id ) throws SQLException
	{
		//Define the PL/SQL block to create a thumbnail. 
		final String createThumbnailStmt  = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	src_id		NUMBER;"
				+ "	dst_id		NUMBER;"
				+ "BEGIN"
				+ "	src_id := ?;"
				+ "	dst_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id = dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id = src_id;"
				+ "	ORDSYS.ORD_IMAGE.thumbnail(src_blob,dst_blob);"
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";
		final OracleCallableStatement pstmt = 
			(OracleCallableStatement)connection.prepareCall(createThumbnailStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.setInt(2, dst_id);
		//Execute the statement.
		pstmt.execute();
		connection.commit();
	}

	/**
	 * Name: convert_example_j
	 * Description: Creates a derivative image from a source image stored in 
	 * the specified row by transcoding the source image and writing the 
	 * resulting image into the destination row. The original source image is not 
	 * modified.
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @param 	dst_id: the row key at which the new image is to be stored.
	 * @param 	fileFormat: the new file format.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public void convert_example_j(int src_id, int dst_id, String fileFormat ) 
			throws SQLException
			{
		//Define the PL/SQL block to change the file format.
		final String convertStmt = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	src_id		NUMBER;"
				+ "	dst_id		NUMBER;"
				+ "BEGIN"
				+ "	src_id := ?;"		
				+ "	dst_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id=dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id=src_id; "
				+ "	ORDSYS.ORD_IMAGE.convert(src_blob,?,dst_blob); "
				+ "	UPDATE image_blob_table SET image_blob = dst_blob "
				+ "		WHERE id = dst_id;"
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(convertStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.setInt(2, dst_id);
		pstmt.setString(3, fileFormat);
		pstmt.execute();
			}

	/**
	 * Name: crop_example_j
	 * Description: Creates a derivative image from a source image stored in the 
	 * specified row by cropping the source image and writing the resulting 
	 * image into the destination row. The original source image is not modified.
	 * 
	 * @param	src_id: the row key from which the original image is fetched.
	 * @param	dst_id: the row key at which the cropped image is to be stored.
	 * @param	originX: crop start X position in original image.
	 * @param	originY: crop start Y position in original image.
	 * @param	width: the width of the crop.
	 * @param	height: the height of the crop.
	 * @throws	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public void crop_example_j(int src_id, int dst_id,
			int originX, int originY, int width, int height) throws SQLException
			{
		final String cropStmt  = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	src_id		NUMBER;"
				+ "	dst_id		NUMBER;"
				+ "BEGIN"
				+ "	src_id := ?;"
				+ "	dst_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id = dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id = src_id;"
				+ "	ORDSYS.ORD_IMAGE.crop(src_blob,?,?,?,?,dst_blob);"
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(cropStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.setInt(2, dst_id);
		pstmt.setInt(3, originX);
		pstmt.setInt(4, originY);
		pstmt.setInt(5, width);
		pstmt.setInt(6, height);

		//Execute the statement.
		pstmt.execute();
		connection.commit();
			}

	/**
	 * Name: rotate_example_j
	 * Description: Rotates an image within the image plane by the angle 
	 * specified, and writes the resulting image into the destination row. 
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @param 	dst_id: the row key at which the rotated image is to be stored.
	 * @param 	angle: rotation angle, in degrees.
	 * 			A positive value specifies a clockwise rotation.  
	 * 			A negative value specifies a counter-clockwise rotation;
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public void rotate_example_j(int src_id, int dst_id, float angle) 
			throws SQLException
			{
		final String cropStmt  = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	src_id		NUMBER;"
				+ "	dst_id		NUMBER;"
				+ "BEGIN"
				+ "	src_id := ?;"
				+ "	dst_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id = dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id = src_id;"
				+ "	ORDSYS.ORD_IMAGE.rotate(src_blob,?, dst_blob);"
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(cropStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.setInt(2, dst_id);
		pstmt.setFloat(3, angle);

		//Execute the statement.
		pstmt.execute();
		connection.commit();
			}

	/**
	 * Name: processCopy_example_j
	 * Description: Creates a derivative image from a source image stored in
	 * the specified row by performing one or more image processing operations
	 * on the source image and writing the resulting image into the destination
	 * row. The original source image is not modified.
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @param 	dst_id: the row key at which the new image is to be stored.
	 * @param 	command: String containing the image processing operations.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public void processCopy_example_j(int src_id, int dst_id, String command) 
			throws SQLException
			{
		final String cropStmt  = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	src_id		NUMBER;"
				+ "	dst_id		NUMBER;"
				+ "BEGIN"
				+ "	src_id := ?;"
				+ "	dst_id := ?;"
				+ "	DELETE FROM image_blob_table WHERE id = dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id = src_id;"
				+ "	ORDSYS.ORD_IMAGE.PROCESSCOPY(src_blob,?, dst_blob);"
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(cropStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.setInt(2, dst_id);
		pstmt.setString(3, command);

		//Execute the statement.
		pstmt.execute();
		connection.commit();
			}

	/**
	 * Name: applyWatermark_example_j
	 * Description: Overlays a text watermark onto a source image stored in the 
	 * specified row and writes it into the destination row.
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @param 	dst_id: dst_id the row key at which the new image is to be stored.
	 * @param 	text: the text to be added to the source image.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public void applyWatermark_example_j(int src_id, int dst_id, String text) 
			throws SQLException
			{
		final String applyWatermarkStmt  = "DECLARE "
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	dst_id		NUMBER;"
				+ "	prop		ordsys.ord_str_list;"
				+ "	log		VARCHAR2(2000);"
				+ "BEGIN"
				+ "	dst_id := ?;"
				+ "	prop := ordsys.ord_str_list("
				+ "		'font_name=Times New Roman', 'font_size=80');"
				+ "	DELETE FROM image_blob_table WHERE id = dst_id;"
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES (dst_id, empty_blob()) "
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table"
				+ "		WHERE id = ?;"
				+ "	ORDSYS.ORD_IMAGE.applywatermark(src_blob,"
				+ "		?, dst_blob,log, prop);"
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";

		final OracleCallableStatement pstmt = 
			(OracleCallableStatement)connection.prepareCall(applyWatermarkStmt);

		//Binding the variables to the statement.
		pstmt.setInt(2, src_id);
		pstmt.setInt(1, dst_id);
		pstmt.setString(3, text);

		//Execute the statement.
		pstmt.execute();
		connection.commit();
			}

	/**
	 * Name: readImageFromDatabase
	 * Description: Retrieve the image stored in the specified row to create a 
	 * local file.
	 *
	 * @param 	src_id: the row key from which the image is fetched.
	 * @param 	fileName: the specified path in which to save the image.
	 * @throws 	SQLException if a database access error occurs.
	 * 			if parameterIndex does not correspond to a parameter 
	 * 			marker in the SQL statement. 
	 * @throws 	IOException if an I/O error occurs.
	 */
	public void readImageFromDatabase(int src_id, String fileName) 
			throws SQLException, IOException
			{

		//Define the PL/SQL block to retrieve the image blob.
		final String selectStmt  = "DECLARE "		 
				+ "	dst_blob	BLOB;" 		
				+ "BEGIN"
				+ "	SELECT image_blob INTO dst_blob FROM image_blob_table"
				+ "		WHERE id = ?;"
				+ "	? := dst_blob;"
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(selectStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.registerOutParameter(2, OracleTypes.BLOB);
		pstmt.execute();

		//Get the blob.
		BLOB img_blob = pstmt.getBLOB(2);

		try{
			//Declare the destination file, input and output streams.
			File file = new File(fileName);
			FileOutputStream outStream = new FileOutputStream(file);

			long remaining = img_blob.length();
			int size = 1024*1024; // 1MB.
			byte[] buffer = new byte[size];
			long position =1;

			//Write to the outputStream until there is no more data.
			while(true)
			{
				if (remaining < size)
					size=(int) remaining;

				buffer= img_blob.getBytes(position, size);

				outStream.write(buffer,0, size);
				outStream.flush();

				position+=size;
				remaining-=size;

				if (remaining ==0 )
					break;
			}

			//Close stream.
			outStream.close();

		} catch (IOException e) {
			throw new IOException("Unable to write to " + fileName);
		}
			}

	/**
	 * Name: putMetadata_example_j
	 * Description: Accepts a BLOB containing an image and a schema-valid XML 
	 * document, and creates a binary packet suitable for embedding in the 
	 * target image file format.The package is encoded using UTF-8 character 
	 * encoding. This procedure writes a new XMP packet to the image, replacing any
	 * existing XMP packets.
	 * The new image with embedded metadata is stored into the destination row.
	 * The original image is not changed.
	 * 
	 * @param 	src_id: the row key from which the original image is feteched.
	 * @param 	dst_id: the row key at which the new image is to be stored.
	 * @param 	metadata: the metadata to be added to the image.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter 
	 * 			marker in the SQL statement.
	 */
	public void putMetadata_example_j (int src_id, int dst_id, String metadata) 
			throws SQLException
			{
		final String putMetadataStmt =	"DECLARE"
				+ "	src_blob	BLOB;"
				+ "	dst_blob	BLOB;"
				+ "	dst_id		NUMBER;  "
				+ "	xmlData		XMLType;"
				+ "BEGIN"
				+ "	dst_id:=?;"
				+ "	SELECT image_blob INTO src_blob FROM image_blob_table "
				+ "		WHERE id = ?;"
				+ "	DELETE FROM image_blob_table WHERE id=dst_id; "
				+ "	INSERT INTO image_blob_table(id, image_blob) "
				+ "		VALUES(dst_id, empty_blob())"
				+ "		RETURNING image_blob INTO dst_blob;"
				+ "	xmlData := "
				+ "		xmltype(?, 'http://xmlns.oracle.com/ord/meta/xmp');"
				+ "	ORDSYS.ORD_Image.putMetadata(src_blob, dst_blob, xmlData,"
				+ "		'xmp', 'utf-8'); "
				+ "	UPDATE image_blob_table SET image_blob = dst_blob"
				+ "		WHERE id = dst_id; "
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(putMetadataStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, dst_id);
		pstmt.setInt(2, src_id);
		pstmt.setString(3, metadata);

		pstmt.execute();
		connection.commit();
			}

	/**
	 * Name: getMetadata_example_j
	 * Description: Extracts the metadata from the image BLOB stored in the
	 * specified row and returns a String containing the metadata.
	 * 
	 * @param 	src_id: the row key from which the original image is fetched.
	 * @return 	a String containing the metadata.
	 * @throws 	SQLException if a database access error occurs;
	 * 			if parameterIndex does not correspond to a parameter marker 
	 * 			in the SQL statement.
	 */
	public String getMetadata_example_j(int src_id) throws SQLException
	{
		//Define the PL/SQL block to change the file format.
		final String metadataStmt = "DECLARE"
				+ "	metav		XMLSequenceType;"
				+ "	tmp		VARCHAR(4000);"
				+ "	meta		VARCHAR(32767);"
				+ "	dest_blob	BLOB;"
				+ "	cursor		xmlToString(x XMLSequenceType) IS"
				+ "		SELECT value(list_of_values).getstringval() "
				+ "		metadata FROM table(x) list_of_values;"
				+ "BEGIN"
				+ "	SELECT t.image_blob INTO dest_blob FROM image_blob_table t"
				+ "		WHERE T.ID = ?;"
				+ "	metav := ORDSYS.ORDImage.getMetadata(dest_blob,'ALL');"
				+ "	OPEN xmlToString(metav);"
				+ "	LOOP"
				+ "		fetch xmlToString INTO tmp;"
				+ "		meta:= meta || tmp;"
				+ "		EXIT WHEN xmlToString%NOTFOUND;"
				+ "	END LOOP;"
				+ "	CLOSE xmlToString;"
				+ "	? := meta;"
				+ "END;";

		final OracleCallableStatement pstmt = 
				(OracleCallableStatement)connection.prepareCall(metadataStmt);

		//Binding the variables to the statement.
		pstmt.setInt(1, src_id);
		pstmt.registerOutParameter(2, OracleTypes.VARCHAR);

		pstmt.execute();

		return pstmt.getString(2);
	}

}
