/**
 * This is the sample code from the interMedia Java Quick Start that
 * shows how to write an interMedia Java client application that does
 * the following:
 * 
 * 1. Creates a table with a ORDImage (image) column in the database
 * 2. Uploads goats.gif into that table 
 * 3. Retrieves and prints the image properties
 * 4. Uses the database image processing functions to generate a thumbnail
 * 5. Exports the newly created image to row2.jpg
 **/

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.DriverManager;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import java.io.IOException;

import oracle.jdbc.OracleResultSet;
import oracle.jdbc.OraclePreparedStatement;

import oracle.ord.im.OrdImage;

public class InterMediaQuickStart {
    /**
     * Entry point.
     * usage: java InterMediaQuickStart connectionString username password
     * e.g.   java InterMediaQuickStart jdbc:oracle:oci:@inst1 scott tiger
     **/
    public static void main(String[] args) throws Exception {
        System.out.println("starting interMedia Java quickstart...");
        if (args.length != 3) {
            System.out.println("usage: java InterMediaQuickStart connectionString username password");
            System.out.println("e.g.   java InterMediaQuickStart jdbc:oracle:oci:@inst1 scott tiger");
            System.exit(-1);
        }

        // Run the examples as they are printed in the quick start documentation

        // register the oracle jdbc driver with the JDBC driver manager
        DriverManager.registerDriver(new oracle.jdbc.driver.OracleDriver());
        
        Connection conn = DriverManager.getConnection(args[0], args[1], args[2]);
        System.out.println("Got database connection");

        System.out.println("Starting verbatim interMedia Java Quick Start code...");
        
        // Note: it is CRITICAL to set the autocommit to false so that 
        // two-phase select-commit of BLOBs can occur.
        conn.setAutoCommit(false);

        // create a JDBC Statement object to execute SQL in the database
        Statement stmt = conn.createStatement();

        // -------------------
        // Creating a Table 
        // -------------------
        {
            // Create the image_table table with two rows: id and image
            String tableCreateSQL = "create table image_table " +
                "(id number primary key, " +
                "image ordsys.ordimage)";
            stmt.execute(tableCreateSQL);
        }

        // -------------------
        // Uploading Images from Files into Tables
        // -------------------
        {
            // insert a row into image_table
            String rowInsertSQL = ("insert into image_table (id, image) values (1,ordsys.ordimage.init())");
            stmt.execute(rowInsertSQL);
            
            // select the new ORDImage into a java proxy OrdImage object (imageProxy)
            String rowSelectSQL = "select image from image_table where id = 1 for update";
            OracleResultSet rset = (OracleResultSet)stmt.executeQuery(rowSelectSQL);
            rset.next();
            OrdImage imageProxy = (OrdImage)rset.getORAData("image", OrdImage.getORADataFactory());
            rset.close(); 
            
            imageProxy.loadDataFromFile("goats.gif");
            imageProxy.setProperties();
            
            String updateSQL = "update image_table set image=? where id=1";
            OraclePreparedStatement opstmt = (OraclePreparedStatement)conn.prepareStatement(updateSQL);
            opstmt.setORAData(1, imageProxy);
            opstmt.execute();
            opstmt.close();
        }

        // -------------------
        // Retrieving Image Properties
        // -------------------

        // Java Accessor Methods
        {
            String rowSelectSQL = "select image from image_table where id = 1";
            OracleResultSet rset = (OracleResultSet)stmt.executeQuery(rowSelectSQL);
            rset.next();
            OrdImage imageProxy = (OrdImage)rset.getORAData("image", OrdImage.getORADataFactory());
            rset.close(); 
            int height = imageProxy.getHeight();
            int width  = imageProxy.getWidth();
            System.out.println("proxy (height x width) = " + height + " x " + width);
        }

        // -------------------
        // Creating Thumbnails and Changing Formats
        // -------------------
        {
            // One could significantly reduce the number of round trip
            // database communications in the following example.
            String rowInsertSQL = ("insert into image_table (id, image) " +
                                   "values (2, ordsys.ordimage.init())");
            stmt.execute(rowInsertSQL);

            // get the source ORDImage object
            String srcSelectSQL = "select image from image_table where id=1";
            OracleResultSet rset = (OracleResultSet)stmt.executeQuery(srcSelectSQL);
            rset.next();
            OrdImage srcImageProxy = (OrdImage)rset.getORAData("image", OrdImage.getORADataFactory());
            rset.close();

            // get the newly inserted destination ORDImage object
            String dstSelectSQL = "select image from image_table where id=2 for update";
            rset = (OracleResultSet)stmt.executeQuery(dstSelectSQL);
            rset.next();
            OrdImage dstImageProxy = (OrdImage)rset.getORAData("image", OrdImage.getORADataFactory());
            rset.close();
            
            // call the processCopy method (processing occurs on the SERVER)
            srcImageProxy.processCopy("maxscale=100 100 fileformat=jfif", dstImageProxy);

            // update the destination image in the second row
            String dstUpdateSQL = "update image_table set image=? where id=2";
            OraclePreparedStatement opstmt = 
                (OraclePreparedStatement)conn.prepareStatement(dstUpdateSQL);
            opstmt.setORAData(1, dstImageProxy);
            opstmt.execute();
            opstmt.close();
        }
            
        // -------------------
        // Downloading Image Data from Tables into Files
        // -------------------
        {
            // export the data in row 2
            String exportSelectSQL = "select image from image_table where id = 2";

            OracleResultSet rset = (OracleResultSet)stmt.executeQuery(exportSelectSQL);

            // get the proxy for the image in row 2
            rset.next();
            OrdImage imageProxy = (OrdImage)rset.getORAData("image", OrdImage.getORADataFactory());
            rset.close();

            // call the getDataInFile method to write the ORDImage in row 2 to disk
            imageProxy.getDataInFile("row2.jpg");
        }
        
        // -------------------
        // Cleaning up
        // -------------------
        {
            // drop the images table
            stmt.executeQuery("drop table image_table");

            // commit all our changes
            conn.commit();
        }

        stmt.close();
        System.out.println("Done with verbatim interMedia Java Quick Start code.");

        // close the database connection to release all the resources
        conn.close();
        return;
    }
}

