#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/checktmpfs.sh /st_has_pt-cvuotn12201/1 2017/11/21 09:38:33 ptare Exp $
#
# checktmpfs.sh
#
# Copyright (c) 2011, 2017, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      checktmpfs.sh - Checking whether /dev/shm is mounted as tmpfs
#                      Also checks for /dev/shm mount options during
#                      pre dbinstall
#
#    DESCRIPTION
#      Pluggable verification framework shell script to check for /dev/shm
#      mounted and mount options during pre dbinstall stage
#      
#    NOTES
#      Checks whether /dev/shm is mounted and checks for an /etc/fstab
#      entry if it is readable. Note: on SuSE systems the /etc/fstab
#      entry is NOT required so this is NOT a failure case.
#      
#    SYNTAX:
#      $checktmpfs.sh [required_size]
#      can be called without argument, or required size as zero integer
#      for example,
#      $checktmpfs.sh 0 or $checktmpfs.sh are same
#      whereas, 
#      $checktmpfs.sh 2048 OR $checktmpfs.sh 45222 specify the valid
#      required_value argument 
#
#    MODIFIED   (MM/DD/YY)
#    ptare       08/09/17 - Fix Bug#26418606 Enhance to report incorrect
#                           permissions
#    jolascua    11/05/15 - Bug 22116467: check if node is running on a
#                           container.
#    maboddu     09/10/15 - Fix bug#21205992 - check the mount options for
#                           /dev/shm file system correctly
#    ptare       04/14/15 - Fix Bug#20880614 use = instead of == as it is not
#                           supported on HP
#    kfgriffi    12/11/14 - Fix bug 19348392 - check for /dev/shm mounted twice
#    mpradeep    11/20/14 - 20033787 - Return success when default size is
#                           given
#    kfgriffi    06/24/14 - Fix bug 19031737 - add support for 'k', 'K', bytes
#    shshrini    05/20/14 - Fix bug 14746135 - Update the script with mount
#                           options check during pre dbinstall stage
#    mpradeep    09/13/13 - 17371426 - Correct Syntax error
#    kfgriffi    06/20/13 - Add support for tmpfs size using % char(bug16958130)
#    ptare       08/28/12 - Fix Bug#14177769 complete overhaul
#    kfgriffi    03/23/12 - Fix bug 13716594
#    narbalas    02/11/11 - Creation
#
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin

MOUNT="mount"
SGREP="grep"
ECHO="echo"
FREE="free"
SAWK="awk"
SED="sed"
LS="ls"
DF="df"
STAT="/usr/bin/stat"
REQUIRED_SIZE=$1
HOST=`/bin/hostname`
CURUSR=`/usr/bin/whoami`
FSTAB=/etc/fstab
STAGE="$2"
IFS=","
EXPECTED_PERM="1777"
DEV_SHM="/dev/shm"
INVALID_OPTIONS="ro,noexec,nosuid"
MISSING_MOUNT_OPTIONS=""
INVALID_MOUNT_OPTIONS_FOUND=""
MOUNT_OPTIONS_ERROR_MESSAGE=""
MOUNT_OPTIONS_ERROR_NO=0
set -- "$VALID_OPTIONS"
IFS=","; declare -a SPLIT_VALID_OPTIONS=($*) 
set -- "$INVALID_OPTIONS"
IFS=","; declare -a SPLIT_INVALID_OPTIONS=($*) 
IFS="##";

#Utility function to check if given mount option is found or not
mountOptionFound()
{
  PARAM1=$1
  PARAM2=$2
  OLD_IFS="$IFS"
  IFS=","
  SPLIT_OPTIONS=( $PARAM1 )
  IFS="$OLD_IFS"

  for OPTION in "${SPLIT_OPTIONS[@]}"
  do
    if [ $OPTION = $PARAM2 ]
    then
      return 0;
    fi
  done
  return 1;
}

#Update the list of found invalid mount options
updateInvalidMountOptions()
{
  if [ -n "$INVALID_MOUNT_OPTIONS_FOUND" ]
  then
    INVALID_MOUNT_OPTIONS_FOUND="$INVALID_MOUNT_OPTIONS_FOUND,$1"
  else
    INVALID_MOUNT_OPTIONS_FOUND="$1"
  fi
}

#Check for the validity of mount options
checkMountOptions()
{
  retVal=0

  INVALID_MOUNT_OPTIONS_FOUND=""
  
  if [ $1 = "current" ]
  then
    #Get the mount options of /dev/shm mount from mount command 
    #Command: mount | egrep ^tmpfs | egrep "/dev/shm" | gawk '{ print $6 }' | sed 's/[()]//g'
    OPTIONS=`${MOUNT} | ${SGREP} '^tmpfs' |  ${SGREP} '/dev/shm' | tail -n 1 | ${SAWK} '{ print $6 }' | sed 's/[()]//g' `
  else
    #Get the configured mount options of /dev/shm mount inside the fstab file
    OPTIONS=`${ECHO} ${FSTAB_DEV_SHM_ENTRY} | ${SAWK} '{ print $4 }'`
  fi

  if [ -z $OPTIONS ] 
  then
    OPTIONS=" "
  fi
  #Check for invalid mount options
  for INVALID_OPTION in "${SPLIT_INVALID_OPTIONS[@]}"
  do
    if mountOptionFound "$OPTIONS" "$INVALID_OPTION" 
    then
      updateInvalidMountOptions "$INVALID_OPTION"
    fi
  done
  #We have checked both required and invalid mount options. 
  #Update messages accordingly
  if [ -z "$INVALID_MOUNT_OPTIONS_FOUND" ]; then
    return 0; 
  fi
  if [ -n "$INVALID_MOUNT_OPTIONS_FOUND" ]; then
    return 1;
  fi
}

frameMountOptionsErrorMessage()
{
  FOUND_OPTIONS=""
  MOUNT_OPTIONS_ERROR_NO=$2
  # Mount options type is either "current" or "configured"
  MOUNT_OPTIONS_TYPE=$1

  if [ -n "$MOUNT_OPTIONS_ERROR_MESSAGE" ]; 
    then
      APPEND=1
    else
      APPEND=0
  fi

  for FOUND_OPTION in "${OPTIONS[@]}"
  do
    if [ -n "$FOUND_OPTIONS" ]; 
    then
      FOUND_OPTIONS="$FOUND_OPTIONS,$FOUND_OPTION"
    else
      FOUND_OPTIONS="$FOUND_OPTION"
    fi
  done
  case $MOUNT_OPTIONS_ERROR_NO in
       #found one or more invalid mount options
       1) 
         if [ $MOUNT_OPTIONS_TYPE = "current" ] 
         then
           MESSAGE="<NLS_MSG><FACILITY>Prve</FACILITY><ID>0086</ID><MSG_DATA><DATA>$FOUND_OPTIONS</DATA><DATA>$INVALID_MOUNT_OPTIONS_FOUND</DATA></MSG_DATA></NLS_MSG>"
         else
           MESSAGE="<NLS_MSG><FACILITY>Prve</FACILITY><ID>0087</ID><MSG_DATA><DATA>$FOUND_OPTIONS</DATA><DATA>$INVALID_MOUNT_OPTIONS_FOUND</DATA></MSG_DATA></NLS_MSG>"
         fi
       ;;
  esac
  if [ $APPEND -eq 1 ] 
  then
    MOUNT_OPTIONS_ERROR_MESSAGE=`printf "$MOUNT_OPTIONS_ERROR_MESSAGE$MESSAGE"`
  else
    MOUNT_OPTIONS_ERROR_MESSAGE="$MESSAGE"
  fi

  return ;
}

  
#Internal function to initialize message strings
updateMessages()
{
  SUCCESSMSG1="<RESULT>SUCC</RESULT><EXPECTED>true</EXPECTED><COLLECTED>true</COLLECTED><TRACE>Check for /dev/shm mounted is enabled and passed on node $HOST</TRACE>"
  SUCCESSMSG2="<RESULT>SUCC</RESULT><EXPECTED>$REQUIRED_SIZE</EXPECTED><COLLECTED>$SIZE_CURRENT</COLLECTED><TRACE>Check for /dev/shm mounted is enabled passed on node $HOST</TRACE>"
  SUCCESSMSG3="<RESULT>SUCC</RESULT><EXPECTED>true</EXPECTED><COLLECTED>true</COLLECTED><TRACE>Check for /dev/shm mounted is enabled and validation of /dev/shm mount options passed on node $HOST</TRACE>"
  ERRMSG1="<RESULT>VFAIL</RESULT><EXPECTED>true</EXPECTED><COLLECTED>false</COLLECTED><TRACE>Check for /dev/shm mounted enabled failed on node $HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0420</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG2="<RESULT>VFAIL</RESULT><EXPECTED>true</EXPECTED><COLLECTED>false</COLLECTED><TRACE>Check failed on node $HOST, entry for /dev/shm is missing inside $FSTAB</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0421</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG3="<RESULT>WARN</RESULT><EXPECTED>$SIZE_CURRENT</EXPECTED><COLLECTED>$SIZE_REBOOT</COLLECTED><TRACE>Check failed on node $HOST, value configured in /etc/fstab is $SIZE_REBOOT MB and is different compared to current size of $SIZE_CURRENT MB with which the memory disk is mounted</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0422</ID><MSG_DATA><DATA>$SIZE_CURRENT</DATA><DATA>$SIZE_REBOOT</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG4="<RESULT>EFAIL</RESULT><EXPECTED>true</EXPECTED><COLLECTED>false</COLLECTED><TRACE>Check failed on node $HOST, $FSTAB file does not exist</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0423</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG5="<RESULT>VFAIL</RESULT><EXPECTED>$REQUIRED_SIZE</EXPECTED><COLLECTED>$SIZE_CURRENT</COLLECTED><TRACE>Check failed on node $HOST, The current mount size of /dev/shm temporary file system is $SIZE_CURRENT MB which is less than that of required size $REQUIRED_SIZE MB</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0426</ID><MSG_DATA><DATA>$SIZE_CURRENT</DATA><DATA>$REQUIRED_SIZE</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG6="<RESULT>EFAIL</RESULT><EXPECTED>true</EXPECTED><COLLECTED>false</COLLECTED><TRACE>Check failed on node $HOST, Could not retrieve the current size of /dev/shm mounted</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0427</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG7="<RESULT>VFAIL</RESULT><EXPECTED>true</EXPECTED><COLLECTED>false</COLLECTED><TRACE>Check failed on node $HOST, entry for /dev/shm is missing inside $FSTAB</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0428</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG8="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED_PERM</EXPECTED><COLLECTED></COLLECTED><TRACE>Failed to obtain the permissions of path /dev/shm on node $HOST</TRACE><NLS_MSG><FACILITY>Prvg</FACILITY><ID>12893</ID><MSG_DATA><DATA>$CURUSR</DATA><DATA>$DEV_SHM</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  ERRMSG9="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED_PERM</EXPECTED><COLLECTED>$CURRENT_PERM</COLLECTED><TRACE>Incorret permissions found for /dev/shm directory on node $HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10257</ID><MSG_DATA><DATA>$DEV_SHM</DATA><DATA>$HOST</DATA><DATA>$EXPECTED_PERM</DATA><DATA>$CURRENT_PERM</DATA></MSG_DATA></NLS_MSG>"
}

errorout()
{
  exit 1
}

exitSucc() 
{
  exit 0
}

#Internal method to convert the size variable into integer representing it
#in megabytes. However the input values may take the form of:
#
#    nnnK - nnn specified as Kilobytes
#    nnnk - nnn specified as Kilobytes
#    nnnM - nnn specified as Megabytes
#    nnnm - nnn specified as Megabytes
#    nnnG - nnn specified as Gigabytes
#    nnng - nnn specified as Gigabytes
#    nnn% - nnn specified as a percentage of RAM memory on the system
#
#Example if the system has 12400 Megabytes of RAM, and the value passed 
#in is 75%, then the conversion should return a value of 9300.
#
#The converted integer value is saved in variable RETURN_VALUE
convertSizeToMB()
{
  INPUT_UNIT=$1
  INT_UNIT=0;
  if [ -n "$INPUT_UNIT" ]
  then
    if [[ "$INPUT_UNIT" = *K* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -FK '{print $1}'`
       INT_UNIT=`expr $INT_UNIT / 1024`
    elif [[ "$INPUT_UNIT" = *k* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -Fk '{print $1}'`
       INT_UNIT=`expr $INT_UNIT / 1024`
    elif [[ "$INPUT_UNIT" = *M* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -FM '{print $1}'`
    elif [[ "$INPUT_UNIT" = *m* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -Fm '{print $1}'`
    elif [[ "$INPUT_UNIT" = *G* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -FG '{print $1}'`
       INT_UNIT=`expr $INT_UNIT \* 1024`
    elif [[ "$INPUT_UNIT" = *g* ]]
    then
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -Fg '{print $1}'`
       INT_UNIT=`expr $INT_UNIT \* 1024`
    elif [[ "$INPUT_UNIT" = *%* ]]
    then
       # The fstab entry can be expressed in the form of a percentage of RAM 
       # memory (e.g. 50%, 75%, etc...). If this is the case then we need to
       # get the size of RAM
       RAM_MEM_SIZE=`${FREE} -m | ${SGREP} 'Mem:' | ${SAWK} '{ print $2 }'`

       # First strip off the '%' sign from the end of the value and save
       INT_UNIT=`$ECHO $INPUT_UNIT | ${SAWK} -F% '{print $1}'`

       # Now convert the integer value into a divisor.
       # Note: we have to use the 'bc' utility to make sure we get correct
       #       values.
       DIVIDER=$($ECHO "scale=10; 100 / ${INT_UNIT}" | bc)

       # Finally divide the amount of RAM memory by the divisor to obtain 
       # the value we will be looking for /DEV/SHM
       INT_UNIT=$($ECHO " ${RAM_MEM_SIZE} / ${DIVIDER} " | bc)
    else
       INT_UNIT=$INPUT_UNIT
    fi
  fi
  RETURN_VALUE=$INT_UNIT
}

# Internal function to parse the size unit string to retrieve size in MB,
# and compare the units. Must be called with 2 arguments, first argument
# being current size and second argument being required size or size to
# compare with returns 0 if current size is greater than or equal to
# required size returns 1 if current size is less than required size 
compareMountSize()
{
  INPUT_UNIT1=$1
  INPUT_UNIT2=$2
  INT_UNIT1=0;
  INT_UNIT2=0;
  if [ -n "$INPUT_UNIT1" ] && [ -n "$INPUT_UNIT2" ]
  then
    convertSizeToMB $INPUT_UNIT1
    INT_UNIT1=$RETURN_VALUE
    convertSizeToMB $INPUT_UNIT2
    INT_UNIT2=$RETURN_VALUE

    if [ $INT_UNIT1 -ge $INT_UNIT2 ]
    then
        return 0
    else
        return 1
    fi
  fi
}

#Internal function to check for the consistency of current /dev/shm size with an entry inside fstab file
checkSizeConsistencyWithFstab()
{
    FSTAB_DEV_SHM_ENTRY=`${SGREP} '^tmpfs' ${FSTAB} | ${SGREP} '/dev/shm'`
    # Check if /dev/shm entry is present inside fstab file
    if [ -n "$FSTAB_DEV_SHM_ENTRY" ]
    then
       # Get the configured size of /dev/shm mount inside the fstab file
       SIZE_REBOOT=`${ECHO} ${FSTAB_DEV_SHM_ENTRY} | ${SAWK} '{ print $4 }' | awk -F= '{ print $2 }'`
       # Now that we have the value from /etc/fstab, convert it to MB if its
       # a numeric value, or if it is a percentage get the percentage of RAM
       convertSizeToMB $SIZE_REBOOT
       SIZE_REBOOT=$RETURN_VALUE

       # If we have retrieved the reboot size then lets compare it with that
       # of current size
       if [ -n "$SIZE_REBOOT" ]
       then 
         if [ "$SIZE_REBOOT" -eq 0 -o "$SIZE_REBOOT" = "defaults" ]
         then
          # Task Success
              ${ECHO} ${SUCCESSMSG}
              exitSucc
         else
           #Check if the reboot size and current size are equal
           compareMountSize $SIZE_REBOOT $SIZE_CURRENT
           if [ $? -eq 0 ]
           then
             if [ "$STAGE" = "pre_db_install" ];
             then
               #Return true to avoid test environment farm job errors
               if [ "${CVU_TEST_ENV}" = "true" ]; 
               then
                 ${ECHO} ${SUCCESSMSG}
                 exitSucc 
               fi
               # check for current mount options
               checkMountOptions  "current"
               retVal=$?
               if [ $retVal -ne 0 ]
               then
                # Update error message to include current mount options mismatch
                frameMountOptionsErrorMessage "current" $retVal
               fi
               # check for configured mount options
               checkMountOptions  "configured"
               retVal=$?
               if [ $retVal -ne 0 ]
               then
                # Update error message to include configured mount options mismatch
                 frameMountOptionsErrorMessage "configured" $retVal
               fi
               if [ $retVal -eq 0 ] && [ -z "$MOUNT_OPTIONS_ERROR_MSG" ]
               then
                 ${ECHO} ${SUCCESSMSG3}
                 exitSucc 
               else
                 MOUNT_OPTIONS_FINAL_ERROR_MSG=`printf "<RESULTSET><RESULT><STATUS>VFAIL</STATUS><TRACE>Errors found during '/dev/shm/' mount options check.</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0088</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>$MOUNT_OPTIONS_ERROR_MESSAGE</RESULT></RESULTSET>"`
                 ${ECHO} ${MOUNT_OPTIONS_FINAL_ERROR_MSG}
                 errorout 
               fi
             fi
             ${ECHO} ${SUCCESSMSG}
             exitSucc 
           else
             updateMessages
             ${ECHO} ${ERRMSG3}
             errorout 
           fi
         fi
       else
          #The fstab file does not define size configured, skip the check fir configured value inside fstab
          ${ECHO} ${SUCCESSMSG}
          exitSucc
       fi
     else

        #On OL7/RHEL7 and further?
        RHELRELEASE=`${LS} /etc | ${SGREP} -i redhat | ${SGREP} -i release`
        RHEL_RELEASE_FILE="/etc/$RHELRELEASE"
        OELRELEASE=`${LS} /etc | ${SGREP} -i oracle | ${SGREP} -i release`
        OEL_RELEASE_FILE="/etc/$OELRELEASE"

        # On SuSE?
        SUSERELEASE=`${LS} /etc | ${SGREP} -i suse | ${SGREP} -i release`
        SUSERELEASE_FILE="/etc/$SUSERELEASE"

        # On NOVELL?
        NOVELLRELEASE=`${LS} /etc | ${SGREP} -i novell | ${SGREP} -i release`
        NOVELLRELEASE_FILE="/etc/$NOVELLRELEASE"

        # On SLES? (not used yet, here for completeness)
        SLESRELEASE=`${LS} /etc | ${SGREP} -i sles | ${SGREP} -i release`
        SLESRELEASE_FILE="/etc/$SLESRELEASE"

        # on OL7/RHEL7 or greater, /dev/shm does not need to be in /etc/fstab
        if [ -f $RHEL_RELEASE_FILE ] || [ -f $OEL_RELEASE_FILE ]
        then
            # Get release major version 
            RELEASE_VER=`/bin/rpm -q --qf "%{VERSION}" redhat-release`
            retVal=$?
            #If not successful then attempt to get OEL release
            if [ $retVal -ne 0 ]
            then
                RELEASE_VER=`/bin/rpm -q --qf "%{VERSION}" oraclelinux-release`
                retVal=$?
            fi
            if [ $retVal -eq 0 ]
            then
                RELEASE_VER=`${ECHO} $RELEASE_VER | ${SAWK} '{print substr($0,0,1)}'`
                if [ $RELEASE_VER -gt 6 ]
                then
                    # Task Success - /dev/shm entry need not be in /etc/fstab on OL7/RHEL7 and further
                    ${ECHO} ${SUCCESSMSG}
                    exitSucc
                fi
            fi
        fi

        # on SuSE 11, or greater, /dev/shm does not need to be in /etc/fstab
        if [ -f $SUSERELEASE_FILE ]
        then
            # Get release major ver (i.e. if 11.1 get 11)
            RELEASE_VER=`/bin/rpm -q --qf "%{VERSION}" sles-release | cut -f 1 -d '.'`
            if [ $RELEASE_VER -gt 10 ]
            then
                # Task Success - /dev/shm does not need to be in /etc/fstab on SuSE
                ${ECHO} ${SUCCESSMSG}
                exitSucc
            fi
        fi

        if [ -f $NOVELLRELEASE_FILE ]
        then
            # Task Success - /dev/shm does not need to be in /etc/fstab for NOVELL
            ${ECHO} ${SUCCESSMSG}
            exitSucc
        fi

        # On Containers
        CGROUP=`${SAWK} -F/ '$2 == "lxc"' '/proc/1/cgroup'`
        if [ -n "$CGROUP" ]
        then
            #There was no entry found in /proc/1/mounts, this is only for containers
            ${ECHO} ${ERRMSG7}
        else
            # There was no entry found in /etc/fstab, report this error.
            ${ECHO} ${ERRMSG2}
        fi
        errorout 
    fi
}

#Script execution begins here
updateMessages

#Assume the success message to be default first
SUCCESSMSG=$SUCCESSMSG1

#First, check the results from running /bin/mount to see if /dev/shm is mounted
#Note: 'tail -n 1' will result in only ONE /dev/shm line being output if it has
#      been mounted twice. See bug 19348392
RSLTMOUNT=`${MOUNT} | ${SGREP} '^tmpfs' |  ${SGREP} '/dev/shm' | tail -n 1`
RET=$?
if [ $RET -ne 0 ]
then
    #/dev/shm not mounted
    ${ECHO} ${ERRMSG1}
    errorout
fi

#Check the permissions. /dev/shm must have 1777 permissions if found mounted.
CURRENT_PERM=`${STAT} -c %a $DEV_SHM`
RET=$?
if [ $RET -ne 0 ]
then
    #failed to obtain the permissions of /dev/shm
    updateMessages
    ${ECHO} ${ERRMSG8}
    errorout
elif [ $CURRENT_PERM != $EXPECTED_PERM ] && [ $CURRENT_PERM != "01777" ]
then
    #Incorrect permissions of /dev/shm
    updateMessages
    ${ECHO} ${ERRMSG9}
    errorout
fi

#Get the current size of mounted /dev/shm with its unit, i.e 7900M or 8G or 8500m or 50% etc
SIZE_ELEM=`${ECHO} ${RSLTMOUNT} | ${SAWK} '{ print $6 }' | ${SAWK} -Fsize= '{ print $2 }' | ${SAWK} -F\) '{print $1}'`
if [ -z "$SIZE_ELEM" ]
then
   #fallback on the df approach
   SIZE_CURRENT=`${DF} -h |  ${SGREP} 'tmpfs' |  ${SGREP} '/dev/shm' | ${SAWK} '{ print $2 }'`
else
   # We are not expecting the 'mount' command to return a percentage value,
   # but to be sure use the conversion subroutine to check it.
   convertSizeToMB $SIZE_ELEM
   SIZE_CURRENT=$RETURN_VALUE
fi

#Check if we have the current size of /dev/shm retrieved
if [ -z "$SIZE_CURRENT" ]
then 
    #Current size of /dev/shm mounted could not be retrieved on this node, report error
    ${ECHO} ${ERRMSG6}
    errorout
fi

#Convert the current size in MB's
convertSizeToMB $SIZE_CURRENT
SIZE_CURRENT=$RETURN_VALUE

#Check if we are asked to check the required size
if [ -n "$REQUIRED_SIZE" ]
then
    convertSizeToMB $REQUIRED_SIZE
    REQUIRED_SIZE=$RETURN_VALUE
fi

#Call update messages to update the messages with current size
updateMessages

# If required value is specified and is greater than zero then
# check if the current size is greater than or equal to the required size
# if we are asked to check it
if [ -n "$REQUIRED_SIZE" ] && [[ "$REQUIRED_SIZE" != 0* ]]
then 
    SUCCESSMSG=$SUCCESSMSG2
    compareMountSize $SIZE_CURRENT $REQUIRED_SIZE
    if [ $? -eq 1 ]
    then
       ${ECHO} ${ERRMSG5}
       errorout
    else
       SUCCESSMSG=$SUCCESSMSG2
    fi
fi 

#Check whether mount information is available at startup in /etc/fstab if /etc/fstab exists and is readable 
#check if node is on container, if that the case /dev/shm is not found on /etc/fstab, entry can be found on /proc/1/mounts
CGROUP=`${SAWK} -F/ '$2 == "lxc"' '/proc/1/cgroup'`
if [ ! -z "$CGROUP" ]
then
  #this is a container, look for entry /dev/shm on /proc/1/mounts
  FSTAB="/proc/1/mounts"
fi

if [ -f $FSTAB ]
then
    #Check if the fstab file is readable
    if [ -r $FSTAB ]
    then
        # /etc/fstab is readable, lets verify the consistency of /dev/shm mount
        # size inside the entry in /etc/fstab file
        checkSizeConsistencyWithFstab
    else
        # if /etc/fstab is not readable we do not verify the /dev/shm entry
        # inside it, hence report success and leave
        ${ECHO} ${SUCCESSMSG}
        exitSucc
    fi
else
    #/etc/fstab file does not exist on this node, report error
    ${ECHO} ${ERRMSG4}
    errorout 
fi

