#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/checkmemlock.sh /st_has_pt-cvuotn12201/1 2018/06/06 07:43:18 maboddu Exp $
#
# checkmemlock.sh
#
# Copyright (c) 2010, 2018, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      checkmemlock.sh - <one-line expansion of the name>
#
#    DESCRIPTION
#      A script to validate whether max. memory locked limit is less than a certain value
#      Consumed by the pluggable framework as a Pluggable Task.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    maboddu     03/15/18 - Fix bug#27695334 - fall back to
#                           /etc/security/limits.conf if the memlock entry for
#                           requested user not found inside
#                           /etc/security/limists.d/
#    ptare       01/26/16 - Enhance the error message reporting
#    maboddu     10/01/15 - Fix bug#20135364 - check the maximum locked memeory
#                           in conf files in /etc/security/limits.d directory
#    ptare       06/24/15 - Correct AMM default status
#    ptare       06/18/14 - Fix Bug#19013789 consider using [[space]] in
#                           command instead of hard space
#    ptare       05/21/14 - Fix Bug#17276570 correct the expected value to be
#                           90% of total RAM if hugepages are enabled
#    ptare       06/05/13 - Correct the retrieval and comparision
#    ptare       02/14/12 - fix Bug#11785435
#    narbalas    05/07/10 - Fix Expected Value
#    narbalas    04/22/10 - Creation
#

#This function checks if hugepages are supported and enabled if supported
checkHugePagesSupportedAndEnabled()
{
PHYSMEM=`$CAT /proc/meminfo | $GREP MemTotal | $AWK '{print $2}'`
ret=$?
if [ $ret -eq 0 ]
then
    #Since the decision to check for enabled hugepages only depends on whether the system has 4GB or more 
    #Physical memory unit is kB. Required memory is calculated in kB (4 GB = 4 * 1048576 kB)
    REQMEM=`expr $REQMEMSIZE \* 1048576`
else
    #Command Failure - Failed to get physical memory
    ERRCODE=4
    prepareResult
    adieu
fi 

#Check if installed physical memory is greater than 4GB
if [ $PHYSMEM -ge $REQMEM ]
then
  #installed memory is more than 4GB
  #check if hugepages are supported and enabled 
  CHECKHUGEPAGES=`$GREP Hugepage /proc/meminfo`
  ret=$?
  if [ $ret -eq 0 ]
  then
    #Huge pages feature is supported on this system, lets check if it is enabled.
    if [ -f $HUGEPAGEPATH ]
    then
      #Huge pages feature is enabled. Lets check if its use is enabled or disabled
      #if the value in /proc/sys/vm/nr_hugepages file is "0" it means HugePages
      #is disable on the system
      HUGEPAGES_TOTAL=`$CAT $HUGEPAGEPATH`
      if [ "$HUGEPAGES_TOTAL" -ne "0" ]; then
        HUGEPAGES_ENABLED="TRUE"
      fi
    fi
  fi
fi
}

#Checks whether automatic memory management is enabled
checkDevShmMounted()    
{
$MOUNT | $GREP "\/dev\/shm" | $GREP -e ramfs -e tmpfs >/dev/null 2>&1
if [ $? -eq 0 ]
then
    DEV_SHM_MOUNTED="TRUE"
else
    #Flag that AMM is not enabled (ERRCODE = 1)
    DEV_SHM_MOUNTED="FALSE"
fi
}

#Check maximum locked memory setting
checkMemLock()
{

if [ "${CVU_TEST_ENV}" = "true" ]; then
     MEMLOCK=$EXPECTED
     ERRCODE=0
     return
fi

if [ -d "$LIMITSDIRPATH" ]
then
   $GREP -h "memlock" $LIMITSDIRCONFPATH | $GREP -v "^#" | $GREP hard >/dev/null 2>&1
   if [ $? -ne 0 ];
   then
      $GREP "memlock" $LIMITSDOTCONF | $GREP -v "^#" | $GREP hard >/dev/null 2>&1
   else
      LIMITSDIREXISTS="true"
   fi 
else
   $GREP "memlock" $LIMITSDOTCONF | $GREP -v "^#" | $GREP hard >/dev/null 2>&1
fi

if [ $? -eq 0 ];
then
    if [ "$LIMITSDIREXISTS" = "true" ];
    then
       #Check if we have the user name entry made inside the /etc/security/limits.d/
       getmemlock
       if [ "$MEMLOCK" = "" ];
       then
          # fall back to check the entry inside /etc/security/limits.conf
          getmemlock $LIMITSDOTCONF
          LIMITSCONFFILE=$LIMITSDOTCONF
       fi
    else
       # /etc/security/limts.d directory not exists. Fall back to get the value from /etc/security/limits.conf
       getmemlock $LIMITSDOTCONF
       LIMITSCONFFILE=$LIMITSDOTCONF
    fi

    #Check if we still do not have the value
    if [ "$MEMLOCK" = "" ]; then
      ERRCODE=2
      return
    fi;

    #if the value is unlimited OR equal to expected or greater than expected then it is success
    if [ "$MEMLOCK" = "unlimited" ] || [ $MEMLOCK -eq $EXPECTED ] || [ $MEMLOCK -gt $EXPECTED ] ;
    then
       #SUCCESS
       ERRCODE=0
       return
    else
       #FAILURE
       ERRCODE=1
       return
    fi
else
    ERRCODE=2
    return
fi
}

#get the value of memlock for the current user
getmemlock()
{
  LIMITSCONFFILEPATH=$1

  if [ "$LIMITSCONFFILEPATH" = "" ];
  then
     #/etc/security/limits.d/ directory exists. Check the entry inside this dir.
     LIMITSCONFFILEPATH=$LIMITSDIRCONFPATH
  fi;

  #Check if we have the user name entry made inside the limits conf file
  MEMLOCK=`$GREP -h "memlock" $LIMITSCONFFILEPATH | $GREP -v "^#" | $GREP "^[[:space:]]*\$CURUSR[[:space:]]" | $GREP 'hard' | $HEAD -n 1 | $AWK '{ print $4 }' 2>&1`
  LIMITSCONFFILE=`$GREP -H "memlock" $LIMITSCONFFILEPATH | $GREP "[[:space:]]*\$CURUSR[[:space:]]" | $GREP 'hard' | $HEAD -n 1 | $AWK -F: '{ print $1 }' 2>&1`
  if [ "$MEMLOCK" = "" ];
  then
    #Get the default entry if existing
    MEMLOCK=`$GREP -h "memlock" $LIMITSCONFFILEPATH | $GREP -v "^#" | $GREP "^[[:space:]]*\*" | $GREP 'hard' | $HEAD -n 1 | $AWK '{ print $4 }' 2>&1`
    LIMITSCONFFILE=`$GREP -H "memlock" $LIMITSCONFFILEPATH | $GREP "[[:space:]]*\*" | $GREP 'hard' | $HEAD -n 1 | $AWK -F: '{ print $1 }' 2>&1`
  fi
}

setupEnv()
{
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin
#use absolute paths for all the commands
CAT="/bin/cat"
GREP="/bin/grep"
AWK="/bin/awk"
HUGEPAGEPATH="/proc/sys/vm/nr_hugepages"
REQMEMSIZE=4
HOST=`/bin/hostname`
CHECKHUGEPAGES=0
MEMLOCK=0
LIMITSDOTCONF="/etc/security/limits.conf"
LIMITSDIRPATH="/etc/security/limits.d"
LIMITSDIRCONFPATH="/etc/security/limits.d/*.conf"
LIMITSCONFFILE=$LIMITSDOTCONF
ERRCODE=9
CURUSR=`/usr/bin/whoami`
HEAD="/usr/bin/head"
HUGEPAGES_ENABLED="FALSE"
MOUNT="/bin/mount"
DEV_SHM_MOUNTED="FALSE"

#Check whether /dev/shm is mounted
checkDevShmMounted

#Assume the default expected value
EXPECTED=3145728

#Check if hugepages are enabled
checkHugePagesSupportedAndEnabled

if [ "$HUGEPAGES_ENABLED" = "TRUE" ]; then
  #Huge pages are enabled and hence calculate
  #the expected minimum value of memlock
  #based on currently installed Physical memory size
  #We expect a minimum 90% of the total RAM
  #size to be set to memlock

  if [ "$PHYSMEM" != "" ]; then
    #Calculate the 90% of it
    LESS=`expr $PHYSMEM / 10`
    EXPECTED=`expr $PHYSMEM - $LESS`
  fi;
fi;

}

prepareResult()
{
 case $ERRCODE in
       0)
          RESULT="<RESULT>SUCC</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check for maximum memory locked limit passed on node $HOST</TRACE>"
          ;;
       1)
          #Frame the verification failure message based on the current environment
          if [ "$HUGEPAGES_ENABLED" = "TRUE" ]; then
            RESULT="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check failed on node $HOST, Maximum locked memory limit is less than $EXPECTED when huge pages are enabled</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0068</ID><MSG_DATA><DATA>$LIMITSCONFFILE</DATA><DATA>$EXPECTED</DATA><DATA>$MEMLOCK</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
          else
            if [ "$DEV_SHM_MOUNTED" = "TRUE" ]; then
              RESULT="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check failed on node $HOST, Maximum locked memory limit is less than $EXPECTED when DB Automatic Memory Management feature is enabled</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0042</ID><MSG_DATA><DATA>$LIMITSCONFFILE</DATA><DATA>$EXPECTED</DATA><DATA>$MEMLOCK</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
            else
              RESULT="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check failed on node $HOST, Maximum locked memory limit is less than $EXPECTED</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0067</ID><MSG_DATA><DATA>$LIMITSCONFFILE</DATA><DATA>$EXPECTED</DATA><DATA>$MEMLOCK</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
            fi;
          fi;
          ;;
       2)
          RESULT="<RESULT>VFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check failed on node $HOST, no values set for Maximum locked memory</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0059</ID><MSG_DATA><DATA>$CURUSR</DATA><DATA>$LIMITSDOTCONF</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
          ;;
       4)
          RESULT="<RESULT>WARN</RESULT><COLLECTED>false</COLLECTED><EXPECTED>true</EXPECTED><EXEC_ERROR>Error while getting physical memory of the system</EXEC_ERROR><TRACE>Unable to get the physical memory of the system</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0022</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
          ;;
       9) RESULT="<RESULT>EFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><COLLECTED>$MEMLOCK</COLLECTED><TRACE>Check for maximum memory locked limit when DB Automatic memory management feature is enabled encountered a command failure </TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0043</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
          ;;
  esac
  return
}

adieu()
{
  echo $RESULT
  exit 
}

#Main
#Setup environment
setupEnv
checkMemLock
prepareResult
adieu



