#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/check_rp_filter.sh /st_has_pt-cvuotn12201/1 2018/02/07 08:56:06 ptare Exp $
#
# check_rp_filter.sh
#
# Copyright (c) 2011, 2018, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      check_rp_filter.sh - check the reverse path filter setting for cluster private interconnect classified NICs 
#
#    DESCRIPTION
#      check the reverse path filter "rp_filter" parameter for NICS selected for private interconnect 
#
#    NOTES
#      Currently this check only applies to OEL6 and further LINUX releases 
#
#    MODIFIED   (MM/DD/YY)
#    ptare       12/12/17 - Fix Bug#27222423 fix the wildcard issue
#    ptare       01/25/16 - Fix Bug#22586981 Perform rp_filters check only when
#                           number of private nics is > 1
#    ptare       01/21/16 - Fix Bug#22500594 implement fix-up for rp_filter
#    ptare       10/30/15 - Fix Bug#22094018 consider all.rp_filter max value with
#                           interface.rp_filter starting kernel 2.6.31
#    ptare       07/23/15 - Fix Bug#21467232 handle the duplicate entries for
#                           similar variables in sysctl conf
#    lcarvall    07/17/15 - Fix Bug 21149938 - rp_filter one private NIC
#    maboddu     06/22/15 - Fix bug#21268330 - Do not account space as delim
#                           for parameters inside /etc/resolv.conf
#    ptare       05/12/15 - Fix Bug#21044983 grep for non commented lines only
#    ptare       04/14/15 - Fix Bug#20880614 use = instead of == as it is not
#                           supported on HP
#    shshrini    02/25/15 - Enhance network checks, project 47212
#    ptare       08/13/14 - Bug#19435017 consider explicit values for
#                           interfaces
#    ptare       04/27/11 - check the reverse path filter rp_filter for NICS on
#                           LINUX
#    ptare       04/27/11 - Creation
#

SECHO="/bin/echo"
SGREP="/bin/grep"
SAWK="/bin/awk"
SSED="/bin/sed"
STAIL="/usr/bin/tail"
SSYSCTL="/sbin/sysctl -q"
SSYSCTLFILE="/etc/sysctl.conf"
SEPARATOR=","
_HOST=`/bin/hostname`
HAS_WILDCARD="FALSE"
PLATFORM=`/bin/uname`
REQ_MIN_KER_VER_FOR_ALL_FILTER=2631
CONF_ALL_RP_FILTER_MAX="FALSE"
CURR_ALL_RP_FILTER_MAX="FALSE"
CURRENT_KERNEL_VER=`/bin/uname -r | $SSED -e 's/\.//g' | cut -b 1-4`
FIXUP_DATA_REQUESTED="FALSE"

#check if the fix-up data is requested.
if [ "X$2" = "X-getfixupdata" ]; then
FIXUP_DATA_REQUESTED="TRUE"
fi

case $PLATFORM in
  SunOS)
    SAWK="/usr/xpg4/bin/awk"
  ;;
esac

# Set default exit status to indicate failure.
existstatus=3
badCurrentValNicList=""
badConfigValNicList=""
expected="0|2"
NUMBER_OF_PVT_NICS=0

#method to get the value of rp_flter kernel parameter for given interface
getRPFilterCurrentValue ()
{
  ifEntry="net.ipv4.conf.$1.rp_filter"

  #Determine the current value
  command="$SSYSCTL $ifEntry 2>/dev/null" #redirect the standard error to /dev/null 
  output=$(/bin/sh -c "$command")
  ret=$?

  if [ $ret -ne 0 ] && [ "$1" != "all" ]; then
    #try for the 'all' value if specified
    command="$SSYSCTL net.ipv4.conf.all.rp_filter 2>/dev/null" #redirect the standard error to /dev/null 
    output=$(/bin/sh -c "$command")
    ret=$?
    if [ $ret -ne 0 ]; then
      #try for the 'default' value if specified
      command="$SSYSCTL net.ipv4.conf.default.rp_filter 2>/dev/null" #redirect the standard error to /dev/null 
      output=$(/bin/sh -c "$command")
      ret=$?
    fi
  fi

  #fetch the value from output
  if [ $ret -eq 0 ] && [ "X$output" != "X" ]; then
    command="$SECHO $output | $SSED 's/[[:space:]]//g' | $SAWK -F"=" '{print \$2}'"
    _current_val=$(/bin/sh -c "$command")
  else
    _current_val=-999
  fi

  return $ret;
}

#method to get the configured value of the RP filter for the specified interface
getRPFilterConfiguredValue ()
{
  ifEntry="\"^[[:space:]]*net.ipv4.conf.$1.rp_filter\""
  #Determine the configured value
  command="$SGREP $ifEntry $SSYSCTLFILE 2>/dev/null" #redirect the standard error to /dev/null 
  output=$(/bin/sh -c "$command")
  ret=$?

  if [ $ret -ne 0 ] && [ "$1" != "all" ]; then
    #try for the 'all' value if specified
    command="$SGREP \"^[[:space:]]*net.ipv4.conf.all.rp_filter\" $SSYSCTLFILE 2>/dev/null" #redirect the standard error to /dev/null 
    output=$(/bin/sh -c "$command")
    ret=$?
    if [ $ret -ne 0 ]; then
      #try for the 'default' value if specified
      command="$SGREP \"^[[:space:]]*net.ipv4.conf.default.rp_filter\" $SSYSCTLFILE 2>/dev/null" #redirect the standard error to /dev/null 
      output=$(/bin/sh -c "$command")
      ret=$?
      #in an environment there might be duplicate entries for same variable in sysctl.conf. we shall handle it by considering the last entry
      if [ $ret -eq 0 ]; then
        command="$SGREP \"^[[:space:]]*net.ipv4.conf.default.rp_filter\" $SSYSCTLFILE | $STAIL -1 2>/dev/null" #redirect the standard error to /dev/null 
        output=$(/bin/sh -c "$command")
      fi
    else
      #in an environment there might be duplicate entries for same variable in sysctl.conf. we shall handle it by considering the last entry
      command="$SGREP \"^[[:space:]]*net.ipv4.conf.all.rp_filter\" $SSYSCTLFILE | $STAIL -1 2>/dev/null" #redirect the standard error to /dev/null 
      output=$(/bin/sh -c "$command")
    fi
  fi

  #fetch the value from output
  if [ $ret -eq 0 ] && [ "X$output" != "X" ]; then
    command="$SECHO $output | $SSED 's/[[:space:]]//g' | $SAWK -F"=" '{print \$2}'"
    _config_val=$(/bin/sh -c "$command")
  else
    _config_val=-999
  fi

  return $ret;
}   

#method to check if passed input follows wild card pattern
checkWildCards()
{
  input=$1
  if [[ $input = *\** ]] ||
     [[ $input = *\[a-z][1-9].* ]] ||
     [[ $input = *\^* ]] ||
     [[ $input = *\$* ]] ||
     [[ $input = *\[* ]] ||
     [[ $input = *\]* ]] ||
     [[ $input = *\(* ]] ||
     [[ $input = *\)* ]] ||
     [[ $input = *\{* ]] ||
     [[ $input = *\}* ]];then
    HAS_WILDCARD="TRUE"
  else
    HAS_WILDCARD="FALSE"
  fi
}


#method to resolve the interface names by given subnet
processWildCardInterfaces()
{
  PVT_SUBNET="SUBNET,$1"
  pattern="<ADAPTER>"
  nicNameWithWildCards=$2

  #Get the network information on this node using exectask 
  currentDir=$(dirname $0)
  output=`$currentDir/exectask -getifinfo`
  #align the output of exectask to be lines starting with <ADAPTER> as this is what we are looking for
  output=`$SECHO $output | $SSED 's/<ADAPTER>/\n&/g'`
  array=""
  #Form only lines starting with <ADAPTER>
  for line in $output
  do
    if [[ $line = $pattern* ]]
    then
     array=$array$'\n'$line
    else
     array=$array$line
    fi
  done

  #Now for all the entries starting with <ADAPTER> search for the PVT_SUBNET matching entry
  for line in $array
  do
    if [[ $line = $pattern* ]]
    then
     if [[ $line = *$PVT_SUBNET* ]]
     then
       line=`$SECHO $line | $SSED 's/</\n/g'`
       for entry in $line
       do
         if [[ $entry = *NAME\,* ]]
         then
           ifname=`$SECHO $entry | $SSED 's/>/\n/g' |  $SAWK -F"," '{print $2}'`

           #check if we have a logical alias or VLAN form of interface name
           #extract the physical interface name from such entries if any
           if [[ $ifname = *\.* ]]
           then
             ifname=`$SECHO $ifname  | $SAWK -F"." '{print $1}'`
           elif [[ $ifname = *\-* ]]
           then
             ifname=`$SECHO $ifname  | $SAWK -F"-" '{print $1}'`
           fi

           #perform regular expression match of the interface name
           if [[ $ifname =~ $nicNameWithWildCards ]]
           then
             addInterfaceToList $ifname
           fi
         fi
       done
     fi
   fi
  done
}

#Add interface to the list
addInterfaceToList()
{
  interfaceName=$1
  if [ "$interfaceName" != "" ]
  then
    NUMBER_OF_PVT_NICS=`expr $NUMBER_OF_PVT_NICS + 1`
    if [ "$PVT_NIC_ARRAY" = "" ]
    then
      PVT_NIC_ARRAY=$interfaceName
    else
      PVT_NIC_ARRAY=$PVT_NIC_ARRAY$'\n'$interfaceName
    fi
  fi
}

#method to retrieve the network interface list
getNetInterfaceList()
{
  # the interface list is passed in the form "SUBNET/INTERFACE,SUBNET/INTERFACE", We filter this input to retrieve the interface names
  pvtNetworkList=$1
  subnetWithIfList=`$SECHO $pvtNetworkList | $SSED 's/,/ /g'`
  for subnetwork in $subnetWithIfList
  do
    subnet=`$SECHO $subnetwork | $SSED 's/>/\n/g' |  $SAWK -F"/" '{print $1}'`
    interface=`$SECHO $subnetwork | $SSED 's/>/\n/g' |  $SAWK -F"/" '{print $2}'`
    #check if the interface follows wildcard pattern
    checkWildCards "$interface"
    #if interface name is wildcard pattern then resolve it by subnet
    if [ "$HAS_WILDCARD" = "TRUE" ]
    then
      processWildCardInterfaces $subnet "$interface"
    else
      addInterfaceToList $interface
    fi
  done

  if [ $NUMBER_OF_PVT_NICS -eq 0 ]
  then
    result="<RESULT>EFAIL</RESULT><EXEC_ERROR>Error while retrieving information about PRIVATE Interconnect list</EXEC_ERROR><TRACE>Unable to get the private interconnect list</TRACE><NLS_MSG><FACILITY>Prvg</FACILITY><ID>1512</ID><MSG_DATA></MSG_DATA></NLS_MSG>"
    existstatus=$ret
    $SECHO $result
    exit $existstatus
  fi
}

#method to add interface into the bad current value network interface list
addNICtoBadCurrentValNicList()
{
  nicToAdd=$1
  
  #check if the script is run to generate fixup. If yes then store complete
  #param name instead of just interface name
  if [ "$FIXUP_DATA_REQUESTED" = "TRUE" ]; then
    nicToAdd="current_net.ipv4.conf."$nicToAdd".rp_filter"
  fi

  if [ "X$badCurrentValNicList" = "X" ]; then
    badCurrentValNicList=$nicToAdd
  else
    badCurrentValNicList=$badCurrentValNicList$SEPARATOR$nicToAdd
  fi
}

#method to add interface into the bad configured value network interface list
addNICtoBadConfigValNicList()
{
  nicToAdd=$1
  #check if the script is run to generate fixup. If yes then store complete
  #param name instead of just interface name
  if [ "$FIXUP_DATA_REQUESTED" = "TRUE" ]; then
    nicToAdd="config_net.ipv4.conf."$nicToAdd".rp_filter"
  fi

  if [ "X$badConfigValNicList" = "X" ]; then
    badConfigValNicList=$nicToAdd
  else
    badConfigValNicList=$badConfigValNicList$SEPARATOR$nicToAdd
  fi
}

#Method to check and set the all.rp_filter consideration for kernel version 2.6.31 and further
checkRPFilterAllValueForMax()
{
  if [ $CURRENT_KERNEL_VER -ge $REQ_MIN_KER_VER_FOR_ALL_FILTER ]
  then
    #check the current value for "all"
    getRPFilterCurrentValue "all"
    ret=$?
    # make sure the command ran successfully and returned the rp_filter value for "all"
    if [ $ret -eq 0 ]; then
      currentVal=$_current_val
      if [ $currentVal -eq 2 ]; then
        CURR_ALL_RP_FILTER_MAX="TRUE"
      fi
    fi

    #check the configured value for "all"
    getRPFilterConfiguredValue "all"
    ret=$?
    # make sure the command ran successfully and returned the rp_filter value for "all" 
    if [ $ret -eq 0 ]; then
      configVal=$_config_val
      if [ $configVal -eq 2 ]; then
        CONF_ALL_RP_FILTER_MAX="TRUE"
      fi
    fi
  fi
}


# set up the values based on the version of the kernel 
checkRPFilterAllValueForMax

# we need to retrieve the network interface list depending on the stage we are in
getNetInterfaceList $1

if [ $NUMBER_OF_PVT_NICS -lt 2 ]	
  then	
    #If there is only one private interface then we need not check the param value and hence declare success	
    result="<RESULT>SUCC</RESULT><COLLECTED>not applicable</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not required to be checked on node $_HOST as there is only one private interfaces in use</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0457</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
    existstatus=0	
    $SECHO $result	
    exit $existstatus	
fi

#Now lets iterate through all the private NIC's to retrieve the rp_filter value set for each of them
for nicname in $PVT_NIC_ARRAY
do
  #Check the interface specific current value only if the all.rp_filter is not set to MAX (i.e. 2)
  if [ "$CURR_ALL_RP_FILTER_MAX" = "FALSE" ]; then
    #check the current value for this NIC
    getRPFilterCurrentValue $nicname
    ret=$?
    # make sure the command ran successfully and returned the rp_filter value for this interface
    if [ $ret -ne 0 ]; then
      # this means the value is either not set or could not be read so add this interface to bad NIC list
      addNICtoBadCurrentValNicList $nicname
    else
      currentVal=$_current_val
      if [ $currentVal != 0 ] && [ $currentVal != 2 ]; then
        # this means the interface has rp_filter value set to something incorrect,
        # lets add this interface to bad interface list
        addNICtoBadCurrentValNicList $nicname
      fi
    fi
  fi

  #Check the interface specific configured value only if the all.rp_filter is not set to MAX (i.e. 2)
  if [ "$CONF_ALL_RP_FILTER_MAX" = "FALSE" ]; then
    #check the configured value for this NIC
    getRPFilterConfiguredValue $nicname
    ret=$?
    # make sure the command ran successfully and returned the rp_filter value for this interface
    if [ $ret -ne 0 ]; then
      # this means the value is either not set or could not be read so add this interface to bad NIC list
      addNICtoBadConfigValNicList $nicname
    else
      configVal=$_config_val
      if [ $configVal != 0 ] && [ $configVal != 2 ]; then
        # this means the interface has rp_filter value set to something incorrect,
        # lets add this interface to bad interface list
        addNICtoBadConfigValNicList $nicname
      fi
    fi
  fi
done 


if [ "$FIXUP_DATA_REQUESTED" = "TRUE" ]; then
  if [ "X$badCurrentValNicList" != "X" ] &&  [ "X$badConfigValNicList" != "X" ]; then
    $SECHO "$badCurrentValNicList,$badConfigValNicList"
  elif [ "X$badCurrentValNicList" != "X" ]; then    
    $SECHO "$badCurrentValNicList"
  elif [ "X$badConfigValNicList" != "X" ]; then
    $SECHO "$badConfigValNicList"
  fi
  exit 0;
else
  if [ "X$badCurrentValNicList" = "X" ] && [ "X$badConfigValNicList" = "X" ]; then
    #This means all the private interconnect interfaces are having rp_filter parameter set correctly
    result="<RESULT>SUCC</RESULT><COLLECTED>$expected</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is correctly configured on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0452</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
    existstatus=0
    $SECHO $result
    exit $existstatus
  else
    if [ "X$badCurrentValNicList" != "X" ] &&  [ "X$badConfigValNicList" != "X" ]; then
      #i.e the configured value as well as current value is not set correctly for some private interfaces
      result="<RESULT>VFAIL</RESULT><COLLECTED>1</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not correctly set for interfaces ($badCurrentValNicList) and not correctly configured inside the /etc/sysctl.conf file for interfaces ($badConfigValNicList) on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0453</ID><MSG_DATA><DATA>$badCurrentValNicList</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG><NLS_MSG><FACILITY>Prve</FACILITY><ID>0456</ID><MSG_DATA><DATA>$badConfigValNicList</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
    elif [ "X$badCurrentValNicList" != "X" ]; then 
      #i.e the current value is not set correctly for some private interfaces
      result="<RESULT>VFAIL</RESULT><COLLECTED>1</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not correctly set for interfaces ($badCurrentValNicList) on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0453</ID><MSG_DATA><DATA>$badCurrentValNicList</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
    elif [ "X$badConfigValNicList" != "X" ]; then 
      #i.e the configured value is not set correctly for some private interfaces
      result="<RESULT>VFAIL</RESULT><COLLECTED>1</COLLECTED><EXPECTED>$expected</EXPECTED><TRACE>Reverse path filter parameter rp_filter is not correctly configured inside the /etc/sysctl.conf file for interfaces ($badConfigValNicList) on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0456</ID><MSG_DATA><DATA>$badConfigValNicList</DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
    fi
  
    #Display the result and exit with appropriate status
    existstatus=2
    $SECHO $result
    exit $existstatus
  fi
fi

