#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/check_ofsctl.sh /main/4 2015/05/05 18:56:03 iestrada Exp $
#
# check_ofsctl.sh
#
# Copyright (c) 2011, 2015, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      check_ofsctl.sh
#
#    DESCRIPTION
#      Pluggable verification framework shell script to check the ACFS 
#      device file '/dev/ofsctl'.
#
#      This check is needed on Linux, AIX and Solaris platforms.
#
#      The attributes for the 'ofsctl' file need to be: 0664:root:<asmadmin>
#      The file should look like:
#            brw-rw-r-- 1 root <adminID> 251, 0 Jun  6 00:32 /dev/ofsctl
#            ^
#
#            or
#
#            crw-rw-r-- 1 root <adminID> 251, 0 Jun  6 00:32 /dev/ofsctl
#            ^
#      Note: b=block (buffered) device special file 
#            c=character device 
#
#      <asmadmin> should be obtained from '$CRS_HOME/bin/osdbagrp -a'
#
#      On Linux only: the UDEV rule 'KERNEL=="ofsctl" ' must be verified
#      (checked) in file: /etc/udev/rules.d/rules.d/55-usm.rules. The 
#      rule should look like:
#               KERNEL=="ofsctl", GROUP="g900", MODE="0664"
#
#    NOTES
#      The "/dev/ofsctl" file is created from the installation tool
#      $ORACLE_HOME/bin/acfsroot. Internally the file is created using
#      the script /usm/src/cmds/internal/lin/usm_load_from_view.sh.
#
#      If the ACFS drivers are loaded:
#        It is a failure if the UDEV rule is not defined on Linux
#        It is a failure if the device file "/dev/ofsctl" is not present
#      
#    SYNTAX:
#      $check_ofsctl.sh $<CRS_HOME>
#
#    MODIFIED   (MM/DD/YY)
#    iestrada    04/29/15 - Fix bug 20851567 check for character device
#    iestrada    04/20/15 - Fix bug 20752216 add -L option to ls command
#    ptare       04/14/15 - Fix Bug#20880614 use = instead of == as it is not
#                           supported on HP
#    kfgriffi    03/15/14 - Creation
#
PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin
PLATFORM=`/bin/uname`
CRS_HOME=$1

SGREP="grep"
ECHO="echo"
SAWK="awk"
SED="sed"
LS="ls"
OFSCTL_FILE=/dev/ofsctl
UDEVCONF="/etc/udev/udev.conf"
UDEV_DEF_LOC="/etc/udev/rules.d"
UDEV_CONF_KEY="udev_rules="
UDEV_RULE_FILE="55-usm.rules"
UDEV_GROUP="Not Applicable"
UDEV_MODE="Not Applicable"
UDEV_CHECK_REQ=false
UDEV_CHECK_PASS=true
OFSCTL_CHECK_PASS=true

# UDEV mode must be equal to this value.
UDEV_MODE_REQ="0664"

# The '/dev/ofsctl' permissions must be equal to the string below. Note: 
# we are not just comparing the 'rwx' values we are also looking to make
# sure the first character is a 'b' or 'c' also. Also define required owner.
OFSCTL_PERM_REQ="brw-rw-r--"
OFSCTL_PERM_REQ_CHAR="crw-rw-r--"
OFSCTL_OWNER_REQ="root"
expected="[${OFSCTL_PERM_REQ} | ${OFSCTL_PERM_REQ_CHAR}]"

# The 'acfsutil' utility is located in the "/sbin" directory on Linux,
# AIX and SunOS. The 'acfsutil version' command is used to determine if 
# the ACFS drivers are installed. If the command fails the drivers are 
# NOT installed.
ACFSUTIL_LOC="/sbin/acfsutil"
ACFSUTIL_DRVR_INSTALL="${ACFSUTIL_LOC} version"

# The 'osdbagrp -a' command is used to obtain the ASM admin group needed
# to verify the device special file.
OSDBAGRP_LOC="${CRS_HOME}/bin/osdbagrp"
OSDBAGRP_CMD="${OSDBAGRP_LOC} -a"

# First get the current host name...
case $PLATFORM in
  Linux)
      HOST=`/bin/hostname`
  ;;
  SunOS | AIX)
      HOST=`/usr/bin/hostname`
  ;;
esac

#Internal function to initialize message strings
updateMessages()
{
  #SUCCESS: The ACFS drivers are not installed no check is needed.
  SUCCESSMSG1="<RESULT>SUCC</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>Check for /dev/ofsctl passed on node $HOST, ACFS drivers not installed.</TRACE>"

  #SUCCESS: The ACFS drivers are installed and we collected all the ofsctl data.
  SUCCESSMSG2="<RESULT>SUCC</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>Collected, and verified, 'ofsctl' verification data successfuly: OFSCTL_OWNER: ${OFSCTL_OWNER} OFSCTL_GROUP: ${OFSCTL_GROUP} OFSCTL_PERMISSIONS: ${OFSCTL_PERMS} OSDBAGRP: ${ASMADMIN} UDEV_GROUP: ${UDEV_GROUP} UDEV_MODE: ${UDEV_MODE} </TRACE>"

  ERRMSG1="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>Failed to execute the command '${OSDBAGRP_CMD}' successfully on node ${HOST}.</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0080</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"

  ERRMSG2="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>The verification check for device special file '/dev/ofsctl' failed. The file '/dev/ofsctl' is not present on node $HOST.</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0075</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"

  ERRMSG3="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>The UDEV rule for \"ofsctl\" was not found in the rule file '${UDEV_LOC}/${UDEV_RULE_FILE}' on node $HOST.</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0079</ID><MSG_DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"

  ERRMSG4="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>The UDEV rule specified in the '${UDEV_LOC}/${UDEV_RULE_FILE}' file and the device special file attributes do not meet the expected requirements on node $HOST.\n[UDEV group: Expected=${ASMADMIN}; Found=${UDEV_GROUP}] [UDEV mode: Expected=${UDEV_MODE_REQ}; Found=${UDEV_MODE}]\n[OFSCTL permissions: Expected=${expected}; Found=${OFSCTL_PERMS}] [OFSCTL owner: Expected=${OFSCTL_OWNER_REQ}; Found=${OFSCTL_OWNER}] OFSCTL group: Expected=${ASMADMIN}; Found=${OFSCTL_GROUP}]</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0083</ID><MSG_DATA><DATA>${HOST}</DATA><DATA>${ASMADMIN}</DATA><DATA>${UDEV_GROUP}</DATA><DATA>${UDEV_MODE_REQ}</DATA><DATA>${UDEV_MODE}</DATA></MSG_DATA></NLS_MSG><NLS_MSG><FACILITY>Prve</FACILITY><ID>0082</ID><MSG_DATA><DATA>$HOST</DATA><DATA>${expected}</DATA><DATA>${OFSCTL_PERMS}</DATA><DATA>${OFSCTL_OWNER_REQ}</DATA><DATA>${OFSCTL_OWNER}</DATA><DATA>${ASMADMIN}</DATA><DATA>${OFSCTL_GROUP}</DATA></MSG_DATA></NLS_MSG>"

  ERRMSG5="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>The device special file attributes do not meet the expected requirements on node $HOST.\n [OFSCTL permissions: Expected=${expected}; Found=${OFSCTL_PERMS}] [OFSCTL owner: Expected=${OFSCTL_OWNER_REQ}; Found=${OFSCTL_OWNER}] OFSCTL group: Expected=${ASMADMIN}; Found=${OFSCTL_GROUP}]</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0082</ID><MSG_DATA><DATA>$HOST</DATA> <DATA>${expected}</DATA><DATA>${OFSCTL_PERMS}</DATA><DATA>${OFSCTL_OWNER_REQ}</DATA><DATA>${OFSCTL_OWNER}</DATA><DATA>${ASMADMIN}</DATA><DATA>${OFSCTL_GROUP}</DATA></MSG_DATA></NLS_MSG>"

  ERRMSG6="<RESULT>VFAIL</RESULT><EXPECTED></EXPECTED><COLLECTED></COLLECTED><TRACE>The UDEV rule specified in the '${UDEV_LOC}/${UDEV_RULE_FILE}' file do not meet the expected requirements on node $HOST.\n[UDEV group: Expected=${ASMADMIN}; Found=${UDEV_GROUP}] [UDEV mode: Expected=${UDEV_MODE_REQ}; Found=${UDEV_MODE}]</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0083</ID><MSG_DATA><DATA>${HOST}</DATA><DATA>${ASMADMIN}</DATA><DATA>${UDEV_GROUP}</DATA><DATA>${UDEV_MODE_REQ}</DATA><DATA>${UDEV_MODE}</DATA></MSG_DATA></NLS_MSG>"

}

errorout()
{
  updateMessages
  ${ECHO} ${RESULTMSG}
  exit 1
}

exitSucc() 
{
  updateMessages
  ${ECHO} ${RESULTMSG}
  exit 0
}

#Script execution begins here
updateMessages

#Assume the success message to be default first
RESULTMSG=${SUCCESSMSG1}

# Make sure the ACFS utility file is available, if not then ACFS
# has probably not been installed. 
if [ ! -f ${ACFSUTIL_LOC} ]
then
  # The ACFS utility file is not present so ACFS has not been loaded on
  # the system so report success and indicate in trace that ACFS drivers
  # not installed.
  exitSucc
fi

# If we get here then the ACFS utility is present, now check to see if
# the drivers are installed. If we get a failure from the command
# execution then the drivers are not installed...
DRIVER_INSTALLED=`${ACFSUTIL_DRVR_INSTALL}`
RET=$?
if [ $RET -ne 0 ]
then
  # The ACFS utility is present but the drivers are not installed. 
  # Return success indicating so.
  exitSucc
fi

# The ACFS utility is present and the drivers are installed therefore we
# MUST have the following:
#   1. The <asmadmin> group ID must be available from 'osdbagrp -a'
#   2. The '/dev/ofsctl' file must be present
#   3. The UDEV rule must be present [Linux only]

# Is 'osdbagrp' in the right place and can we execute it?
if [ ! -f ${OSDBAGRP_LOC} ]
then
  # Update messages with collected data...
  updateMessages
  RESULTMSG=${ERRMSG1}
  errorout
fi

ASMADMIN=`${OSDBAGRP_CMD}`
RET=$?
if [ $RET -ne 0 ]
then
  # Update messages with collected data...
  updateMessages
  RESULTMSG=${ERRMSG1}
  errorout
fi

OFSCTL_LINE=`${LS} -lL ${OFSCTL_FILE}`
RET=$?

# Is the '/dev/ofsctl' file present?
if [ $RET -eq 0 ]
then
  # The '/dev/ofsctl' file should exist, get owner, group and perms...
  # We are expecting the file to look something like:
  #     brw-rw-r-- 1 root asmadmin 251, 0 Jun  6 00:32 /dev/ofsctl
  #
  # Note: On Linux the file permissions may contain a trailing "." character,
  #       make sure it is removed before comparison...
  OFSCTL_LINE=`${LS} -lL ${OFSCTL_FILE}`
  OFSCTL_PERMS=`${ECHO} ${OFSCTL_LINE} | ${SAWK} '{ print $1 }' | sed 's/\./ /'`
  OFSCTL_OWNER=`${ECHO} ${OFSCTL_LINE} | ${SAWK} '{ print $3 }'`
  OFSCTL_GROUP=`${ECHO} ${OFSCTL_LINE} | ${SAWK} '{ print $4 }'`
else
  # The '/dev/ofsctl' file is not present and it is expected, indicate 
  # this in the error result...
  # Update messages with collected data...
  updateMessages
  RESULTMSG=${ERRMSG2}
  errorout
fi

# Check if we are on Linux, if we are we need to handle UDEV
case ${PLATFORM} in
  Linux)
    UDEV_LOC=`${SGREP} ${UDEV_CONF_KEY} ${UDEVCONF} | ${SAWK} -F= ' { print $2 }'`
    RET=$?
    if [ $RET -ne 0 ] || [ "X${UDEV_LOC}" = "X" ]
    then
      UDEV_LOC="${UDEV_DEF_LOC}"
    fi

    # If the UDEV rule file is not present (i.e. 55-usm.rules) return an 
    # error that the UDEV rule could not be found.
    if [ ! -f "${UDEV_LOC}/${UDEV_RULE_FILE}" ]
    then
      # Update messages with collected data...
      updateMessages
      RESULTMSG=${ERRMSG3}
      errorout
    fi

    # UDEV_RULE_LINE should look like:
    #       KERNEL=="ofsctl", GROUP="g900", MODE="0664"
    UDEV_RULE_LINE=`${SGREP} 'KERNEL=="ofsctl\"' ${UDEV_LOC}/${UDEV_RULE_FILE}`
    RET=$?
    if [ $RET -ne 0 ]
    then
      # Update messages with collected data...
      updateMessages
      RESULTMSG=${ERRMSG3}
      errorout
    fi

    UDEV_GROUP=`${ECHO} ${UDEV_RULE_LINE} | ${SAWK} '{ print $2 }' | ${SED} 's/"/ /' | ${SAWK} '{ print $2 } ' | ${SED} 's/"/ /' | ${SAWK} '{ print $1 }'`
    UDEV_MODE=`${ECHO} ${UDEV_RULE_LINE} | ${SAWK} '{ print $3 }' | ${SED} 's/"/ /' | ${SAWK} '{ print $2 } ' | sed 's/"/ /' | ${SAWK} '{ print $1 }'`

    UDEV_CHECK_REQ=true

  ;;
  SunOS | AIX)
    # Not on Linux so do not process UDEV info...
  ;;
esac

# Update messages with collected data...
updateMessages

# If there were any issues found with collecting the required data then 
# those errors would have already been reported and we would not have 
# reached this point. Now verify all the required data...

# First check to see if we need to verify UDEV and if so then do so...
if [ ${UDEV_CHECK_REQ} = true ]
then
  if [ ${UDEV_MODE_REQ} != ${UDEV_MODE} ] || [ ${ASMADMIN} != ${UDEV_GROUP} ]
  then
    UDEV_CHECK_PASS=false
  fi
fi

# Now verify the '/dev/ofsctl' data...
if ([ ${OFSCTL_PERM_REQ} != ${OFSCTL_PERMS} ] && [ ${OFSCTL_PERM_REQ_CHAR} != ${OFSCTL_PERMS} ]) ||
   [ ${ASMADMIN} != ${OFSCTL_GROUP} ] ||
   [ ${OFSCTL_OWNER_REQ} != ${OFSCTL_OWNER} ]
then
  OFSCTL_CHECK_PASS=false
fi

# Report results...
if [ ${UDEV_CHECK_PASS} = false -a ${OFSCTL_CHECK_PASS} = false ]
then
  # Both checks failed, report the error...
  # Update messages with collected data...
  updateMessages
  RESULTMSG=$ERRMSG4
  errorout
fi

# If both checks failed then we would not get here, so look for the 
# OFSCTL file check failure.
if [ ${OFSCTL_CHECK_PASS} = false ]
then
  # The device file special check failed, report the error. 
  # Update messages with collected data...
  updateMessages
  RESULTMSG=$ERRMSG5
  errorout
fi

# If the OFSCTL check did not fail then check for the UDEV verification.
if [ ${UDEV_CHECK_PASS} = false ]
then
  # The UDEV check failed, report the error. 
  RESULTMSG=$ERRMSG6
fi

# Everything passed so report the success!
RESULTMSG=$SUCCESSMSG2
exitSucc

