#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/check_nfsserver_options.sh /st_has_pt-cvuotn12201/1 2017/04/13 08:53:40 shhirema Exp $
#
# check_nfsserver_options.sh
#
# Copyright (c) 2016, 2017, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      check_nfsserver_options.sh - check 'insecure' nfs server exports option
#
#    DESCRIPTION
#      Check to warn users if the nfs used for datafile is exported without
#      'insecure' option. The option is necessary if dnfs is enabled for the
#       Oracle home.
#
#    NOTES
#      Takes three arguments.
#      1. Datafile path
#      2. NFS server export for the datafile.
#      3. NFS mount options with which datafile nfs is mounted.
#
#    MODIFIED   (MM/DD/YY)
#    shhirema    12/08/16 - Creation
#

PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin
HOST=`hostname`
PLATFORM=`/bin/uname`
MOUNT="mount"
UMOUNT="umount"
MOUNT_POINT="/mnt/cvuchk_nfsserver"
MKDIR="/bin/mkdir"

# nfs mount configuration is passed as executable arguments
DATAFILE_PATH=$1
NFS_EXPORT=$2
NFS_MOUNT_OPTIONS=$3

if [ -z "$DATAFILE_PATH" ]
then
  result="<RESULT>VFAIL</RESULT><TRACE>Argument is empty</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0018</ID></NLS_MSG>"
  echo $result
  exit
fi

# if the entries are present in ORACLE_HOME dbs/oradnfstab then the path was 
# mounted using oradnfs
if [ "X$NFS_EXPORT" == "XNA" -o "X$NFS_MOUNT_OPTIONS" == "XNA" ]
then
  echo "<RESULT>SUCC</RESULT><TRACE>The path ${DATAFILE_PATH} is DNFS mounted.</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10242</ID></NLS_MSG>"
  exit
fi

MOUNT_OPTIONS=$NFS_MOUNT_OPTIONS

if [[ $NFS_MOUNT_OPTIONS == *"noresvport"* ]]
then
  # the path was already mounted with noresvport, so the server is already
  # allowing non privileged source port connections necessary for dNFS
  echo "<RESULT>SUCC</RESULT><TRACE>The path ${DATAFILE_PATH} is already mounted with noresvport option</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10242</ID></NLS_MSG>"
  exit
else
  if [[ $NFS_MOUNT_OPTIONS == *"resvport"* ]]
  then
    MOUNT_OPTIONS=${NFS_MOUNT_OPTIONS/resvport/noresvport}
  else
    MOUNT_OPTIONS=${NFS_MOUNT_OPTIONS},noresvport
  fi
fi

# cleanup if the mount point was not cleaned up
# during previous checks
if [ -e $MOUNT_POINT ]
then
  $UMOUNT $MOUNT_POINT 2>/dev/null
else
  $MKDIR -p $MOUNT_POINT
fi

MOUNT_CMD=
retd=0
case $PLATFORM in
  Linux)
    MOUNT_CMD="$MOUNT -t nfs $NFS_EXPORT $MOUNT_POINT -o $MOUNT_OPTIONS"
    ;;
  SunOS)
    MOUNT_CMD="$MOUNT -F nfs $NFS_EXPORT $MOUNT_POINT -o $MOUNT_OPTIONS"
    ;;
esac

ret=0
MOUNT_OP=`$MOUNT_CMD 2>&1`
ret=$?

if [ $ret -ne 0 ]
then
  echo "RESULT>VFAIL</RESULT><TRACE>Failed with error \"${MOUNT_OP}\" to mount ${NFS_EXPORT} on ${HOST}:${MOUNT_POINT} with options ${MOUNT_OPTIONS}</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10243</ID><MSG_DATA><DATA>${NFS_EXPORT}</DATA><DATA>${MOUNT_POINT}</DATA></MSG_DATA><MSG_ADD>${MOUNT_OP}</MSG_ADD></NLS_MSG>"
else 
  echo "<RESULT>SUCC</RESULT><TRACE>Successfully mounted the ${NFS_EXPORT} on ${HOST}:${MOUNT_POINT} with options ${MOUNT_OPTIONS}</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10242</ID></NLS_MSG>"
  # cleanup
  $UMOUNT $MOUNT_POINT 2>/dev/null
fi
