#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/check_jumbo_frames.sh /st_has_pt-cvuotn12201/1 2018/03/04 22:37:59 shhirema Exp $
#
# check_jumbo_frames.sh
#
# Copyright (c) 2010, 2018, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      check_jumbo_frames.sh - Jumbo frames configuration for interconnect
#
#    DESCRIPTION
#      Script to check jumbo frames configuration for interconnect
#
#    NOTES
#      This check uses hard coded value of 7000 for MTU when interface type
#      is infiniband 
#
#    MODIFIED   (MM/DD/YY)
#    ptare       02/13/18 - Fix Bug#27516078 compare the actual value to see if
#                           it is greater
#    maboddu     12/11/15 - Fix bug#22305002 - fix syntax error
#    ptare       09/02/15 - Fix Bug#21767756 Consider the ifconfig output
#                           diplaying mtu using space separator
#    ptare       04/14/15 - Fix Bug#20880614 use = instead of == as it is not
#                           supported on HP
#    shshrini    04/09/15 - Fix bug 20803442 - Use cut command directly instead
#                           of using /bin/cut
#    shshrini    02/25/15 - Enhance network checks, project 47212
#    ptare       02/04/15 - Bug#20398052 use 7000 as expected value for
#                           infiniband interfaces
#    nvira       06/17/14 - review comment
#    nvira       06/17/14 - bug fix 18990354, use nawk instead of awk
#    fjlee       09/27/13 - bug 17500747
#    dsaggi      08/29/13 - XbranchMerge dsaggi_bug-17309882 from st_has_11.2.0
#    nvira       10/10/12 - bug fix 14364702
#    nvira       06/06/12 - fix message
#    nvira       08/25/10 - pluggable task script to check jumbo frame settings
#    nvira       08/25/10 - Creation
#

PATH=/bin:/usr/bin:/sbin:/usr/sbin:/etc

SGREP="/bin/grep"
SAWK="/bin/awk"
SCUT="cut"
SNETSTAT="netstat"
SUNIQ="/usr/bin/uniq"
STAIL="/usr/bin/tail"
SSED="/bin/sed"
SIFCONFIG="/sbin/ifconfig"
SNWMGR="nwmgr"

#returns list of all interfaces
listOfInterfaces()
{
  INTERFACE_LIST=`$SNETSTAT -i | $SSED '1d' | cut -d ' ' -f 1 | cut -d ':' -f 1 | sort -u`
}

#Method to determine if the specified interface is infiniband type
#RETURN: 1 if the specified interface was found to be infiniband type, 0 otherwise
isInfiniBandInterface ()
{
  ADAPTER="<NAME,$1>"
  array=""
  #Split and align the exectask output
  for line in $exectask_output
  do
    if [[ $line = $ADAPTER_PATTERN* ]]
    then
     array=$array$'\n'$line
    else
     array=$array$line
    fi
  done

  #Now process each adopter entry
  for line in $array
  do
    if [[ $line = $ADAPTER_PATTERN* ]]
    then
      if [[ $line = *$ADAPTER* ]] && [[ $line = *$INFINIBAND_YES* ]]; then
         #Interface is infiniband return 1 
         return 1;
      fi
    fi
  done
  #interface is not infiniband return 0
  return 0;
}

#expand list of interfaces for any wildcard characters 
expandInterfaces()
{
  ARG=$1
  EXPANDED_LIST=`echo $INTERFACE_LIST| $SAWK '{gsub(" ","\n", $0); print}' | $SSED "s/$/ $ARG/" | $SAWK '{if ($1 ~ $2) print $1 }'`
}

verifyInterface()
{
interfaceName=$1
#first check if there is any interface specific value defined for the parameter

case $PLATFORM in
  Linux)
      # On Linux platform the ifconfig command displays MTU in following two different formats
      # MTU:1500 OR mtu 1500
      # We must handle both the cases
      paramValue=`$SIFCONFIG $interfaceName | $SGREP -i -e "mtu[: ][0123456789]*" -o | $SCUT -c5-`
  ;;
  SunOS)
      paramValue=`$SIFCONFIG $interfaceName | $SGREP -i mtu | $SAWK '{print \$4}'`
  ;;
  AIX)
      paramValue=`$SNETSTAT -I $interfaceName | $SSED '1d' | $SAWK '{print \$2}' | $STAIL -1`
  ;;
  HP-UX)
      paramValue=`$SNWMGR -g -A mtu -c $interfaceName | $SSED '1d' | $SAWK '{print \$3}'`
  ;;
esac
ret=$?

if [ $ret -eq 0 ]
then
  if [ $paramValue -ge $expected ]
  then 
    ERROR_CODE=0
  else
    ERROR_CODE=3
  fi   
else
  ERROR_CODE=4
fi   
return $ERROR_CODE
}


PLATFORM=`/bin/uname`

case $PLATFORM in
  Linux)
      _HOST=`/bin/hostname`
  ;;
  AIX | HP-UX)
      _HOST=`/usr/bin/hostname`
  ;;
  SunOS)
      SAWK="nawk"
      _HOST=`/usr/bin/hostname`
  ;;
esac

CRS_HOME=$1
requiredValue=$2
INFINIBAND_YES="<HARDWARE_TYPE,32>"
ADAPTER_PATTERN="<ADAPTER>"
currentDir=`dirname $0`
exectask_output=`$currentDir/exectask -getifinfo`
#align the exectask_output of exectask to be lines starting with <ADAPTER> as this is what we are looking for
exectask_output=`echo $exectask_output | sed 's/<ADAPTER>/\n&/g'`

# Set default exit message to indicate failure.
result="<RESULT>EFAIL</RESULT><EXPECTED>$EXPECTED</EXPECTED><EXEC_ERROR>Unable to get the Jumbo Frames setting information on the system</EXEC_ERROR><TRACE>Unable to get the Jumbo Frames setting information on the system</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0294</ID><MSG_DATA><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"

listOfInterfaces

# ensure that the clusterware stack is up and that the cluster interfaces can be retrieved using oifcfg

interfaceNames=`$CRS_HOME/bin/oifcfg getif | $SGREP cluster_interconnect | $SAWK '{print \$1}'`
ret=$?
if [ $ret -ne 0 -o "X${interfaceNames}" = "X" ]
then
  echo $result
  exit 0
fi
#strip off any new line characters
interfaceNames=`echo $interfaceNames`

#expand any wild cards
expandedList=""
for interfaceName in $interfaceNames; do
  expandInterfaces "$interfaceName"
    expandedList=`echo "$expandedList $EXPANDED_LIST"`
done

expandedList=`echo $expandedList |  $SAWK '{gsub("\n"," ", $0); print}'`


FAILED_INTERFACES=""
FAILED_COLLECTED=""

for interfaceName in $expandedList; do

 #Check if the interface is infiniband type
 isInfiniBandInterface "$interfaceName"
 ret=$?

 if [ $ret -eq 1 ]; then
   #INTERFACE is infiniband
   #Use 7000 for infiniband interfaces, refer Bug#20398052
   expected=7000
 else
   #NOT infiniband"
   expected=$requiredValue
 fi

 #update the EXPECTED value string for this interface
 if [ -n "$EXPECTED" ]
 then
   EXPECTED=`echo "$EXPECTED;"`
 fi
 EXPECTED=`echo "$EXPECTED$interfaceName=$expected"`

 verifyInterface "$interfaceName"
 ret=$?

 #update the COLLECTED value for this interface
 if [ -n "$COLLECTED" ]
 then
   COLLECTED=`echo "$COLLECTED;"`
 fi
 COLLECTED=`echo "$COLLECTED$interfaceName=$paramValue"`

 if [ $ret -ne 0 ]
 then
   if [ -n "$FAILED_INTERFACES" ]
   then
    FAILED_INTERFACES=`echo "$FAILED_INTERFACES,"`
    FAILED_COLLECTED=`echo "$FAILED_COLLECTED;"`
   fi
   FAILED_INTERFACES=`echo "$FAILED_INTERFACES$interfaceName"`
   FAILED_COLLECTED=`echo "$FAILED_COLLECTED$interfaceName=$paramValue"`
 fi
done

if [ "X$FAILED_INTERFACES" = "X" -a "X$COLLECTED" != "X" ]
then
    result="<RESULT>SUCC</RESULT><COLLECTED>$COLLECTED</COLLECTED><EXPECTED>$EXPECTED</EXPECTED><TRACE>Jumbo frames or mini jumbo frames are configured for interconnect on node $_HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0292</ID><DATA>$_HOST</DATA></MSG_DATA></NLS_MSG>"
else
    result="<RESULT>VFAIL</RESULT><COLLECTED>$FAILED_COLLECTED</COLLECTED><EXPECTED>$EXPECTED</EXPECTED><TRACE>Jumbo frames are not configured for interfaces \"$FAILED_INTERFACES\" on node $_HOST.[Expected=$EXPECTED; Found=$FAILED_COLLECTED]</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0293</ID><MSG_DATA><DATA>$_HOST</DATA><DATA>$EXPECTED</DATA><DATA>$FAILED_COLLECTED</DATA><DATA>$FAILED_INTERFACES</DATA></MSG_DATA></NLS_MSG>"    
fi

echo $result
exit 0
