#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/check_afd_drivers.sh /st_has_12.2.0.1.0/1 2016/08/19 11:24:45 ptare Exp $
#
# check_afd_drivers.sh
#
# Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      check_afd_drivers.sh - check if afd driver files are present
#
#    DESCRIPTION
#      Checks for ASM filter driver files in a pre install/upgrade scenario
#
#    NOTES
#      takes three arguments.
#      1. AFD driver name
#      2. AFD lib file name
#      3. AFD configuration file - passed to the script only in pre-upgrade 
#         scenario in a valid upgrade path ( AFD being supported for the source
#         grid version on the current platform )
#
#    MODIFIED   (MM/DD/YY)
#    ptare       08/15/16 - Report warning and not error
#    shhirema    07/12/16 - Bug 23633769: check for driver files
#    shhirema    05/05/16 - Bug 23229324: check for drivers loaded
#    shhirema    01/21/16 - Creation
#

PATH=/bin:/usr/bin:/sbin:/usr/sbin:/usr/local/bin
FIND="/bin/find"
SED="/bin/sed"
HOST=`hostname`
PLATFORM=`/bin/uname`
GREP="/bin/grep"
SAWK="/bin/awk"

# set library/driver file locations and names

# names of files to be checked are passed as executable arguments
DRV_NAME=$1
LIB_FILE_NAME=$2
AFD_CONF_FILE=$3

# if AFD configuration file (specified only in pre upgrade scenarios)
# is found to be present then report success.

if [ -n "$AFD_CONF_FILE" ]
then
  if [ -f $AFD_CONF_FILE ]
  then
    echo "<RESULT>SUCC</RESULT><TRACE>AFD configuration file \"$AFD_CONF_FILE\" present in a pre upgrade scenario</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10236</ID></NLS_MSG>"
    exit
  fi
fi

LIB_FILE_LOC='/opt/oracle/extapi'

#check if the argument is passed or not. if not then exit with the error.
if [ -z "$DRV_NAME" -o -z "$LIB_FILE_NAME" ]
then
  result="<RESULT>VFAIL</RESULT><TRACE>Argument is empty</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0018</ID></NLS_MSG>"
  echo $result
  exit
fi

LIB_FILES=
DRV_LOADED=
SEP=

# check if there are any stray AFD files or drivers loaded
retd=0
case $PLATFORM in
  Linux)
    LISTMOD="/sbin/lsmod"
    SAWK="/bin/awk"
    DRV_FILE_NAME=${DRV_NAME}.ko
    DRV_FILE_LOC="/lib/modules"
    DRV_LOADED=`$LISTMOD | $GREP $DRV_NAME | $SAWK '{print $1}' 2>/dev/null`
    ;;
  SunOS)
    LISTMOD="/usr/sbin/modinfo"
    SAWK="/usr/xpg4/bin/awk"
    DRV_FILE_NAME=$DRV_NAME
    DRV_FILE_LOC="/usr/kernel/drv"
    DRV_LOADED=`$LISTMOD | $GREP $DRV_NAME | $SAWK '{print $6}' 2>/dev/null`
    ;;
esac

retd=$?

AFD_FILES=
DRV_FILES=
LIB_FILES=
SEP=

FIND_DRV_COMMAND="$FIND $DRV_FILE_LOC -name $DRV_FILE_NAME"
FIND_LIB_COMMAND="$FIND $LIB_FILE_LOC -name $LIB_FILE_NAME"

retdf=0
# check if there are any stray AFD files
if [ -d "$DRV_FILE_LOC" ]
then
  DRV_FILES=`$FIND_DRV_COMMAND | $SED -e ':a' -e '$!N' -e '$!ba' -e 's/\n/,/g' 2>/dev/null`
  retdf=$?
fi

retl=0
if [ -d "$LIB_FILE_LOC" ]
then
  LIB_FILES=`$FIND_LIB_COMMAND | $SED -e ':a' -e '$!N' -e '$!ba' -e 's/\n/,/g' 2>/dev/null`
  retl=$?
fi

if [ $retd -ne 0 -o $retl -ne 0 -o $retdf -ne 0 ]
then
  # report operation failure
  echo "<RESULT>EFAIL</RESULT><TRACE>Error while checking for ASM Filter driver configuration</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10238</ID><MSG_DATA><DATA>${FIND_LIB_COMMAND}, ${LISTMOD} | $GREP $DRV_NAME</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  exit 1 
fi

if [ "X$DRV_FILES" != "X" -a "X$LIB_FILES" != "X" ]
then
  SEP=','
fi
AFD_FILES=$DRV_FILES$SEP$LIB_FILES

TRACE_LIB_FOUND=
ERROR_MSG_LIB=
TRACE_DRV_FOUND=
ERROR_MSG_DRV=

if [ "X$AFD_FILES" != "X" -o "X$DRV_LOADED" != "X" ]
then
  ERROR_STATUS="<RESULT>WARN</RESULT>";

  if [ "X$AFD_FILES" != "X" ]
  then
    TRACE_LIB_FOUND="ASM Filter driver files \"$AFD_FILES\" were found on node \"$HOST\"."
    ERROR_MSG_LIB="<NLS_MSG><FACILITY>Prve</FACILITY><ID>10237</ID><MSG_DATA><DATA>$HOST</DATA><DATA>$AFD_FILES</DATA></MSG_DATA></NLS_MSG>"
  fi
  if [ "X$DRV_LOADED" != "X" ]
  then
    TRACE_DRV_FOUND="ASM Filter driver \"$DRV_LOADED\" was found loaded on the node \"$HOST\"."
    ERROR_MSG_DRV="<NLS_MSG><FACILITY>Prve</FACILITY><ID>10239</ID><MSG_DATA><DATA>$HOST</DATA><DATA>$DRV_LOADED</DATA></MSG_DATA></NLS_MSG>"
  fi

  echo "${ERROR_STATUS}<TRACE>${TRACE_LIB_FOUND}${TRACE_DRV_FOUND}</TRACE>${ERROR_MSG_LIB}${ERROR_MSG_DRV}"
else
  echo "<RESULT>SUCC</RESULT><TRACE>ASM Filter driver configuration was not found</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10236</ID></NLS_MSG>"
fi

