#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/checkLogindIPCSettings.sh /main/1 2015/12/14 22:20:57 maboddu Exp $
#
# checkLogindIPCSettings.sh
#
# Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      checkLogindIPCSettings.sh - checks the RemoveIPC option is enabled or not
#
#    DESCRIPTION
#      Checks RemoveIPC option in /etc/systemd/logind.conf and report error
#      if it is enabled on OL/RHEL 7.2
#
#    NOTES
#      In RHEL7.2 a new feature was enabled to remove all ipc segments when the 
#      last user logs out. This is controlled by RemoveIPC in 
#      /etc/systemd/logind.conf. The default value for RemoveIPC in RHEL7.2 is 
#      YES, meaning, that when last oracle/grid user logs out, the SGAs and IPC 
#      segments will be removed and cause crs/asm instances to crash.
#
#    MODIFIED   (MM/DD/YY)
#    maboddu     12/04/15 - Fix bug#22275121 - check RemoveIPC is not enabled
#    maboddu     12/04/15 - Creation
#

LOGIND_CONF="/etc/systemd/logind.conf"
IPC_PARAM="RemoveIPC"
SGREP="/bin/grep"
SSED="/bin/sed"
SAWK="/bin/awk"
HOST=`/bin/hostname -s`

checkRemoveIPCParameter()
{
  if [ -f "$LOGIND_CONF" ]
  then
     cmd="$SGREP \"^[^#]\" $LOGIND_CONF | $SGREP \"$IPC_PARAM\" | $SSED 's/[[:space:]]//g' | $SAWK -F\"=\" '{print \$2}'"
     output=$(/bin/sh -c "$cmd")
  
     if [ "X$output" = "XNO" ] || [ "X$output" = "Xno" ]
     then
        # RemoveIPC parameter is not enabled in /etc/systemd/logind.conf. 
        # Report success
        ERRCODE=0
     elif [ "X$output" = "XYES" ] || [ "X$output" = "Xyes" ]
     then
        # RemoveIPC parameter is enabled in /etc/systemd/logind.conf. 
        # Report error
        ERRCODE=1
     else
        # RemoveIPC parameter enabled or removed or commented out in 
        # /etc/systemd/logind.conf
        # By default the RemoveIPC option enabled. Report the error.
        ERRCODE=2
     fi
  else
     #logind.conf file does not exist. Report success
     ERRCODE=0
  fi
}

# check the RemoveIPC option in /etc/systemd/logind.conf

checkRemoveIPCParameter

if [ "$ERRCODE" = "0" ] 
then
   result="<RESULT>SUCC</RESULT><COLLECTED>$output</COLLECTED><EXPECTED>no</EXPECTED><TRACE>systemd login manager parameter $IPC_PARAM disabled on node $HOST</TRACE>"
elif [ "$ERRCODE" = "1" ]
then
   result="<RESULT>VFAIL</RESULT><COLLECTED>$output</COLLECTED><EXPECTED>no</EXPECTED><TRACE>systemd login manager parameter $IPC_PARAM enabled on node $HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10232</ID><MSG_DATA><DATA>$LOGIND_CONF</DATA><DATA>$HOST</DATA><DATA>$output</DATA></MSG_DATA></NLS_MSG>"
else
   result="<RESULT>VFAIL</RESULT><EXPECTED>no</EXPECTED><TRACE>systemd login manager parameter $IPC_PARAM entry not found or commented on node $HOST</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10233</ID><MSG_DATA><DATA>$LOGIND_CONF</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
fi

echo $result






