#!/bin/sh
#
# $Header: opsm/cvutl/pluggable/unix/checkFSMountOptions.sh /main/4 2016/04/20 09:05:49 maboddu Exp $
#
# checkFSMountOptions.sh
#
# Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      checkFSMountOptions.sh - checks mount options
#
#    DESCRIPTION
#      checks for nosuid mount option
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    maboddu     03/31/16 - Check file system mount options correctly
#    maboddu     09/30/15 - Fix bug#21921909 - Get the correct file system
#    prajm       09/14/15 - fix bug#20534196 - Reporting error in case the path
#                           is null
#    maboddu     08/28/14 - Fix bug#9664096 - check mount options
#    maboddu     08/28/14 - Creation
#

SGREP="/bin/grep"
SAWK="/bin/awk"
SED="/bin/sed"
HEAD="/usr/bin/head"
DF="/bin/df"
UNAME="/bin/uname"
MOUNT_LINUX="/bin/mount"
MOUNT_HP_SOL="/sbin/mount"
MOUNT_AIX="/usr/sbin/mount"

NOSUID="nosuid"
NOSETUID="nosetuid"

PLATFORM=`$UNAME`
HOST=`$UNAME -n`

#File system path
FS_PATH=$1

getFileSystem()
{
  # gets the file system for the path
  case $PLATFORM in
    Linux | AIX)
       FILE_SYSTEM=`$DF $1 | $SED 1d | $HEAD -1 | $SAWK '{ print $1 }'`
    ;;
    SunOS | HP-UX)
       FILE_SYSTEM=`$DF $1 | $SAWK -F'(' '{print $2}' | $SAWK -F')' '{print $1}'`
    ;;
  esac
}

checkNOSUIDMountOption()
{
  # checks the mount option nosuid
  case $PLATFORM in
    Linux)
       MOUNT_OPTION=`$MOUNT_LINUX | $SGREP "$1[[:space:]]" | $SGREP -w $1 | $SGREP -i $NOSUID`
    ;;
    SunOS)
       MOUNT_OPTION=`$MOUNT_HP_SOL | $SGREP "$1[[:space:]]" | $SGREP -w $1 | $SGREP -i $NOSETUID`
    ;;
    HP-UX)
       MOUNT_OPTION=`$MOUNT_HP_SOL | $SGREP "$1[[:space:]]" | $SGREP -w $1 | $SGREP -i $NOSUID`
    ;;
    AIX)
       # skip the node info from the file system
       if [[ $1 = *:* ]]
       then
         FS=`echo $1 | $SAWK -F':' '{ print $2 }'`
       else
         FS=$1
       fi
       MOUNT_OPTION=`$MOUNT_AIX | $SGREP "$FS[[:space:]]" | $SGREP -w $FS | $SGREP -i $NOSUID`
    ;;
  esac
}

#check if the argument is passed or not if not then exit with the error.
if [ -z "$FS_PATH" ]
then
  result="<RESULT>VFAIL</RESULT><TRACE>Argument is empty</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>0018</ID></NLS_MSG>"
  echo $result
  exit
fi

# Check for nosuid mount option for the given path
getFileSystem $FS_PATH

if [ "X$FILE_SYSTEM" != "X" ]
then
  checkNOSUIDMountOption $FILE_SYSTEM
   
  if [ "X$MOUNT_OPTION" = "X" ]
  then
    result="<RESULT>SUCC</RESULT><TRACE>File system $FS_PATH is not mounted with nosuid option on \"$HOST\"</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10182</ID><MSG_DATA><DATA>$FS_PATH</DATA></MSG_DATA></NLS_MSG>"
  else
    result="<RESULT>VFAIL</RESULT><TRACE>File System $FS_PATH is mounted with nosuid option on \"$HOST\"</TRACE>><NLS_MSG><FACILITY>Prve</FACILITY><ID>10183</ID><MSG_DATA><DATA>$FS_PATH</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
  fi  
else
  result="<RESULT>WARN</RESULT><EXEC_ERROR>Error while getting the file system for the path ($FS_PATH) on node \"$HOST\"<EXEC_ERROR><TRACE>Failed to get the file system for the path ($FS_PATH) on node \"$HOST\"</TRACE><NLS_MSG><FACILITY>Prve</FACILITY><ID>10184</ID><MSG_DATA><DATA>$FS_PATH</DATA><DATA>$DF $FS_PATH</DATA><DATA>$HOST</DATA></MSG_DATA></NLS_MSG>"
fi

echo $result

  
