# 
# $Header: has/install/crsconfig/s_oraocr.pm /unix/5 2016/07/03 16:28:20 luoli Exp $
#
# s_oraocr.pm
# 
# Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      s_oraocr.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    luoli       06/28/16 - Fix bug 23621915
#    bbeelamk    04/12/16 - Remove msg 212
#    xyuan       07/29/15 - Remove 9i configuration file srvConfig.loc
#    jmarcias    07/06/15 - Fix bug 21186706
#    sbezawad    12/23/14 - Bug 20019354: Migrate OCR and OLR to new framework
#    sbezawad    12/23/14 - Creation
# 

package oraClusterwareComp::oraocr;

use parent 'oraClusterwareComp';

use strict;
use English;
use Carp;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;

use crsutils;
use s_crsutils;

my $dev_null          = "/dev/null";

#
# Export functions specific to oraocr class.
#

sub s_ResetOCR
#------------------------------------------------------------------------------
# Function: Reset OCR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $self    = shift;
  my $bin_dd  = "/bin/dd";
  my ($ocr_loc, @mirror_loc, $loc);

  trace ("Reset OCR");

  my $olr_file = s_get_olr_file("olrconfig_loc");

  if (-f $olr_file)
  {
    trace("Removing OLR file: $olr_file");
    s_remove_file ("$olr_file");
  }
  else
  {
    trace ("Removing contents from OLR file: $olr_file");
    system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$olr_file > $dev_null");
  }

  s_remove_file ($CFG->params('OLRCONFIG'));

  if (! $CFG->LASTNODE)
  {
    s_remove_file ($CFG->params('OCRCONFIG'));
    return TRUE;
  }

  if (! -f $CFG->params('OCRCONFIG'))
  {
    # ocr.loc file does not exist. Take ocr location of srvconfig.loc for setting
    # file permissions
    if (-f $CFG->params('SRVCONFIG'))
    {
      $ocr_loc =  get_srvdisk ();
    }
  }
  else
  {
    $ocr_loc = $self->get_ocrdisk();
    push @mirror_loc, $self->get_ocrmirrordisk();
    push @mirror_loc, $self->get_ocrloc3disk();
    push @mirror_loc, $self->get_ocrloc4disk();
    push @mirror_loc, $self->get_ocrloc5disk();
  }

  foreach my $loc (@mirror_loc)
  {
    if (($loc) && ($loc ne $dev_null))
    {
      if (-f $loc)
      {
        # OCR mirror device is specified and enabled
        trace("Removing OCR mirror device: $loc");
        s_remove_file ($loc);
      }
      elsif (! isPathonASM($loc))
      {
        trace ("Removing contents from OCR mirror device: $loc");
        system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$loc > $dev_null");
      }
    }
  }

  # reset OCR device if it's not on ASM
  if (($CFG->LASTNODE) &&
      (! $CFG->DOWNGRADE) &&
      (! $CFG->ASM_STORAGE_USED))
  {
    trace ("Removing contents from OCR device");

    if (-f $ocr_loc)
    {
      trace("Removing OCR device: $ocr_loc");
      s_remove_file ("$ocr_loc");
    }
    elsif (!isPathonASM($ocr_loc))
    {
      trace ("Removing contents from OCR device: $ocr_loc");
      system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$ocr_loc > $dev_null");
    }
  }

  #remove the ocr.loc in the lastnode in case of ASM storage as well
  if ($CFG->LASTNODE)
  {
    s_remove_file ($CFG->params('OCRCONFIG'));
  }

  # reset permissions of ocr_loc files
  if (-f $ocr_loc)
  {
    chmod(0644, $ocr_loc);

    if (isOwnerGroupValid())
    {
      chown ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
             $ocr_loc);
    }
  }

  foreach my $loc (@mirror_loc)
  {
    if ((not -z $loc) && (-f $loc))
    {
      chmod(0644, $loc);

      if (isOwnerGroupValid())
      {
        chown ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
               $loc);
      }
    }
  }
  return TRUE;
}

sub s_backupocrloc
#------------------------------------------------------------------------------
# Function: Back up OCR configuration during upgrade
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $self      = shift;
  my $ocrloc    = $CFG->params('OCRCONFIG');
  my $ocrlocbkp = $ocrloc . ".bkp";

  trace("Back up the ocr.loc file");
  return copy_file($ocrloc, $ocrlocbkp);
}

#
# Private methods
#

sub s_validateOCR
#------------------------------------------------------------------------------
# Function: Validating OCR locations based on existing ocr settings
# Args    : [0] Path for Oracle CRS home
#           [1] Cluster name
#           [2] Comma separated OCR locations
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $crshome        = shift;
  my $clustername    = shift;
  my $ocrlocations   = shift;
  my $OCR_SYNC_FILE  = catfile ($crshome, "srvm", "admin",
                                $CFG->params('OCRLOC'));
  my $OCR_CDATA_DIR  = catfile ($crshome, "cdata");
  my $OCR_BACKUP_DIR = catfile ($crshome, "cdata", $clustername);

  ##Checking the OCR locations used by 10gR1 or previous 10gR2 installations
  my $OCRCONFIG_LOC = "";
  my $OCRMIRRORCONFIG_LOC = "";
  my $OCRCONFIG_LOC3 = "";
  my $OCRCONFIG_LOC4 = "";
  my $OCRCONFIG_LOC5 = "";

  if (-f $CFG->params('OCRCONFIG'))
  {
    trace ("Retrieving OCR location used by previous installations");
    # ocr.loc file exists and ocr location set here is
    $OCRCONFIG_LOC       = s_get_config_key ("ocr", "ocrconfig_loc");
    $OCRMIRRORCONFIG_LOC = s_get_config_key ("ocr", "ocrmirrorconfig_loc");
    $OCRCONFIG_LOC3      = s_get_config_key ("ocr", "ocrconfig_loc3");
    $OCRCONFIG_LOC4      = s_get_config_key ("ocr", "ocrconfig_loc4");
    $OCRCONFIG_LOC5      = s_get_config_key ("ocr", "ocrconfig_loc5");
  }

  my $OCRFILE = $CFG->params('OCRCONFIG');

  trace ("Checking if OCR sync file exists");

  if (-f $OCR_SYNC_FILE)
  {
    trace ("$OCR_SYNC_FILE exists");
    ##Checking the OCR locations used by existing nodes in the cluster
    my $NEW_OCR_FILE = "";
    my $NEW_OCRMIRROR_FILE = "";
    my $NEW_OCRMIRROR_LOC3 = "";
    my $NEW_OCRMIRROR_LOC4 = "";
    my $NEW_OCRMIRROR_LOC5 = "";

    open (OCRSYNCFILE, "<$OCR_SYNC_FILE");

    while (<OCRSYNCFILE>)
    {
      if (/^ocrconfig_loc=(\S+)/) { $NEW_OCR_FILE = $1; }
      if (/^ocrmirrorconfig_loc=(\S+)/) { $NEW_OCRMIRROR_FILE = $1; }
      if (/^ocrconfig_loc3=(\S+)/) { $NEW_OCRMIRROR_LOC3 = $1; }
      if (/^ocrconfig_loc4=(\S+)/) { $NEW_OCRMIRROR_LOC4 = $1; }
      if (/^ocrconfig_loc5=(\S+)/) { $NEW_OCRMIRROR_LOC5 = $1; }
    }
    close (OCRSYNCFILE);

    trace ("NEW_OCR_FILE=$NEW_OCR_FILE");
    trace ("NEW_OCRMIRROR_FILE=$NEW_OCRMIRROR_FILE");
    trace ("NEW_OCRMIRROR_LOC3=$NEW_OCRMIRROR_LOC3");
    trace ("NEW_OCRMIRROR_LOC4=$NEW_OCRMIRROR_LOC4");
    trace ("NEW_OCRMIRROR_LOC5=$NEW_OCRMIRROR_LOC5");

    $ocrlocations = $NEW_OCR_FILE;

    if ($NEW_OCRMIRROR_FILE) {
        $ocrlocations = "$ocrlocations|$NEW_OCRMIRROR_FILE";
    }
    if ($NEW_OCRMIRROR_LOC3) {
        $ocrlocations = "$ocrlocations|$NEW_OCRMIRROR_LOC3";
    }
    if ($NEW_OCRMIRROR_LOC4) {
        $ocrlocations = "$ocrlocations|$NEW_OCRMIRROR_LOC4";
    }
    if ($NEW_OCRMIRROR_LOC5) {
        $ocrlocations = "$ocrlocations|$NEW_OCRMIRROR_LOC5";
    }

    trace ("OCR locations (obtained from $OCR_SYNC_FILE) = $ocrlocations");
  }
  else
  {
    ##Syncing of OCR disks is not required
    trace ("No need to sync OCR file");
  }

  my ($OCR_LOCATION,$OCR_MIRROR_LOCATION,$OCR_MIRROR_LOC3,
      $OCR_MIRROR_LOC4,$OCR_MIRROR_LOC5) = split (/\s*[,|]\s*/, $ocrlocations);

  trace ("OCR_LOCATION=$OCR_LOCATION");
  trace ("OCR_MIRROR_LOCATION=$OCR_MIRROR_LOCATION");
  trace ("OCR_MIRROR_LOC3=$OCR_MIRROR_LOC3");
  trace ("OCR_MIRROR_LOC4=$OCR_MIRROR_LOC4");
  trace ("OCR_MIRROR_LOC5=$OCR_MIRROR_LOC5");
  trace ("Current OCR location= $OCRCONFIG_LOC");
  trace ("Current OCR mirror location= $OCRMIRRORCONFIG_LOC");
  trace ("Current OCR mirror loc3=$OCRCONFIG_LOC3");
  trace ("Current OCR mirror loc4=$OCRCONFIG_LOC4");
  trace ("Current OCR mirror loc5=$OCRCONFIG_LOC5");
  trace ("Verifying current OCR settings with user entered values");

  if ($OCRCONFIG_LOC)
  {
    if ($OCR_LOCATION ne $OCRCONFIG_LOC)
    {
      my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
      print_error(210, $OCRCONFIG_LOC, $OCRFILE, $OCR_LOCATION);
      trace ("Update either \"$OCRFILE\" to use \"$OCR_LOCATION\" or " .
             "variable OCR_LOCATIONS property set in " .
             catfile ($crshome, "crs", "install", "crsconfig_params") .
             "with \"$OCRCONFIG_LOC\" then rerun this script");
      return FALSE;
    }
  }
  else
  {
    #set ocrconfig_loc = OCR_LOCATION
    $OCRCONFIG_LOC = $OCR_LOCATION;
  }

  if ($OCRMIRRORCONFIG_LOC)
  {
    if ($OCR_MIRROR_LOCATION ne $OCRMIRRORCONFIG_LOC)
    {
      my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
      print_error(211, $OCRMIRRORCONFIG_LOC, $OCRFILE, $OCR_MIRROR_LOCATION);
      trace ("Update either \"$OCRFILE\" to use " .
             "\"$OCR_MIRROR_LOCATION\" or variable OCR_LOCATIONS " .
             "property set in " .
             catfile ($crshome, "crs" . "install" . "crsconfig_params") .
             " with \"$OCRMIRRORCONFIG_LOC\" then rerun this script");
      return FALSE;
    }
  }
  else
  {
    #set the mirror location=user entered value for OCR_MIRROR_LOCATION
    $OCRMIRRORCONFIG_LOC = $OCR_MIRROR_LOCATION;
  }

  if ($OCRCONFIG_LOC3)
  {
    if ($OCR_MIRROR_LOC3 ne $OCRCONFIG_LOC3)
    {
      my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
      print_error(211, $OCRCONFIG_LOC3, $OCRFILE, $OCR_MIRROR_LOC3);
      trace ("Update either \"$OCRFILE\" to use " .
             "\"$OCR_MIRROR_LOC3\" or variable OCR_LOCATIONS " .
             "property set in " .
             catfile ($crshome, "crs" . "install" . "crsconfig_params") .
             " with \"$OCRCONFIG_LOC3\" then rerun this script");
       return FALSE;
    }
  }
  else
  {
    #set the mirror location=user entered value for OCR_MIRROR_LOCATION
    $OCRCONFIG_LOC3 = $OCR_MIRROR_LOC3;
  }

  if ($OCRCONFIG_LOC4)
  {
    if ($OCR_MIRROR_LOC4 ne $OCRCONFIG_LOC4)
    {
      my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
      print_error(211, $OCRCONFIG_LOC4, $OCRFILE, $OCR_MIRROR_LOC4);
      trace ("Update either \"$OCRFILE\" to use " .
             "\"$OCR_MIRROR_LOC4\" or variable OCR_LOCATIONS " .
             "property set in " .
             catfile ($crshome, "crs" . "install" . "crsconfig_params") .
             " with \"$OCRCONFIG_LOC4\" then rerun this script");
      return FALSE;
    }
  }
  else
  {
    #set the mirror location=user entered value for OCR_MIRROR_LOCATION
    $OCRCONFIG_LOC4 = $OCR_MIRROR_LOC4;
  }

  if ($OCRCONFIG_LOC5)
  {
    if ($OCR_MIRROR_LOC5 ne $OCRCONFIG_LOC5)
    {
      my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
      print_error(211, $OCRCONFIG_LOC5, $OCRFILE, $OCR_MIRROR_LOC5);
      trace ("Update either \"$OCRFILE\" to use " .
             "\"$OCR_MIRROR_LOC5\" or variable OCR_LOCATIONS " .
             "property set in " .
             catfile ($crshome, "crs" . "install" . "crsconfig_params") .
             " with \"$OCRCONFIG_LOC5\" then rerun this script");
      return FALSE;
    }
  }
  else
  {
    #set the mirror location=user entered value for OCR_MIRROR_LOCATION
    $OCRCONFIG_LOC5 = $OCR_MIRROR_LOC5;
  }

  trace ("Setting OCR locations in " . $CFG->params('OCRCONFIG'));
  s_validate_ocrconfig ($ocrlocations, 0) or (return FAILED);

  if (-f $OCR_SYNC_FILE)
  {
    trace ("Removing OCR sync file: $OCR_SYNC_FILE");
    s_remove_file ("$OCR_SYNC_FILE");
  }

  return TRUE;
}

sub s_validate_ocrconfig
#------------------------------------------------------------------------------
# Function: Creates OCR config file if it does not exists.
# Args    : [0] List of OCR locations
#           [1] TRUE  - SIHA mode
#               FALSE - Cluster mode
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $ocrlocations = shift;
  my $isHas        = shift;
  my $OCRCONFIG    = $CFG->params('OCRCONFIG');

  trace ("Validating OCR locations in " . $CFG->params('OCRCONFIG'));
  trace ("Checking for existence of " . $CFG->params('OCRCONFIG'));

  if (-f $OCRCONFIG)
  {
    trace ("Backing up " . $OCRCONFIG . " to " . $OCRCONFIG . ".orig");
    copy ($OCRCONFIG, $OCRCONFIG . ".orig") || return FAILED;
  }

  my ($ocrlocation,
      $ocrmirrorlocation,
      $ocrlocation3,
      $ocrlocation4,
      $ocrlocation5) = split (/\s*[,|]\s*/, $ocrlocations);

  open (OCRCFGFILE, ">$OCRCONFIG") || return FAILED;

  trace ("Setting ocr location " . $ocrlocation);
  print OCRCFGFILE "ocrconfig_loc=$ocrlocation\n";

  if ($ocrmirrorlocation)
  {
    trace ("Setting ocr mirror location " . $ocrmirrorlocation);
    print OCRCFGFILE "ocrmirrorconfig_loc=$ocrmirrorlocation\n";
  }

  if ($ocrlocation3)
  {
    trace ("Setting ocr location3 " . $ocrlocation3);
    print OCRCFGFILE "ocrconfig_loc3=$ocrlocation3\n";
  }

  if ($ocrlocation4)
  {
    trace ("Setting ocr location4 " . $ocrlocation4);
    print OCRCFGFILE "ocrconfig_loc4=$ocrlocation4\n";
  }

  if ($ocrlocation5)
  {
    trace ("Setting ocr location5 " . $ocrlocation5);
    print OCRCFGFILE "ocrconfig_loc5=$ocrlocation5\n";
  }

  if ($isHas)
  {
    print OCRCFGFILE "local_only=TRUE\n";
  }
  else
  {
    print OCRCFGFILE "local_only=FALSE\n";
  }

  close (OCRCFGFILE);
  return TRUE;
}

sub s_createLocalOnlyOCR
#------------------------------------------------------------------------------
# Function: Create local-only OCR
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $owner      = $CFG->params('ORACLE_OWNER');
  my $dba_group  = $CFG->params('ORA_DBA_GROUP');
  my $ocr_config = $CFG->params('OCRCONFIG');

  trace ("create Local Only OCR on Linux...");

  # create ocr.loc w/ local_only=TRUE and set ownergroup
  open (FILEHDL, ">$ocr_config") or die(dieformat(255, $ocr_config, $!));
  print FILEHDL "local_only=TRUE\n";
  close (FILEHDL);
  s_set_ownergroup ($owner, $dba_group, $ocr_config)
                                    || die(dieformat(152, $ocr_config));
  s_set_perms ("0640", $ocr_config) || die(dieformat(153, $ocr_config));

  return TRUE;
}

sub s_copyOCRLoc
#------------------------------------------------------------------------------
# Function: Copy the local OCR configuration to all new nodes that are being
#           added
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
# NOTES: OCR handles the ocr.loc updates on its own, and this is kept around
#        as a fallback
#-------------------------------------------------------------------------------
{
  my $cluutil     = catfile ($CFG->ORA_CRS_HOME, 'bin', 'cluutil');
  my $ocrloc_temp = catfile ($CFG->ORA_CRS_HOME, 'srvm', 'admin', 'ocrloc.tmp');
  my $ocrloc_file = catfile ($CFG->ORA_CRS_HOME, 'srvm', 'admin', 
                              $CFG->params('OCRLOC'));
  my @node_list   = getCurrentNodenameList();
  my $success     = FALSE;
  my @capout      = ();
  my @cmd;
  my $rc;

  if (! (-e $cluutil))
  {
    trace("$cluutil not found");
    trace("Unable to copy OCR locations");
    return FALSE;
  }

  my $host = $CFG->HOST;
  foreach my $node (@node_list)
  {
    if ($node !~ /\b$host\b/i)
    {
      @cmd = ("$cluutil", '-copy', '-sourcefile', $CFG->params('OCRCONFIG'),
              '-sourcenode', $node, '-destfile', $ocrloc_temp,
              '-nodelist', $node);
      $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout, @cmd);

      if ($rc == 0)
      {
        trace("@cmd ... passed");
        $success = TRUE;
        last;
      }
      else
      {
        trace("@cmd ... failed");
        if (scalar(@capout) > 0) { trace("capout=@capout"); }
      }
    }
    else
    {
      trace("Avoiding self copy of ocr.loc on node: $node");
    }
  }

  if ($success)
  {
    rename ($ocrloc_temp, $ocrloc_file);
  }
  else
  {
    trace("@cmd ... failed");
    s_remove_file ("$ocrloc_temp");
    return FALSE;
  }

  return TRUE;
}

sub s_restoreocrloc
#------------------------------------------------------------------------------
# Function: During downgrade, restore OCR configuration from a backup
# Args    : None
# Returns : TRUE  if success
#           FALSE if failed
#------------------------------------------------------------------------------
{
  my $ocrloc    = $CFG->params('OCRCONFIG');
  my $ocrlocbkp = $ocrloc . ".bkp";

  trace("Restore old ocr.loc file");
  unless (-e $ocrlocbkp)
  {
    trace("Failed to restore the file refers to the OCR location because the ".
          "backup file $ocrlocbkp does not exist.");
    return FAILED;
  }
  return copy_file($ocrlocbkp, $ocrloc);  
}

1;

