# 
# $Header: has/install/crsconfig/oraqosmserver.pm /main/5 2016/07/14 01:10:44 muhe Exp $
#
# oraqosmserver.pm
# 
# Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      oraqosmserver.pm - Library module for qosmserver root install/upgrade
#                         functions.
#
#    DESCRIPTION
#      Contains installation and upgrade routines for qosmserver
#
#    NOTES
#      FIXME - Refactor all $CFG references to test this module in isolation.
#
#    MODIFIED   (MM/DD/YY)
#    muhe        07/06/16 - Fix bug 23747273
#    bbeelamk    03/08/16 - Fix srvctl error
#    jgrout      02/16/16 - More crsutils::srvctl_capture instead of
#                           run_as_user2
#    jgrout      02/05/16 - Fix bug 21419874
#    gmaldona    10/06/15 - Use crsutils::srvctl_capture instead of
#                           run_as_user2 
#    espgarci    05/06/15 - Creation
# 

package oraqosmserver;

use English;
use Exporter;
use strict;

use crsutils;

use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(add_qosmserver start_qosmserver upgrade_credstore
                   stop_qosmserver disable_qosmserver
                   stop_and_disable_J2EEContainer);

push @EXPORT, @exp_func; 

my $run_as_owner = TRUE;

sub add_qosmserver
#---------------------------------------------------------------------
# Function: Add the resource qosmserver.
# Returns : SUCCESS if the resource was added, FAILED otherwise
#           (HP-UX and z/Linux) SUCCESS (resource not added)
# Args    : none 
#---------------------------------------------------------------------
{
   my $rc = FAILED;
   if (check_qosmserver_supported()) {
      my @out    = ();
      my @opts   = ('add', 'qosmserver');
      my $rf     = srvctl_capture($run_as_owner, \@out, join(' ', @opts));
   
      #We have success with these conditions:
      # - ($rf == 0) : The qosmserver resource was added with success.
      # - PRCR-1086 : resource ora.qosmserver is already registered
      #               now proceed to create the credential store.
      if (($rf == 0) || (scalar(grep(m/PRCR-1086/i, @out)) > 0)) {
         if ($rf == 0) {
            trace("oraqosmserver: resource added.");
         } else {
            trace("oraqosmserver: resource is already registered.");
            # Stop the resource if it is running
            if (stop_qosmserver() == FAILED) {
              return $rc;
            }
         }
         $rc = install_credstore(); 
      } else {
         my $trace_msg = "oraqosmserver: failed to add resource.";
         srvctl_error(1100, $trace_msg, \@out, @opts);
      }
   } else {
      trace("oraqosmserver: unsupported platform, resource not added.");
      $rc = SUCCESS;
   }
   return $rc;
}

sub install_credstore
#---------------------------------------------------------------------
# Function: Add the credential store and users qosadmin and jwcadmin
#           for qosmserver resource.
# Args    : none
# Returns : SUCCESS if the credential store was created and the 
#           qosadmin and jwcadmin users were added, FAILED otherwise
#---------------------------------------------------------------------
{  
   my $rc     = FAILED;
   my @opts = ('-install');
   my @out    = ();
   my $rf     = qosctl_cmd(\@out, @opts);

   if ($rf == 0) {
      trace("oraqosmserver: qosctl output is " . join("\n", @out));
      $rc = SUCCESS;      
   } else {
      qosctl_error(1105, \@out, @opts);
   }
   return $rc;
}

sub upgrade_credstore
#---------------------------------------------------------------------
# Function: Upgrade the credential store for qosmserver resource
# Args    : none
# Returns : SUCCESS if the credential store is upgraded
#           (HP-UX and z/Linux) SUCCESS if the credential store is removed
#           FAILED otherwise
#---------------------------------------------------------------------
{
   my $rc        = FAILED;
   my @activever = get_crs_version($CFG->oldconfig('ORA_CRS_HOME'));
   my $version   = join('.', @activever);
   my @opts    = ('-upgrade', $version);
   my @out       = ();
   my $rf        = qosctl_cmd(\@out, @opts);

   if ($rf == 0) {
      trace("oraqosmserver: qosctl output is " . join("\n", @out));
      $rc = SUCCESS;
   } else {
      qosctl_error(1106, \@out, @opts);
   }
   return $rc;
}

sub start_qosmserver
#---------------------------------------------------------------------
# Function: Start the qosmserver resource
# Args    : none
# Returns : SUCCESS if the resource was started, FAILED otherwise
#           (HP-UX and z/Linux) SUCCESS (resource not started)
#---------------------------------------------------------------------
{
   my $rc = SUCCESS;
   if (check_qosmserver_supported()) {
      if (check_qosmserver_enable()) {
         # Enable qosmserver resource
         $rc = enable_qosmserver();
         if ($rc == SUCCESS) {
            # Start qosmserver resource
            my @out    = ();
            my @opts   = ('start', 'qosmserver');
            my $rf     = srvctl_capture($run_as_owner,
                                        \@out, join(' ', @opts));
      
            #We have success with these conditions:
            # - ($rf == 0) : qosmserver started 
            # - PRCC-1014 : qosmserver is already running
            if (($rf == 0) || (scalar(grep(m/PRCC-1014/i, @out)) > 0)) {
               $rc = SUCCESS;
               trace ("oraqosmserver: resource started.");
            } else {
               $rc = FAILED;
               my $trace_msg = "oraqosmserver: failed to start resource.";
               srvctl_error(1102, $trace_msg, \@out, @opts);
            }
         }
      } else {
         # Disable qosmserver resource
         $rc = disable_qosmserver();
      }
   }
   return $rc;
}

sub enable_qosmserver
#---------------------------------------------------------------------
# Function: Enable qosmserver resource
# Args    : none
# Returns : SUCCESS if the resource was enabled, FAILED otherwise
#---------------------------------------------------------------------
{
   my $rc = FAILED;
   
   # Enable qosmserver resource
   my @out    = ();
   my @opts   = ('enable', 'qosmserver');
   my $rf     = srvctl_capture($run_as_owner, \@out, join(' ', @opts));
   
   # We have success with these conditions:
   # - ($rf == 0) : qosmserver was successfully enabled
   # - PRKF-1321 : qosmserver is already enabled
   if (($rf == 0) || (scalar(grep(m/PRKF-1321/i, @out)) > 0)) {
      $rc = SUCCESS;
      trace("oraqosmserver: resource enabled.");
   } else {
      my $trace_msg = "oraqosmserver: failed to enable resource.";
      srvctl_error(1101, $trace_msg, \@out, @opts);
   }
   return $rc;
}

sub disable_qosmserver
#---------------------------------------------------------------------
# Function: Disable qosmserver resource
# Args    : none
# Returns : SUCCESS if the resource was disabled, FAILED otherwise
#---------------------------------------------------------------------
{
   my $rc = FAILED;

   #  Disable resource qosmserver.
   my @out    = ();
   my @opts   = ('disable', 'qosmserver');
   my $rf     = srvctl_capture($run_as_owner, \@out, join(' ', @opts));
   
   # We have success with these conditions:
   # - ($rf == 0) : This means qosmserver was successfully disabled.
   # - PRKF-1320 : qosmserver is already disabled
   if (($rf == 0) || (scalar(grep(m/PRKF-1320/i, @out)) > 0)) {
      $rc = SUCCESS;
      trace("oraqosmserver: resource disabled.");
   } else {
      my $trace_msg = "oraqosmserver: failed to disable resource.";
      srvctl_error(1103, $trace_msg, \@out, @opts);
   }
   return $rc;
}

sub stop_qosmserver
#---------------------------------------------------------------------
# Function: Stop the qosmserver resource
# Args    : none
# Returns : SUCCESS if the resource was stopped, FAILED otherwise
#---------------------------------------------------------------------
{
   my $rc = FAILED;
   
   # Stop qosmserver resource
   my @out    = ();
   my @opts   = ('stop', 'qosmserver');
   my $rf     = srvctl_capture($run_as_owner, \@out, join(' ', @opts));
      
   #We have success with these conditions:
   # - ($rf == 0) : qosmserver stopped
   # - PRCC-1016 : qosmserver is already stopped
   if (($rf == 0) || (scalar(grep(m/PRCC-1016/i, @out)) > 0)) {
      $rc = SUCCESS;
      trace ("oraqosmserver: resource stopped.");
   } else {
      $rc = FAILED;
      my $trace_msg = "oraqosmserver: failed to stop resource.";
      srvctl_error(1104, $trace_msg, \@out, @opts);
   } 
   return $rc;
}

sub qosctl_cmd
#--------------------------------------------------------------------- 
# Function: Execute qosctl commands
# Args    : [0] Output reference
#           [1-n] Options to the command to execute
# Returns : The result of executing the command
#--------------------------------------------------------------------- 
{
   my $out     = shift;
   my @opts    = @_;
   my $user    = $CFG->params('ORACLE_OWNER'); 
   my @cmd     = (get_qosctl_path(), @opts);
   return run_as_user2($user, $out, @cmd); 
}

sub srvctl_error
#--------------------------------------------------------------------- 
# Function: Log traces and print error messages for srvctl commands
# Args    : [0] Error number
#           [1] Message trace
#           [2] Command output
#           [3-n] Command options
# Returns : None
#--------------------------------------------------------------------- 
{
   my $error_num = shift;
   my $trace_msg = shift;
   my $out       = shift;
   my @opts      = @_;
   my $srvctl    = crs_exec_path('srvctl');
   my @cmd       = ($srvctl, @opts);
   trace(join(">  ", ("Error encountered in the command @cmd \n", 
      @$out, "End Command output")));
   print_lines(@$out);
   trace($trace_msg);
   print_error($error_num); 
}

sub qosctl_error
#--------------------------------------------------------------------- 
# Function: Log traces and print error messages for qosctl commands
# Args    : [0] Error number
#           [1] Command output
#           [2-n] Command options
# Returns : None
#--------------------------------------------------------------------- 
{
   my $error_num = shift;
   my $out       = shift;
   my @opts      = @_;
   my @cmd       = (get_qosctl_path(), @opts);
   trace(join(">  ", ("Error encountered in the command @cmd \n", 
      @$out, "End Command output")));
   print_error($error_num); 
}

sub check_qosmserver_supported
#---------------------------------------------------------------------
# Function: Determine whether or not the qosmserver
#           resource is supported on this platform
# Args    : none 
# Returns : TRUE if the resource is supported, FALSE otherwise
#---------------------------------------------------------------------
{
   # If Unix, call Unix OSD
   if ($CFG->platform_family eq "unix") {
      return is_platform_supported();
   }

   # If Windows, platform is supported
   return TRUE;
}

sub is_platform_supported
#---------------------------------------------------------------------
# Function: Determine whether or not the qosmserver
#           resource is supported on this platform
# Args    : none 
# Returns : TRUE if the resource is supported, FALSE otherwise
#---------------------------------------------------------------------
{
  # qosmserver not supported on HP-UX (bug 12648797)
  if ($OSNAME eq 'hpux') {
    return FALSE;
  }
  
  # qosmserver not supported on z/Linux (bug 14615073)
  my $mach =`uname -m`;
  chomp($mach);

  if ($mach eq "s390x" || $mach eq "s390") {
    return FALSE;
  }

  return TRUE;
}

sub check_qosmserver_enable
#---------------------------------------------------------------------
# Function: Assuming that the qosmserver resource has been created,
#           determine whether or not it should be enabled
# Args    : none 
# Returns : TRUE if the resource should be enabled, FALSE otherwise
#---------------------------------------------------------------------
{
   # Disable qosmserver on shiphome without management DB (bug 14741962)
   if ((is_dev_env() == FALSE) &&
       !(($CFG->defined_param('MGMT_DB')) &&
         ($CFG->params('MGMT_DB') =~ m/true/i)))
   {
     return FALSE;
   }

   return TRUE;
}

sub stop_and_disable_J2EEContainer
#---------------------------------------------------------------------
# Function: Stop and Disable OC4J Container (before upgrade)
# Args    : [0] -> CRS home from where execute the command
#---------------------------------------------------------------------
{
   my $crs_home = $_[0];
   # Stop oc4j resource using specified CRS home
   my @cmd = ('stop', 'oc4j');
   my @out  = ();
   my $status =
       srvctl_capture($run_as_owner, \@out,
                      join(' ', @cmd), $crs_home);
      
   #We have success with this condition:
   # - PRCR-1001 : Resource ora.oc4j does not exist
   if (($status != 0) &&
       (scalar(grep(m/PRCR-1001/i, @out)) > 0)) {
       trace ("J2EE (OC4J) Container Resource Does Not Exist ",
              "... passed ...");
   } else {
       # We have success with these conditions:
       # - ($status == 0) : OC4J was stopped
       # - PRCC-1016 : OC4J was stopped already
       if (($status != 0) &&
           (scalar(grep(m/PRCC-1016/i, @out)) == 0)) {
           trace(join(">  ", ("Error encountered in the command ",
                              "@cmd \n", @out, "End Command output")));
           trace("J2EE (OC4J) Container Resource Stop ... failed ...");
           die(dieformat(1009));
       } 
       trace ("J2EE (OC4J) Container Resource Stop ... passed ...");

       #  Disable OC4J resource using specified CRS home
       @cmd  = ('disable', 'oc4j');
       @out  = ();
       my $status =
           srvctl_capture($run_as_owner, \@out,
                          join(' ', @cmd), $crs_home);
   
       # We have success with these conditions:
       # - ($status == 0) : OC4J was disabled
       # - PRKO-2115 : OC4J was disabled already
       if (($status != 0) && (scalar(grep(m/PRKO-2115/i, @out)) == 0)) {
           trace(join(">  ", ("Error encountered in the command ",
                              "@cmd \n", @out, "End Command output")));
           trace("J2EE (OC4J) Container Resource Disable " .
                 "... failed ...");
           die(dieformat(1005));
       }
       trace("J2EE (OC4J) Container Resource Disable ... passed");
   }
}
