# Copyright (c) 2011, 2016, Oracle and/or its affiliates. All rights reserved.
#
#   NAME
#      crstfa.pm
#
#   DESCRIPTION
#      This module contains functions related to TFA [Trace File Analyzer]
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#      arupadhy 06/23/16 - Added .bat extension for 23637585
#      arupadhy 01/13/16 - Adding windows support for TFA
#      bburton  04/04/14 - Fix bug 18410410 - setup TFA from old home after
#                          downgrade
#      cnagur   12/10/13 - Removed all use constant statements
#      cnagur   10/28/13 - Added print_info
#      cnagur   10/17/13 - TFA Functions for 12c
#

package crstfa;

use strict;
use English;
use Exporter;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use File::Temp qw/ tempfile/;
use Sys::Hostname;
use Carp;
use Cwd;
use Env qw(NLS_LANG);
use Env qw(SRVM_TRACE);
use Term::ANSIColor;
use Scalar::Util 'blessed';

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
our $CFG;

BEGIN {
   @ISA = qw(Exporter);

   my @exp_const = qw(
                     );

   my @exp_func  = qw(setup_tfa remove_tfa startup_tfa shutdown_tfa patch_tfa
                     );

   @EXPORT  = qw($CFG);
   push @EXPORT, @exp_const, @exp_func;
}

# root script modules
use crsutils;


# TFA Functions:

#-------------------------------------------------------------------------------
# Function: Calls the TFA setup script
# Args    : Patch Argument
# Returns : none
#-------------------------------------------------------------------------------

sub setup_tfa
{
  my $PATCH = shift;
  my $TFASCRIPT;
  my $TFA_CRS_HOME;
  
  # if we are passed downgrade then we need to run the setup script from the old home
  # the old script could be tfa_setup.sh for 11.2

  if ( $PATCH eq "downgrade" ) {
      $TFA_CRS_HOME = $CFG->OLD_CRS_HOME;
      if($CFG->platform_family eq 'windows'){
        $TFASCRIPT = catfile($TFA_CRS_HOME, "crs", "install", "tfa_setup.bat"); 
      }else{
        $TFASCRIPT = catfile($TFA_CRS_HOME, "crs", "install", "tfa_setup.sh");
      }
      if ( ! -f "$TFASCRIPT" ) {
          $TFASCRIPT = catfile($TFA_CRS_HOME, "crs", "install", "tfa_setup");
      }
  } else { 
      $TFA_CRS_HOME = $CFG->ORA_CRS_HOME;
      if($CFG->platform_family eq 'windows'){
        $TFASCRIPT = catfile($CFG->ORA_CRS_HOME, "crs", "install", "tfa_setup.bat"); 
      }else{
        $TFASCRIPT = catfile($CFG->ORA_CRS_HOME, "crs", "install", "tfa_setup");
      }
  }

  if ( -f "$TFASCRIPT" ) {
      my $cmd = "$TFASCRIPT -silent -crshome ".$TFA_CRS_HOME;
      if ( $PATCH ne "patch" ) {
          print_info(4001);
      }
      my @output = system_cmd_capture( "$cmd" );
      my $status = shift @output;

      if ( $status == 0 ) {
         if ( $PATCH ne "patch" ) {
            print_info(4002);
         } else {
            print_info(4003);
         }
      } else {

         # TFA does not yet have NLS messages but we have requested the TFA- prefix.
         # As system_command_capture treats stderr and stdout the same we have to search for messages we want to show to the console.
         for (my $idx = 0; $idx < scalar(@output); $idx++)
         {
            if ( $output[$idx] =~ /TFA-/ ) {
               print "$output[$idx]\n";
            }
         }

         if ( $PATCH ne "patch" ) {
            print_error(4004);
         } else {
            print_error(4005);
         }
         trace ("Command: $cmd, Return Status: $status\n".join("\n", @output)."\n");
      }
  }
  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function: Remove TFA
# Args    : none
# Returns : none
#-------------------------------------------------------------------------------

sub remove_tfa
{

  my $HOST = tolower_host();

  # Check if the TFA is running from GIHOME
  my $TFAHOME = catfile($CFG->ORA_CRS_HOME, "tfa", "$HOST", "tfa_home");

  if ( -d "$TFAHOME" ) {
      my $SCRIPT;
      
      if($CFG->platform_family eq 'windows'){
        $SCRIPT = catfile($TFAHOME, "bin", "tfactl.bat");
      } else {
        $SCRIPT = catfile($TFAHOME, "bin", "uninstalltfa");
      }

      if ( -f "$SCRIPT" ) {
          my $cmd;
          if($CFG->platform_family eq 'windows'){
            $cmd = "$SCRIPT uninstall";
          }else{
            $cmd = "$SCRIPT -silent -local -crshome ".$CFG->ORA_CRS_HOME;
          }
          
          print_info(4006);
          my @output = system_cmd_capture( "$cmd" );
          my $status = shift @output;

          if ( $status == 0 ) {
              print_info(4007);
          } else {

              for (my $idx = 0; $idx < scalar(@output); $idx++)
              {
                 if ( $output[$idx] =~ /TFA-/ ) {
                    print "$output[$idx]\n";
                 }
              }

              print_error(4008, "$TFAHOME");
              trace ("Command: $cmd, Return Status: $status\n".join("\n", @output)."\n");
          }
      }
  }
  return SUCCESS;
}

#------------------------------------------------------------------------------
# Function: Startup TFA
# Args    : none
# Returns : none
#------------------------------------------------------------------------------

sub startup_tfa
{
  my $HOST = tolower_host();

  my $TFAHOME = catfile($CFG->ORA_CRS_HOME, "tfa", "$HOST", "tfa_home");

  if ( -d "$TFAHOME" ) {
      my $TFACTL;

      if($CFG->platform_family eq 'windows'){
        $TFACTL = catfile($TFAHOME, "bin", "tfactl.bat");
      } else {
        $TFACTL = catfile($TFAHOME, "bin", "tfactl");
      }

      if ( -f "$TFACTL" ) {
          my $cmd = "$TFACTL start";
          print_info(4009);
          my @output = system_cmd_capture( "$cmd" );
          my $status = shift @output;

          if ( $status == 0 ) {
              print_info(4010);
          } else {

             for (my $idx = 0; $idx < scalar(@output); $idx++)
              {
                 if ( $output[$idx] =~ /TFA-/ ) {
                    print "$output[$idx]\n";
                 }
              }

              print_error(4011);
              trace ("Command: $cmd, Return Status: $status\n".join("\n", @output)."\n");
          }
      }
  }
  return SUCCESS;
}


#------------------------------------------------------------------------------
# Function: Shutdown TFA
# Args    : Patch Argument
# Returns : none
#------------------------------------------------------------------------------

sub shutdown_tfa
{
  my $PATCH = shift;

  my $HOST = tolower_host();

  my $TFAHOME = catfile($CFG->ORA_CRS_HOME, "tfa", "$HOST", "tfa_home");

  if ( -d "$TFAHOME" ) {
        my $TFACTL;
        
        if($CFG->platform_family eq 'windows'){
          $TFACTL = catfile($TFAHOME, "bin", "tfactl.bat");
        } else {
          $TFACTL = catfile($TFAHOME, "bin", "tfactl");
        }

        if ( -f "$TFACTL" ) {
            my $cmd = "$TFACTL shutdown";
            print_info(4012);
            my @output = system_cmd_capture( "$cmd" );
            my $status = shift @output;

            if ( $status == 0 ) {
                print_info(4013);
            } else {

                for (my $idx = 0; $idx < scalar(@output); $idx++)
                {
                   if ( $output[$idx] =~ /TFA-/ ) {
                      print "$output[$idx]\n";
                   }
                }

                print_error(4014);
                trace ("Command: $cmd, Return Status: $status\n".join("\n", @output)."\n");
            }
        }
  }
  return SUCCESS;
}


#-------------------------------------------------------------------------------
# Function: Patch TFA
# Args    : Source and destination GI home ( same for In place )
# Returns : none
# ------------------------------------------------------------------------------

sub patch_tfa
{
  my $CONFGIHOME = shift;
  my $PATCHGIHOME = shift;

  print_info(4015);

  if ( $CONFGIHOME eq $PATCHGIHOME ) {
      setup_tfa("patch");
  } else {
      shutdown_tfa("patch");
      setup_tfa("patch");
  }
  return SUCCESS;
}


1;
