/*
 * Copyright (c) 2016, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.quickstart;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Base64;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;

public class DirectActivation {
	String signatureAlgorithm = "SHA256withRSA";
	String secretHashAlgorithm = "HmacSHA256";
	String keyAlgorithm = "RSA";
	String keyFormat = "X.509";
	String directActivationDeviceModel = "urn:oracle:iot:dcd:capability:direct_activation";
	String uniqueURN;
	String endpointId;
	String activationSecret;
	PublicKey publicKey;
	PrivateKey privateKey;
	
	/**
	 * Prepare the parameters for direct activation
	 * @param endpointId device end point id
	 * @param activationSecret shared secret for activating device
	 * @param urn device model URN attached to the device while activating
	 * @param privateKeyFileName RSA private key 
	 * @param publicKeyFileName RSA public key
	 */
	public DirectActivation(String endpointId, String activationSecret, String urn, String privateKeyFileName, String publicKeyFileName) {
		this.endpointId = endpointId;
		this.activationSecret = activationSecret;
		this.uniqueURN = urn;
		try {
			privateKey = RSAKeyHelper.getPrivateKeyFromFile(privateKeyFileName);
			publicKey = RSAKeyHelper.getPublicKeyFromFile(publicKeyFileName);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Generate activation payload and write to .json file
	 */
	public void printDirectActivationPayload() {
		byte[] publicKeyByte = publicKey.getEncoded();
		byte[] signature = getSignature();
		JsonArray deviceModels = Json.createArrayBuilder().add(directActivationDeviceModel)
				.add(uniqueURN).build();
		JsonObject subjectPublicKeyInfo = Json.createObjectBuilder().add("algorithm", keyAlgorithm)
				.add("publicKey", Base64Encode(publicKeyByte)).add("format", keyFormat)
				.add("secretHashAlgorithm", secretHashAlgorithm).build();
		JsonObject certificationRequestInfo = Json.createObjectBuilder().add("subject", endpointId)
				.add("subjectPublicKeyInfo", subjectPublicKeyInfo).add("attributes", Json.createObjectBuilder().build())
				.build();
		JsonObject activationPayload = Json.createObjectBuilder().add("deviceModels", deviceModels)
				.add("certificationRequestInfo", certificationRequestInfo).add("signatureAlgorithm", signatureAlgorithm)
				.add("signature", Base64Encode(signature)).build();
		System.out.println("Direct Activation Payload:\n" + activationPayload.toString());
		writePayloadToFile(activationPayload.toString(), "../json/activation_payload.json");
	}
	
	private byte[] getSecretHash() {
		byte[] activationSecretByte = toUTF8(activationSecret);
		SecretKeySpec key = new SecretKeySpec(activationSecretByte, secretHashAlgorithm);
		Mac mac = null;
		try {
			mac = Mac.getInstance(secretHashAlgorithm);
			mac.init(key);
			mac.update(toUTF8(endpointId));
		} catch (NoSuchAlgorithmException | InvalidKeyException e) {
			e.printStackTrace();
		}
		return mac.doFinal();
	}

	public static byte[] toUTF8(String input) {
		return input.getBytes(StandardCharsets.UTF_8);
	}

	public static String Base64Encode(byte[] input) {
		return Base64.getEncoder().encodeToString(input);
	}

	private byte[] getSignaturePayload() {
		byte[] secretHash = getSecretHash();
		byte[] publicKeyByte = publicKey.getEncoded();
		String payload = endpointId + "\n" + keyAlgorithm + "\n" + keyFormat + "\n" + secretHashAlgorithm + "\n";
		byte[] payloadBytes = toUTF8(payload);
		byte[] signatureBytes = new byte[payloadBytes.length + secretHash.length + publicKeyByte.length];
		System.arraycopy(payloadBytes, 0, signatureBytes, 0, payloadBytes.length);
		System.arraycopy(secretHash, 0, signatureBytes, payloadBytes.length, secretHash.length);
		System.arraycopy(publicKeyByte, 0, signatureBytes, secretHash.length + payloadBytes.length,
				publicKeyByte.length);
		return signatureBytes;
	}

	private byte[] getSignature() {
		byte[] signaturePayload = getSignaturePayload();
		try {
			Signature signature = Signature.getInstance(signatureAlgorithm);
			signature.initSign(privateKey);
			signature.update(signaturePayload);
			return signature.sign();
		} catch (NoSuchAlgorithmException | InvalidKeyException | SignatureException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	private static void writePayloadToFile(String payload, String fileName) {
		try (PrintWriter out = new PrintWriter(fileName)) {
			out.println(payload);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		
	}
}
