/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.app.temperaturesensor;

import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEndpoint;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooColdEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooHotEvent;
import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;
;
import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;

import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * This temperature sensor application is an example of an application which is an endpoint itself and as such,
 * generates and sends temperature sensor readings to the IoT CS.
 *
 * Note: Some other sample applications receive events from their respective EndpointImpl's, then convert the event to a
 * message and send it to the IoT CS.  In these cases, the EndpointImpl is the endpoint and the application is sending
 * the message.  This sample *does not* do that.
 */
@IoTDeviceApplication
public class TemperatureSensorApplication extends AbstractApplication implements TemperatureSensorEndpoint {

    private static final AtomicBoolean done = new AtomicBoolean(false);
    private final static long OUTPUT_INTERVAL = 2000;
    private static Random random = new Random();
    private boolean isOn;
    private Date startTimeValue;
    private Float   maxTempValue = 0f;
    // The maximum threshold value, initialized to the default in the device model.
    private Integer maxThresholdValue = 70;
    private Float   minTempValue = 0f;
    // The minimum threshold value, initialized to the default in the device model.
    private Integer minThresholdValue = 0;
    private Float   previousMaxTempValue = 0f;
    private Float   previousMinTempValue = 0f;

    // Used to generate a humidity value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;
    // the fluctuating humidity sensor value
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 2f;

    // The current read temperature value.
    private Float tempValue = 0f;
    // The previously read temperature value.
    private Float previousTempValue = 0f;
    private SimpleDeviceAttribute<Integer> minThreshold;
    private SimpleDeviceAttribute<Integer> maxThreshold;
    private SimpleReadOnlyDeviceAttribute<Float> maxTemp;
    private SimpleReadOnlyDeviceAttribute<Float> minTemp;
    private SimpleReadOnlyDeviceAttribute<Date> startTime;
    private SimpleReadOnlyDeviceAttribute<Float> temp;
    private SimpleReadOnlyDeviceAttribute<String> unit;
    private String unitValue = "C";
    private Thread measurementThread = null;

    private MessagingService ms;
    /**
     * Creates a new instance of the {@code SensorApplication}
     */
    @Inject
    public TemperatureSensorApplication(MessagingService messagingService) {
        ms = messagingService;
    }

    @Override
    protected void start() {
        EventService eventService = getEndpointContext().getEventService();
        startTimeValue = new Date(Instant.now().toEpochMilli());
        // Instantiate the temperature sensor attribute, passing a lambda which simply returns the endpoint's stored
        // temperature value.
        temp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "temperature",
                eventService,
                ()->tempValue);

        // Instantiate the temperature sensor attributes, passing lambdas which simply sets or returns the endpoint's
        // stored unit value.
        unit = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "unit",
                eventService,
                () -> unitValue);

        minThreshold = new SimpleDeviceAttribute<Integer>(
                this,
                "minThreshold ",
                eventService,
                () -> minThresholdValue,
                (s) -> {
                    minThresholdValue = s;
                    System.out.println(new Date().toString() + " : " + getId() + " : Set : \"minThreshold\" = " + minThresholdValue);
                    
                    ms.submit(new TemperatureSensorEvent.Builder(this)
                            .temp(tempValue)
                            .minThreshold(minThresholdValue)
                            .build().toDataMessage());
                });

        maxThreshold = new SimpleDeviceAttribute<Integer>(
                this,
                "maxThreshold",
                eventService,
                () -> maxThresholdValue,
                (s) -> {
                    maxThresholdValue = s;
                    System.out.println(new Date().toString() + " : " + getId() + " : Set : \"maxThreshold\" = " + maxThresholdValue);
                    
                    ms.submit(new TemperatureSensorEvent.Builder(this)
                            .temp(tempValue)
                            .maxThreshold(maxThresholdValue)
                            .build().toDataMessage());
                });

        maxTemp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "maxTemp",
                eventService,
                () -> maxTempValue);

        minTemp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "minTemp",
                eventService,
                () -> minTempValue);

        startTime = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "startTime",
                eventService,
                () -> startTimeValue);

        processDeviceData();
    }



    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> maxTempProperty() {
        return maxTemp;
    }

    @Override
    public DeviceAttribute<Integer> minThresholdProperty() {
        return minThreshold;
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> minTempProperty() {
        return minTemp;
    }

    @Override
    public void power(Boolean isOn) {
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"power\" = " + isOn);
        this.isOn = isOn;
    }

    @Override
    public void reset() {
        // Reset the temperature.
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"reset\"");
        minTempValue = tempValue;
        maxTempValue = tempValue;
    }

    protected void processDeviceData() {
        done.set(false);

        measurementThread = new Thread(() -> {
            isOn = true;

            // TODO: add shutdown logic to this loop
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }

                    if (isOn) {
                        // Generate a temperature value which exceeds the min and max threshold values.  The min and max
                        // threshold values can be found in the TemperatureSensor device model.
                        tempValue = calculateNextValue();

                        StringBuilder outputMsg = new StringBuilder(new Date().toString()).append(" : ").append(getId());

                        // Only send events if maxTemp, minTemp, or temperature has changed.
                        if (tempValue.intValue() != previousTempValue.intValue()) {

                            outputMsg.append(" : Data : ").append("\"temp\" = ").append(tempValue.floatValue());
                            TemperatureSensorEvent.Builder event = new TemperatureSensorEvent.Builder(this);
                            event.temp(tempValue);
                            
                            if (tempValue < minTempValue) {
                                minTempValue = tempValue;
                                event.maxTemp(maxTempValue);
                                outputMsg.append(" : ").append("\"minTemp\" = ").append(minTempValue.floatValue());
                            }
                            
                            if (tempValue > maxTempValue) {
                                maxTempValue = tempValue;
                                event.minTemp(minTempValue);
                                outputMsg.append(" : ").append("\"maxTemp\" = ").append(maxTempValue.floatValue());
                            }

                            ms.submit(event.build().toDataMessage());

                            // Send too hot and too cold events if needed.
                            if (tempValue < minThresholdValue) {
                                outputMsg.append("\n").append(new Date().toString()).append(" : ").append(getId())
                                    .append(" : Alert : ").append("\"temp\" = ").append(tempValue.floatValue());
                                ms.submit(new TemperatureSensorTooColdEvent.Builder(this)
                                        .temp(tempValue)
                                        .minThreshold(minThresholdValue)
                                        .build().toAlertMessage());

                            } else if (maxThresholdValue < tempValue) {
                                outputMsg.append("\n").append(new Date().toString()).append(" : ").append(getId())
                                        .append(" : Alert : ").append("\"temp\" = ").append(tempValue.floatValue());
                                
                                ms.submit(new TemperatureSensorTooHotEvent.Builder(this)
                                        .temp(tempValue)
                                        .maxThreshold(maxThresholdValue)
                                        .build().toAlertMessage());
                            }
                            System.out.println(outputMsg.toString());
                        }
                    }
                }
            }
        });

        measurementThread.start();
    }

    @Override
    public ReadOnlyDeviceAttribute<Date> startTimeProperty() {
        return startTime;
    }

    /**
     * Calculates a temperature value.
     *
     * @return a generated humidity value.
     */
    private float calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // round the temperature to 2 places and return
        int l = (int)((setPoint + delta) * 100.0);
        return(float)l / 100;
    }
    /**
     * Stops the device endpoint.
     * Unregisters from receiving sensor value updates.
     */
    @Override
    protected void stop() {
        done.set(true);
        isOn = false;

        if (measurementThread != null) {
            synchronized (measurementThread) {
                measurementThread.notify();
                try {
                    measurementThread.join();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        }

        super.stop();
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> tempProperty() {
        return temp;
    }

    @Override
    public ReadOnlyDeviceAttribute<String> unitProperty() {
        return unit;
    }

}
