/*
 * Copyright (c) 2014, 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.temperature;

import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEndpoint;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooColdEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooHotEvent;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;
import oracle.iot.message.AlertMessage;
import oracle.iot.message.DataMessage;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

 /**
 * The temperature sensor endpoint represents the endpoint for a temperature sensor device.  The endpoint abstracts the
 * device within the IoT system.  Since the IoT system deals mainly with the endpoint, the underlying device can be
 * replaced without effecting the system.  The change in the IoT system is the association between the endpoint and
 * the new and old device.
 *
 * This implementation also acts as a temperature sensor by generating temperature values.  When used with a real
 * device, this class would communicate with the real device for getting and setting information.
 *
 * This sample implements both the traditional application approach and the new Implicit Edge Computing (IEC) approach.
 * These two approaches are encapsulated in the WithoutPoliciesThread and WithPoliciesThread respectively.  The default
 * is the traditional approach.  Setting the com.oracle.iot.sample.use_policy system property will use the IEC approach
 * instead.  The IEC approach pushes the application logic to the device policies and obviates the need for an
 * associated application.
 */
@IoTDeviceEndpoint
public class TemperatureSensorEndpointImpl extends AbstractDeviceEndpoint implements TemperatureSensorEndpoint {
    /**
     * This sample can be used with policies, or without policies. By default,
     * the sample does not use policies. Set the property
     * "com.oracle.iot.sample.use_policy" to true to use policies.
     */
    private static final boolean usePolicy;
    private final static long OUTPUT_INTERVAL = 2000;
    private static final AtomicBoolean done = new AtomicBoolean(false);
    private static Random random = new Random();
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:temperature_sensor:attributes";

    private boolean isOn;
    private Date startTimeValue;
    private Float   maxTempValue = 0f;
    // The maximum threshold value, initialized to the default in the device model.
    private Integer maxThresholdValue = 70;
    private Float   minTempValue = 0f;
    // The minimum threshold value, initialized to the default in the device model.
    private Integer minThresholdValue = 0;

    // Used to generate a Temperature value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;
    // the fluctuating Temperature sensor value
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a Temperature value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 2f;

    private final EventService eventService;
    // The current read temperature value.
    private Float tempValue = 0f;
    // The previously read temperature value.
    private Float previousTempValue = 0f;
    private final MessagingService messagingService;
    private SimpleDeviceAttribute<Integer> minThreshold;
    private SimpleDeviceAttribute<Integer> maxThreshold;
    private SimpleReadOnlyDeviceAttribute<Float> maxTemp;
    private SimpleReadOnlyDeviceAttribute<Float> minTemp;
    private SimpleReadOnlyDeviceAttribute<Date> startTime;
    private SimpleReadOnlyDeviceAttribute<Float> temp;
    private SimpleReadOnlyDeviceAttribute<String> unit;
    private String unitValue = "C";
    private TemperatureSensorThread thread = null;


    static {
        final String value = System.getProperty("com.oracle.iot.sample.use_policy", "true");
        usePolicy = "".equals(value) || Boolean.parseBoolean(value);
    }

    @Inject
    public TemperatureSensorEndpointImpl(Logger logger, MessagingService messagingService, EventService eventService) {
        super();
        logger.log(Level.INFO, "Created new temperature sensor endpoint.");
        this.messagingService = messagingService;
        this.eventService = eventService;
    }

    /**
     * Calculates a temperature value.
     *
     * @return a generated Temperature value.
     */
    private float calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // round the temperature to 2 places and return
        int l = (int) ((setPoint + delta) * 100.0);
        return(float) l / 100;
    }

    /**
     * Initializes the device endpoint.
     */
    void init() {
        // Which sensor data is processed depends on whether or not policies are used.
        // Compare the code in the run method of the WithPolicies class
        // to that of the WithoutPolicies class.
        if (usePolicy) {
            thread = new TemperatureSensorEndpointImpl.WithPoliciesThread(this);
        } else {
            thread = new TemperatureSensorEndpointImpl.WithoutPoliciesThread(this);
        }
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> maxTempProperty() {
        return maxTemp;
    }

    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> minTempProperty() {
        return minTemp;
    }

    @Override
    public DeviceAttribute<Integer> minThresholdProperty() {
        return minThreshold;
    }

    @Override
    public void power(Boolean isOn) {
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"power\" = " + isOn);
        this.isOn = isOn;
    }

    @Override
    public void reset() {
        // Reset the temperature.
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"reset\"");
        minTempValue = tempValue;
        maxTempValue = tempValue;
    }

    @Override
    protected void start() throws Exception {
        done.set(false);
        super.start();
        thread.start();
    }

    @Override
    public ReadOnlyDeviceAttribute<Date> startTimeProperty() {
        return startTime;
    }

    /**
     * Stops the device endpoint.
     * Unregisters from receiving sensor value updates.
     */
    @Override
    protected void stop() throws Exception {
        done.set(true);
        isOn = false;

        if (thread != null) {
            synchronized(thread) {
                thread.notify();
            }
            thread.join();
        }

        super.stop();
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> tempProperty() {
        return temp;
    }

    @Override
    public ReadOnlyDeviceAttribute<String> unitProperty() {
        return unit;
    }

    private abstract class TemperatureSensorThread extends Thread {
        TemperatureSensorEndpoint temperatureSensorEndpoint;

        TemperatureSensorThread(TemperatureSensorEndpoint TemperatureSensorEndpoint) {
            this.temperatureSensorEndpoint = TemperatureSensorEndpoint;

            // Events are sent to the local Gateway event system and are picked up by anyone listening.  In this case,
            // the TemperatureSensorApplication is listening and will forward data to the IoT CS.
            System.out.println("TemperatureSensorEndpointImpl.init deviceModels=" + getDeviceModels());
            startTimeValue = new Date(Instant.now().toEpochMilli());

            // Instantiate the temperature sensor attribute, passing a lambda which simply returns the endpoint's stored
            // temperature value.
            temp = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "temperature",
                    eventService,
                    ()->tempValue);

            // Instantiate the temperature sensor attributes, passing lambdas which simply sets or returns the endpoint's
            // stored unit value.
            unit = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "unit",
                    eventService,
                    () -> unitValue);

            minThreshold = new SimpleDeviceAttribute<>(
                    this,
                    "minThreshold ",
                    eventService,
                    () -> minThresholdValue,
                    (s) -> {
                        minThresholdValue = s;
                        System.out.println(new Date().toString() + " : " + getId() + " : Set : \"minThreshold\" = " + minThresholdValue);

                        eventService.fire(new TemperatureSensorEvent.Builder(temperatureSensorEndpoint)
                                .temp(tempValue)
                                .minThreshold(minThresholdValue)
                                .build()
                        );
                    });

            maxThreshold = new SimpleDeviceAttribute<>(
                    this,
                    "maxThreshold",
                    eventService,
                    () -> maxThresholdValue,
                    (s) -> {
                        maxThresholdValue = s;
                        System.out.println(new Date().toString() + " : " + getId() + " : Set : \"maxThreshold\" = " + maxThresholdValue);

                        eventService.fire(new TemperatureSensorEvent.Builder(temperatureSensorEndpoint)
                                .temp(tempValue)
                                .maxThreshold(maxThresholdValue)
                                .build()
                        );
                    });

            maxTemp = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "maxTemp",
                    eventService,
                    () -> maxTempValue);

            minTemp = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "minTemp",
                    eventService,
                    () -> minTempValue);

            startTime = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "startTime",
                    eventService,
                    () -> startTimeValue);
        }
    }

    private class WithPoliciesThread extends TemperatureSensorThread {
        WithPoliciesThread(TemperatureSensorEndpoint TemperatureSensorEndpoint) {
            super(TemperatureSensorEndpoint);
        }

        @Override
        public void run() {
            isOn = true;

            // TODO: add shutdown logic to this loop
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }

                    if (isOn) {
                        // Generate a temperature value which exceeds the min and max threshold values.  The min and max
                        // threshold values can be found in the TemperatureSensor device model.
                        tempValue = calculateNextValue();

                        System.out.println(new Date().toString() + " : " + getId() + " : Data : \"temp\" = " +
                                tempValue);

                        previousTempValue = tempValue;

                        final DataMessage.Builder builder = new DataMessage.Builder()
                                .source(temperatureSensorEndpoint.getId())
                                .eventTime(new Date())
                                .format(FORMAT_URN)
                                .dataItem("temp", tempValue)
                                .dataItem("maxThreshold", maxThresholdValue);

                        if (tempValue < minTempValue) {
                            minTempValue = tempValue;
                            builder.dataItem("minTemp", minTempValue);
                        }

                        if (tempValue > maxTempValue) {
                            maxTempValue = tempValue;
                            builder.dataItem("maxTemp", maxTempValue);
                        }

                        messagingService.offer(builder.build());
                    }
                }
            }
        }
    }

    private class WithoutPoliciesThread extends TemperatureSensorThread {
        WithoutPoliciesThread(TemperatureSensorEndpoint temperatureSensorEndpoint) {
            super(temperatureSensorEndpoint);
        }

        @Override
        public void run() {
            isOn = true;

            // TODO: add shutdown logic to this loop
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }

                    if (isOn) {
                        // Generate a temperature value which exceeds the min and max threshold values.  The min and max
                        // threshold values can be found in the TemperatureSensor device model.
                        tempValue = calculateNextValue();

                        // Only send events if maxTemp, minTemp, or temperature has changed.
                        if ((tempValue.intValue() != previousTempValue.intValue())) {
                            previousTempValue = tempValue;

                            TemperatureSensorEvent.Builder event =
                                    new TemperatureSensorEvent.Builder(temperatureSensorEndpoint);

                            event.temp(tempValue);

                            if (tempValue < minTempValue) {
                                minTempValue = tempValue;
                                event.minTemp(minTempValue);
                            }

                            if (tempValue > maxTempValue) {
                                maxTempValue = tempValue;
                                event.maxTemp(maxTempValue);
                            }

                            // Create and send an event indicating the new value.  Note that this is sent over the
                            // Gateway's local event service to be handled by an event listener (e.g. the
                            // HumiditySensorApplication).  The HumiditySensorApplication may choose to send the data
                            // to the IoT CS, but this method call does not send the data to the IoT CS.
                            eventService.fire(event.build());

                            // Send too hot and too cold events if needed.
                            if (tempValue < minThresholdValue) {
                                eventService.fire(new TemperatureSensorTooColdEvent.Builder(temperatureSensorEndpoint)
                                        .temp(tempValue)
                                        .minThreshold(minThresholdValue)
                                        .severity(AlertMessage.Severity.CRITICAL)
                                        .build());
                            } else if (maxThresholdValue < tempValue) {
                                eventService.fire(new TemperatureSensorTooHotEvent.Builder(temperatureSensorEndpoint)
                                        .temp(tempValue)
                                        .maxThreshold(maxThresholdValue)
                                        .build());
                            }
                        }
                    }
                }
            }
        }
    }
}
