/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.app.sensor;

import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEvent;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorTooHumidEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooColdEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooHotEvent;
import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;
import oracle.iot.event.EventHandlerReference;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.util.ArrayList;
import java.util.Date;
import java.util.ResourceBundle;

/**
 * Application that receives events from the TemperatureSensor and HumiditySensor adapters and sends messages to the IoT
 * CS.
 *
 * Note that this application is not the endpoint.  TemperatureSensorEndpointImpl and HumiditySensorEndpointImpl are the
 * endpoints.  This application registers to receive events from the Gateway event subsystem, which the
 * TemperatureSensorEndpointImpl and HumiditySensorEndpointImpl send over the event subsystem, converts the events to
 * messages, and sends the messages to the IoT CS.
 */
@IoTDeviceApplication
public class SensorApplication extends AbstractApplication {

    private ResourceBundle messages;
    private ArrayList<EventHandlerReference> eventHandlers = new ArrayList<>();
    /**
     * Creates a new instance of the {@code SensorApplication}
     */
    @Inject
    public SensorApplication() {
        messages = ResourceBundle.getBundle("Messages");
    }

    @Override
    protected void start() {
        final String alertFrom = messages.getString("alertfrom");
        final String eventFrom = messages.getString("eventfrom");
        synchronized(eventHandlers) {
            EventService eventService = getEndpointContext().getEventService();

            // Register for humidity sensor events.
            eventHandlers.add(eventService.register(HumiditySensorEvent.REPORT,
                (event) -> {
                    // Print the event to console
                    System.out.println(new Date().toString() + " : " + getId() + eventFrom + event.getSource().getId()
                            + " : Data : \"humidity\" = " + event.getHumidity());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(event.toDataMessage());
                }));

            // Register for humidity sensor alerts.
            eventHandlers.add(eventService.register(HumiditySensorTooHumidEvent.ALERT,
                (alert) -> {
                    // Print the event to console
                    System.out.println(new Date().toString() + " : " + getId() + alertFrom + alert.getSource().getId()
                            + " : Alert : \"humidity\" = " + alert.getHumidity());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(alert.toAlertMessage());
                }));

            // Register for temperature sensor events.
            eventHandlers.add(eventService.register(TemperatureSensorEvent.REPORT,
                (event) -> {
                    // Print the event to console
                    // Print the event to console
                    System.out.println(new Date().toString() + " : " + getId() + eventFrom + event.getSource().getId()
                            + " : Data : \"temp\" = " + event.getTemp());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(event.toDataMessage());
                }));

            // Register for temperature sensor alerts.
            eventHandlers.add(eventService.register(TemperatureSensorTooHotEvent.ALERT,
                (alert) -> {
                    // Print the event to console
                    System.out.println(new Date().toString() + " : " + getId() + alertFrom + alert.getSource().getId()
                            + " : Alert : \"temp\" = " + alert.getTemp());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(alert.toAlertMessage());
                }));
            // Register for temperature sensor alerts.
            eventHandlers.add(eventService.register(TemperatureSensorTooColdEvent.ALERT,
                    (alert) -> {
                        // Print the event to console
                        System.out.println(new Date().toString() + " : " + getId() + alertFrom + alert.getSource().getId()
                                + " : Alert : \"temp\" = " + alert.getTemp());

                        // Send corresponding data message to the server
                        getEndpointContext().getMessagingService().submit(alert.toAlertMessage());
                    }));
        }
    }

    @Override
    protected void stop() {
        super.stop();

        // unregister our event handlers
        synchronized (eventHandlers) {
            for (EventHandlerReference handler : eventHandlers) {
                handler.unregister();
            }
        }
    }
}
