/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.motionactivatedcamera;

import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEndpoint;
import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraRecordingEvent;
import oracle.iot.beans.InvalidationListener;
import oracle.iot.beans.Observable;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.event.EventService;
import oracle.iot.storage.StorageSyncStatus;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.logging.Logger;


/**
 * InvalidationListener that receives events when uploading an image completes.
 */
class VideoUploadInvalidationListener implements InvalidationListener {
    private int duration = 0;
    private Date startTime = null;
    private EventService eventService = null;
    private Logger logger = null;
    private MotionActivatedCameraEndpoint motionActivatedCameraEndpoint;


    VideoUploadInvalidationListener(MotionActivatedCameraEndpoint motionActivatedCameraEndpoint,
                                    EventService eventService, Logger logger, Date startTime,
                                    int duration)
    {
        this.motionActivatedCameraEndpoint= motionActivatedCameraEndpoint;
        this.eventService = eventService;
        this.logger = logger;
        this.duration = duration;
        this.startTime = startTime;
    }

    /**
     * When a video has completed uploading, get the data and send a MotionActivatedCameraEvent
     * over the event service.
     *
     * @inheritDoc
     */
    @Override
    public void invalidated(Observable observable) {
        System.out.println("Completed uploading video.");

        if ((eventService != null) && (observable instanceof ObservableFuture)) {
            ObservableFuture<StorageSyncStatus> observableFuture = (ObservableFuture) observable;

            try {
                StorageSyncStatus storageSyncStatus = observableFuture.get();

                if (storageSyncStatus.getState() == StorageSyncStatus.State.COMPLETED) {
                    URI uri = new URI(storageSyncStatus.getStorageObject().getURI());

                    MotionActivatedCameraRecordingEvent.Builder eventBuilder =
                        new MotionActivatedCameraRecordingEvent.Builder(motionActivatedCameraEndpoint)
                            .startTime(startTime)
                            .duration(duration)
                            .video(uri.toString());

                    logger.info("Firing MotionActivatedCameraEvent for video upload.");
                    eventService.fire(eventBuilder.build());
                }
            } catch (ExecutionException | InterruptedException | URISyntaxException e) {
                e.printStackTrace();
            }
        }
    }
}
