/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.motionactivatedcamera;

import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCamera;
import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEndpoint;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;
import javax.inject.Inject;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.endpoint.EndpointContext;
import oracle.iot.event.EventService;
import oracle.iot.storage.StorageObject;
import oracle.iot.storage.StorageService;
import oracle.iot.storage.StorageSyncStatus;

/**
 * Example endpoint implementation of an endpoint which implements the motion activated camera
 * sensor device model.  The main thread looks for images in a directory specified by a system
 * property and uploads the images into the storage service.
 *
 * Uploads images with a .jpeg extension found in the directory specified by the following property:
 * com.oracle.iot.sample.mac.storageUploadImagePath property into the storage service.
 *
 * The record action "simulates" recording a video.  It does this by using the videos with a .mp4
 * extension found in the directory specified by the com.oracle.iot.sample.storageUploadVideoPath
 * property and uploading it to the storage service.
 */
@IoTDeviceEndpoint
public class MotionActivatedCameraEndpointImpl extends AbstractDeviceEndpoint
    implements MotionActivatedCameraEndpoint
{
    // Flags to indicate when to stop the endpoint.
    private static final AtomicBoolean imagesStopped = new AtomicBoolean(false);
    private static final AtomicBoolean videosStopped = new AtomicBoolean(false);
    // Property to specify the directory to find videos to "record" and upload them.
    private static final String RECORD_VIDEOS_PATH_PROPERTY =
        "com.oracle.iot.sample.mac.recordVideoPath";
    // Property to specify the directory to find images and upload them.
    private static final String UPLOAD_IMAGES_PATH_PROPERTY =
        "com.oracle.iot.sample.mac.storageUploadImagePath";

    private EventService eventService;
    // List of images found in the specified image directory.
    private final List<File> imageFiles = new ArrayList<>();
    // List of videos found in the specified video directory.
    private final List<File> videoFiles = new ArrayList<>();
    private final Logger logger;
    private String recordVideosPath = null;
    private StorageService storageService;
    private String uploadImagesPath = null;
    private final String endpointId;
    private Thread imagesThread;
    private Thread videosThread;

    //////////////////////////////////
    // From MotionActivatedCameraEvent
    private Date imageTimeValue = null;
    private SimpleReadOnlyDeviceAttribute<String> image;
    private SimpleReadOnlyDeviceAttribute<Date> imageTime;
    private String imageValue = null;
    //////////////////////////////////


    @Inject
    public MotionActivatedCameraEndpointImpl(EndpointContext endpointContext,
                                             EventService eventService,
                                             StorageService storageService,
                                             Logger logger)
    {
        super();
        this.endpointId = endpointContext.getEndpointId();
        this.eventService = eventService;
        this.storageService = storageService;
        this.logger = logger;
        String tmpDir = "/tmp";
        this.recordVideosPath = System.getProperty(RECORD_VIDEOS_PATH_PROPERTY, tmpDir);
        this.uploadImagesPath = System.getProperty(UPLOAD_IMAGES_PATH_PROPERTY, tmpDir);
        logger.info(MessageFormat.format("|endpointContext={0}|eventService={1}|storageService={2}|logger={3}.",
            endpointContext, eventService, storageService, logger));
    }

    /**
     * Returns the file with the name which has the time which is closest to numSeconds.  It's
     * expected that the video file name have this format:
     *
     * stopwatch_<N>s.mp4
     *
     * Where N is the number of seconds of video.  For example, a video which is 30 seconds long
     * is named:
     *
     * stopwatch_30s.mp4.
     *
     * @param numSeconds the number of seconds for a video.
     * @param videoFiles a {@link List} of {@link File}s.
     * @return the video file with the name which contains the closest number of seconds to the
     *         requested length, or the first file if none match.
     */
    private File getClosestInTime(int numSeconds, List<File> videoFiles) {
        File closestVideoInTime = null;

        if ((videoFiles != null) && (!videoFiles.isEmpty())) {
            closestVideoInTime = videoFiles.get(0);  // Default is first file.
            int closestTimeDiff = Integer.MAX_VALUE;

            for (File file : videoFiles) {
                String fileName = file.getName();
                String key1 = "stopwatch_";
                String key2 = "s.mp4";

                // First, check for an exact match.
                if ((key1 + numSeconds + key2).equals(fileName)) {
                    closestVideoInTime = file;
                    break;
                }

                if (fileName.contains(key1) && fileName.contains(key2)) {
                    String timeStr = fileName.substring(fileName.indexOf(key1) + key1.length(),
                            fileName.indexOf(key2));

                    int time = Integer.MAX_VALUE;

                    try {
                        time = Integer.valueOf(timeStr);
                    } catch (NumberFormatException nfe) {
                    }

                    int timeDiff = time < numSeconds ?
                        numSeconds - time : time - numSeconds;

                    if (timeDiff < closestTimeDiff) {
                        closestVideoInTime = file;
                        closestTimeDiff = timeDiff;
                    }
                }
            }
        }

        return closestVideoInTime;
    }

    /**
     * Returns a list of the files in the specified directory with the specified extension.
     *
     * @param dir the directory to search.
     * @param ext the extension of the files to find.
     * @return a list of the files in the specified directory with the specified extension.
     */
    private List<File> getFiles(final String dir, final String ext) {
        List<File> filesList = new ArrayList<>();
        final File directory = new File(dir);

        // File name filter for the dir/extension.
        final File[] files = directory.isDirectory() ?
           directory.listFiles((dir1, name) -> name.endsWith(ext)) : null;

        if (files != null && files.length > 0) {
            // Order the files by name.
            filesList.sort((file1, file2) -> {
                final String name1 = file1.getName();
                final String name2 = file2.getName();
                int c = name1.length() - name2.length();

                if (c == 0) {
                    c = name1.compareTo(name2);
                }

                return c;
            });

            filesList = Arrays.asList(files);
        } else {
            String message = !directory.isDirectory()
                ? "Could not find: " + directory.getPath()
                : "Could not find images in path: " + directory.getPath();

            logger.info(message);
        }

        return filesList;
    }

    @Override
    public ReadOnlyDeviceAttribute<String> imageProperty() {
        return image;
    }

    @Override
    public ReadOnlyDeviceAttribute<Date> imageTimeProperty() {
        return imageTime;
    }

    /**
     * Initialize the endpoint.  Setup the attributes.
     */
    public void init() {
        // Images will be sent when this endpoint implementation runs.
        imageFiles.addAll(getFiles(uploadImagesPath, "jpeg"));
        // Videos will be sent as part of the 'record' action.
        videoFiles.addAll(getFiles(recordVideosPath, "mp4"));

        // Initialize the attributes.
        image = new SimpleReadOnlyDeviceAttribute<>(
            this,
            MotionActivatedCamera.IMAGE_ATTRIBUTE,
            eventService,
            () -> imageValue);

        imageTime = new SimpleReadOnlyDeviceAttribute<>(
            this,
            MotionActivatedCamera.IMAGE_TIME_ATTRIBUTE,
            eventService,
            () -> imageTimeValue);
    }

    @Override
    public void record(int numSecondsToRecord) {
        logger.info(MessageFormat.format("{0} : {1} : Action: \"record\" for {2} seconds.",
            new Date().toString(), endpointId, numSecondsToRecord));

        final int duration = numSecondsToRecord;

         File videoFile = getClosestInTime(numSecondsToRecord, videoFiles);

        logger.info(MessageFormat.format("Recording {0} for {1} seconds.",
            videoFile.getAbsolutePath(), duration));

        // Thread to call the storage service to upload videos.
        videosThread = new Thread(() -> {
            videosStopped.set(false);

            while (!videosStopped.get() && !videoFiles.isEmpty()) {
                Date startTime = new Date();

                // Simulate the time it takes to record the video by sleeping
                try {
                    Thread.sleep(duration);
                } catch (InterruptedException e) {
                    // Restore the interrupted status
                    Thread.currentThread().interrupt();
                }

                try {
                    // Upload using the storage service.
                    // If the com.oracle.iot.client.disable_storage_object_prefix property is set to
                    // true, you may want to include the endpoint ID in the video name like so:
                    // "motion_activated_camera_" + "_video-" + startTime.toString() + ".mp4"
                    final StorageObject storageObject =
                       storageService.createStorageObject("motion_activated_camera_" +
                            "video-" + startTime.toString() + ".mp4", "video/mp4");

                    storageObject.setInputStream(new FileInputStream(videoFile));
                    storageObject.setCustomMetadata("duration", String.valueOf(duration));
                    storageObject.setTimeToLive(1, TimeUnit.DAYS);

                    ObservableFuture<StorageSyncStatus> storageSyncStatusObservable =
                        storageService.sync(storageObject);

                    VideoUploadInvalidationListener videoUploadInvalidationListener =
                        new VideoUploadInvalidationListener(this, eventService, logger,
                            startTime, duration);

                    storageSyncStatusObservable.addListener(videoUploadInvalidationListener);

                    logger.info(MessageFormat.format("{0} : {1} : Data: \"recording\"={2}",
                        new Date(), endpointId, storageObject.getURI()));
                } catch (GeneralSecurityException | IOException e) {
                    e.printStackTrace();
                }
            }
        });

        videosThread.setName("MotionActivatedCameraEndpointImpl Videos Thread");
        videosThread.start();
    }

    @Override
    protected void start() throws Exception {
        super.start();
        imagesStopped.set(false);

        // Create the main processing thread.  This thread "simulates" getting images from a
        // motion activated camera when the image is created via the motion.  In reality, we are
        // uploading specified images from a directory.
        imagesThread = new Thread(() -> {
            // Done when processing all image files or when stopped.
            while (!imagesStopped.get() && !imageFiles.isEmpty()) {
                synchronized (this) {
                    try {
                        // Simulate the time it takes to create an image and store it.
                        Thread.sleep(5000L);
                    } catch (InterruptedException ignore) {
                    }

                    try {
                        // Get the first image file in the List.
                        File imageFile = imageFiles.remove(0);

                        // If the com.oracle.iot.client.disable_storage_object_prefix property is
                        // set to true, you may want to include the endpoint ID in the image name
                        // like so:
                        // "motion_activated_camera_" + endpointId + "_" +*/imageFile.getName()
                        final StorageObject uploadStorageObject =
                            storageService.createStorageObject("motion_activated_camera_" +
                                  imageFile.getName(), "image/jpeg");

                        uploadStorageObject.setInputStream(new FileInputStream(imageFile));
                        uploadStorageObject.setTimeToLive(30, TimeUnit.DAYS);
                        Date startTime = new Date();

                        ObservableFuture<StorageSyncStatus> storageSyncStatusObservable =
                            storageService.sync(uploadStorageObject);

                        // Uncomment this code if you want to use Future.get calls instead of the
                        // Observable listener.
//                        try {
//                            // If we want to wait for the sync to complete and get the object this
//                            // way, as opposed to the listener callback.
//                            StorageSyncStatus storageSyncStatus = storageSyncStatusObservable.get();
//
//                            // If we want to wait for a certain period of time or for the sync to
//                            // complete and get the object this way, as opposed to the listener
//                            // callback.
//                            StorageSyncStatus storageSyncStatus =
//                                storageSyncStatusObservable.get(10000L, TimeUnit.MILLISECONDS);
//
//                            System.out.println("storageSyncStatus=" + storageSyncStatus);
//                        } catch(Exception e) {
//                            e.printStackTrace();
//                        }

//                        // If we want to cancel...
//                        System.out.println("Cancelled = " +
//                            storageSyncStatusObservable.cancel(true));

                        ImageUploadInvalidationListener imageUploadInvalidationListener =
                            new ImageUploadInvalidationListener(this, eventService, logger,
                                startTime);

                        storageSyncStatusObservable.addListener(imageUploadInvalidationListener);
                    } catch(GeneralSecurityException | IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        });

        imagesThread.setName("MotionActivatedCameraEndpointImpl Images Thread");
        imagesThread.start();
        System.out.println("Motion activated camera endpoint started.");
    }

    /**
     * Stops the device endpoint and unregisters it from receiving value updates.
     */
    @Override
    protected void stop() throws Exception {
        imagesStopped.set(true);
        videosStopped.set(true);

        if (imagesThread != null) {
            synchronized(imagesThread) {
                imagesThread.notify();
            }

            imagesThread.join();
        }

        if (videosThread != null) {
            synchronized(videosThread) {
                videosThread.notify();
            }

            videosThread.join();
        }

        imageFiles.clear();
        videoFiles.clear();
        super.stop();
    }
}
