/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.motionactivatedcamera;

import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;
import oracle.iot.endpoint.EndpointContext;

import javax.inject.Inject;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Example of a device adapter implementation for handling motion activated camera device model
 * endpoints.  Device adapters interface with devices to send/receive data and commands to/from the
 * device.
 */
@IoTDeviceAdapter
public class MotionActivatedCameraDeviceAdapter extends AbstractDeviceAdapter {
    private Logger logger;
    private final String endpointId;


    @Inject
    public MotionActivatedCameraDeviceAdapter(EndpointContext endpointContext, Logger logger) {
        this.endpointId = endpointContext.getEndpointId();
        this.logger = logger;
        logger.info("Created new motion activated camera device adapter.");
        logger.info(MessageFormat.format("|endpointContext={0}|logger={1}.",
            endpointContext, logger));
    }

    /**
     * Starts the device adapter.
     */
    @Override
    protected void start() throws Exception {
        new Thread(() -> {
            synchronized(this) {
                try {
                    this.wait(5000);
                } catch(InterruptedException ie) {
                };
            }

            String prefix = "MotActCam:";
            String address = prefix + endpointId;
            String manufacturer = "Manu:" + address;

            // Max length of Manufacturer is 64 characters, trim if needed.
            if (manufacturer.length() > 64) {
                int lengthMinusId = "Manu:".length();
                manufacturer = "Manu:" + prefix + endpointId.substring(0, 64 - lengthMinusId);
            }

            Metadata metadata = Metadata.builder()
                .protocol("device-model-sample")
                .protocolDeviceId(address)
                .manufacturer(manufacturer)
                .deviceClass("MotionActivatedCameraDevice")
                .serialNumber("Serial:" + address)
                .build();

            // Register the device and initialize the endpoint on the callback.
            ObservableFuture<MotionActivatedCameraEndpointImpl> future = registerDevice(
                manufacturer + "Serial:" + address,
                metadata, MotionActivatedCameraEndpointImpl.class,
                (device) -> device.init());

            // Listen for the new device representation being ready
            future.addListener((observable) -> {
                try {
                    MotionActivatedCameraEndpointImpl device = future.get();
                    logger.log(Level.FINE, "Motion activated camera registered: " +
                        device.getId());
                } catch (final Exception e) {
                    e.printStackTrace();
                }
            });
        }).start();
    }

    /**
     * Stops the device adapter.
     */
    @Override
    protected void stop() throws Exception {
        super.stop();
    }
}
