/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.motionactivatedcamera;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;
import java.util.Date;

/**
 * Event representing the recording from a camera.
 *
 * @param <T> the endpoint type that generated this event.
 */
public class MotionActivatedCameraRecordingEvent<T extends MotionActivatedCameraEndpoint>
    extends DataEvent.Default<T>
{
    private static final String FORMAT_URN =
        "urn:com:oracle:iot:device:motion_activated_camera:video";
    private static final String DESCRIPTION = "A video has been recorded.";

    private Date startTime = null;
    private Integer duration = null;
    private String video = null;



    /**
     * Creates a new instance of {@code MotionActivatedCameraRecordingEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected MotionActivatedCameraRecordingEvent(final T source,
                                              final EventType<? extends MotionActivatedCameraEvent> type,
                                              final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(DataMessage.Builder builder) {
        if (startTime != null) {
            builder.dataItem("startTime", startTime.toString());
        }

        if (duration != null) {
            builder.dataItem("duration", duration.toString());
        }

        if (video != null) {
            builder.dataItem("video", video.toString());
        }

        return builder;
    }

    /**
     * Gets the duration in this event.
     *
     * @return the duration.
     */
    public final Integer getDuration() {
        return duration;
    }

    /**
     * Gets the start time in this event.
     *
     * @return the start time.
     */
    public Date getStartTime() {
        return startTime;
    }

    /**
     * Gets the URI for the video in this event.
     *
     * @return the video URI.
     */
    public String getVideo() {
        return video;
    }

    @Override
    public String toString() {
        ToStringBuilder builder = new ToStringBuilder("MotionActivatedCameraRecordingEvent");

        if (startTime != null) {
            builder.append("startTime", startTime.toString());
        }

        if (duration != null) {
            builder.append("duration", duration.toString());
        }

        if (video != null) {
            builder.append("video", video.toString());
        }

        return builder.toString();
    }

    public static class Builder<T extends MotionActivatedCameraEndpoint> {
        private Date startTime = null;
        private Instant timestamp;
        private Integer duration = null;
        private T source;
        private String video = null;


        public Builder(final T source) {
            this.source = source;
            this.timestamp = Instant.now();
        }

        public Builder(final T source,final Instant timestamp) {
            this.source = source;
            this.timestamp = timestamp;
        }

        public MotionActivatedCameraRecordingEvent build() {
            MotionActivatedCameraRecordingEvent event = new MotionActivatedCameraRecordingEvent(source,
                MotionActivatedCameraRecordingEvent.REPORT, Instant.now());

            event.startTime = startTime;
            event.duration = duration;
            event.video = video;
            return event;
        }

        public MotionActivatedCameraRecordingEvent.Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public MotionActivatedCameraRecordingEvent.Builder startTime(Date startTime) {
            this.startTime = startTime;
            return this;
        }

        public MotionActivatedCameraRecordingEvent.Builder video(String video) {
            this.video = video;
            return this;
        }
    }
}
