/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.motionactivatedcamera;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;
import java.util.Date;

/**
 * Event representing an image or video being captured by a camera.
 */
public class MotionActivatedCameraEvent<T extends MotionActivatedCameraEndpoint>
    extends DataEvent.Default<T>
{
    private static final String FORMAT_URN =
        "urn:com:oracle:iot:device:motion_activated_camera:image";

    /**
     * Event type matching any {@code MotionActivatedCameraEvent}.
     */
    public static final
    EventType<MotionActivatedCameraEvent<? extends MotionActivatedCameraEndpoint>> REPORT =
        new EventType<>("MOTION_ACTIVATED_CAMERA", DataEvent.ANY);

    private Date imageTime = null;
    private String image = null;


    /**
     * Creates a new instance of {@code MotionActivatedCameraEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    MotionActivatedCameraEvent(final T source,
                               final EventType<? extends MotionActivatedCameraEvent> type,
                               final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }


    @Override
    protected DataMessage.Builder constructDataMessage(DataMessage.Builder builder) {
        if (image != null) {
            builder.dataItem("image", image.toString());
        }

        if (imageTime != null) {
            builder.dataItem("imageTime", imageTime.toString());
        }

        return builder;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);

        if (image != null) {
            out.append("image",image);
        }

        if (imageTime != null) {
            out.append("imageTime",imageTime);
        }

        return out;
    }

    @Override
    public MotionActivatedCameraEvent<T> copy(final T newSource) {
        MotionActivatedCameraEvent event =
            new MotionActivatedCameraEvent(newSource, MotionActivatedCameraEvent.REPORT,
                getTimestamp());

        event.image = image;
        event.imageTime = imageTime;
        return event;
    }

    /**
     * Gets the URI of the image in storage as a {@link String}.
     *
     * @return the URI as a string, or {@code null} if there is no URI.
     */
    public final String getImage() {
        return image;
    }

    /**
     * Gets the image time.
     *
     * @return the image time.
     */
    public final Date getImageTime() {
        return imageTime;
    }

    @Override
    public String toString() {
        ToStringBuilder builder = new ToStringBuilder("MotionActivatedCameraVideoEvent");

        if (image != null) {
            builder.append("image", image);
        }

        if (imageTime != null) {
            builder.append("imageTime", imageTime.toString());
        }

        return builder.toString();
    }

    /**
     * Builder for events.
     */
    public static class Builder<T extends MotionActivatedCameraEndpoint> {
        private Date imageTime = null;
        private Instant timestamp = null;
        private T source;
        private String image = null;


        public Builder(final T source) {
            this.source = source;
            this.timestamp = Instant.now();
        }

        public Builder(final T source,final Instant timestamp) {
            this.source = source;
            this.timestamp = timestamp;
        }

        /**
         * Builds and returns a MotionActivatedCameraEvent from the data in the builder.
         *
         * @return a MotionActivatedCameraEvent.
         */
        public MotionActivatedCameraEvent build() {
            MotionActivatedCameraEvent event = new MotionActivatedCameraEvent(source,
                MotionActivatedCameraEvent.REPORT, timestamp);

            event.image = image;
            event.imageTime = imageTime;
            return event;
        }

        public Builder image(String image) {
            this.image = image;
            return this;
        }

        public Builder imageTime(Date imageTime) {
            this.imageTime = imageTime;
            return this;
        }
    }
}
