/*
 * Copyright (c) 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.humidity;

import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEndpoint;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEvent;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorTooHumidEvent;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;
import oracle.iot.message.AlertMessage;
import oracle.iot.message.DataMessage;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The humidity sensor endpoint represents the endpoint for a humidity sensor device.  The endpoint
 * abstracts the device within the IoT system.  Since the IoT system deals mainly with the endpoint,
 * the underlying device can be replaced without effecting the system.  The change in the IoT system
 * is the association between the endpoint and the new and old device.
 *
 * This implementation also acts as a humidity sensor by generating humidity values.  When used with
 * a real device, this class would communicate with the real device for getting and setting
 * information.
 */
@IoTDeviceEndpoint
public class HumiditySensorEndpointImpl extends AbstractDeviceEndpoint implements HumiditySensorEndpoint {
    /**
     * This sample can be used with policies, or without policies. By default,
     * the sample does not use policies. Set the property
     * "com.oracle.iot.sample.use_policy" to true to use policies.
     */
    private static final boolean usePolicy;
    private final static long OUTPUT_INTERVAL = 2000;
    private static final AtomicBoolean done = new AtomicBoolean(false);
    // Use random to generate probability that we'll exceed threshold.
    private static Random random = new Random();
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:humidity_sensor:attributes";

    // The maximum threshold value, initialized to the default in the device model.
    private int maxThresholdValue = 80;
    // The fluctuating humidity sensor value.
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 5f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;

    // The attribute holding the maximum threshold value.
    private DeviceAttribute<Integer> maxThreshold;
    private final EventService eventService;
    // The generated humidity value.
    private Integer humidityValue = 0;
    private final MessagingService messagingService;
    // The attribute holding the humidity value.
    private ReadOnlyDeviceAttribute<Integer> humidity;
    private MainLogic thread = null;


    static {
        // Treat -Dcom.oracle.iot.sample.use_policy (without the =<value> part) the same as
        // -Dcom.oracle.iot.sample.use_policy=true.
        usePolicy = Boolean.getBoolean("com.oracle.iot.sample.use_policy");
    }

    @Inject
    public HumiditySensorEndpointImpl(Logger logger,
                                      MessagingService messagingService,
                                      EventService eventService) {
        super();
        logger.log(Level.INFO, "Created new Humidity Sensor endpoint.");
        this.messagingService = messagingService;
        this.eventService = eventService;
    }

    /**
     * Calculates a humidity value.
     *
     * @return a generated humidity value.
     */
    private int calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // return an rounded humidity point
        return (int)Math.round(setPoint + delta);
    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> humidityProperty() {
        return humidity;
    }

    /**
     * Initializes the device endpoint.
     */
    void init() {
        // Which sensor data is processed depends on whether or not policies are used.
        // Compare the code in the processSensorData method of the WithPolicies class
        // to that of the WithoutPolicies class.
        if (usePolicy) {
            System.out.println("Using device policies.");
            thread = new WithPoliciesThread(this);
        } else {
            thread = new WithoutPoliciesThread(this);
        }
    }

    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }

    @Override
    protected void start() throws Exception {
        done.set(false);
        super.start();
        thread.start();
    }

    @Override
    protected void stop() throws Exception {
        done.set(true);

        if (thread != null) {
            synchronized(thread) {
                thread.notify();
            }
            try {
                thread.join();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        super.stop();
    }

    private abstract class MainLogic extends Thread {
        HumiditySensorEndpoint humiditySensorEndpoint;

        MainLogic(HumiditySensorEndpoint humiditySensorEndpoint) {
            this.humiditySensorEndpoint = humiditySensorEndpoint;

            humidity = new SimpleReadOnlyDeviceAttribute<>(
                    this,
                    "humidity",
                    eventService,
                    () -> humidityValue);

            maxThreshold = new SimpleDeviceAttribute<>(
                    this,
                    "maxThreshold",
                    eventService,
                    () -> maxThresholdValue,
                    (s) -> {
                        System.out.println(new Date().toString() + " : " + getId() + " : " +
                            " Set : \"maxThreshold\" = " + s);

                        maxThresholdValue = s;
                        eventService.fire(
                                new HumiditySensorEvent.Builder(humiditySensorEndpoint)
                                        .humidity(humidityValue)
                                        .maxThreshold(maxThresholdValue)
                                        .build());
                    });
        }
    }

    private class WithPoliciesThread extends MainLogic {
        WithPoliciesThread(HumiditySensorEndpoint humiditySensorEndpoint) {
            super(humiditySensorEndpoint);
        }

        @Override
        public void run() {
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }
                }

                // Get a generated random number for the humidity value.
                humidityValue = calculateNextValue();

                System.out.println(new Date().toString() + " : " + getId() +
                    " : Data : \"humidity\" = " + humidityValue);

                // Only send events if the humidity has changed
                final DataMessage.Builder builder = new DataMessage.Builder()
                        .source(humiditySensorEndpoint.getId())
                        .eventTime(new Date())
                        .format(FORMAT_URN)
                        .dataItem("humidity", humidityValue)
                        .dataItem("maxThreshold", maxThresholdValue);

                messagingService.offer(builder.build());
            }
        }
    }

    private class WithoutPoliciesThread extends MainLogic {
        WithoutPoliciesThread(HumiditySensorEndpoint humiditySensorEndpoint) {
            super(humiditySensorEndpoint);
        }

        @Override
        public void run() {
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }
                }

                // Get a generated random number for the humidity value.
                humidityValue = calculateNextValue();
                System.out.println(new Date().toString() + " : " + getId() +
                    " : Data : \"humidity\" = " + humidityValue);

                // Only send events if the humidity has changed
                Integer previousHumidityValue = 0;
                if (humidityValue.intValue() != previousHumidityValue.intValue()) {

                    // Create and send an event indicating the new value.
                    eventService.fire(
                            new HumiditySensorEvent.Builder(humiditySensorEndpoint)
                                    .humidity(humidityValue)
                                    .build());

                    // Send an alert if the humidity value is outside the threshold.
                    if (humidityValue > maxThresholdValue) {
                        System.out.println(new Date().toString() + " : " + getId() +
                            " : Alert : \"humidity\" = " + humidityValue + ", \"maxThreshold\" = " +
                            maxThresholdValue);

                        eventService.fire(
                                new HumiditySensorTooHumidEvent.Builder(humiditySensorEndpoint)
                                        .humidity(humidityValue)
                                        .severity(AlertMessage.Severity.SIGNIFICANT)
                                        .build());
                    }
                }
            }
        }
    }
}
