/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
        
package com.oracle.iot.sample.daf.type.humidity;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a humidity sensor measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class HumiditySensorEvent<T extends HumiditySensorEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:humidity_sensor:attributes";
    private Integer humidity = null;
    private Integer maxThreshold = null;

    /**
     * Gets the humidity measurement contained in this event
     * 
     * @return the humidity
     */
    public final Integer getHumidity() {
        return humidity;
    }

    /**
     * Gets the max threshold of the device when this event was generated
     *
     * @return the max threshold
     */
    public Integer getMaxThreshold() {
        return maxThreshold;
    }
    
    /**
     * Event type matching any {@code HumiditySensorEvent}.
     */
    public static final EventType<HumiditySensorEvent<? extends HumiditySensorEndpoint>> REPORT =
            new EventType<>("HUMIDITY", DataEvent.ANY);

    /**
     * Creates a new instance of {@code HumiditySensorEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected HumiditySensorEvent(final T source,
                                  final EventType<? extends HumiditySensorEvent> type,
                                  final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(humidity != null) builder.dataItem("humidity", new Float(humidity));
        if(maxThreshold != null) builder.dataItem("maxThreshold", new Float(maxThreshold));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(humidity != null) out.append("humidity",humidity);
        if(maxThreshold != null) out.append("maxThreshold",maxThreshold);
        return out;
    }

    @Override
    public HumiditySensorEvent<T> copy(final T newSource) {
        HumiditySensorEvent event = new HumiditySensorEvent(newSource, HumiditySensorEvent.REPORT, getTimestamp());
        event.humidity = humidity;
        event.maxThreshold = maxThreshold;
        return event;
    }
    
    
    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("HumiditySensorEvent")).toString();
    }
    
    public static class Builder<T extends HumiditySensorEndpoint> {
        T source;
        Instant timestamp;
        Integer humidity;
        Integer maxThreshold;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder humidity(Integer humidity) { this.humidity = humidity; return this;}
        public Builder maxThreshold(Integer maxThreshold) { this.maxThreshold = maxThreshold; return this;}
        public HumiditySensorEvent build() {
            HumiditySensorEvent event = new HumiditySensorEvent(source, HumiditySensorEvent.REPORT, timestamp);
            event.humidity = humidity;
            event.maxThreshold = maxThreshold;
            return event;
        }
    }
}
