/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.helloworldadapter;

import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.message.DataMessage;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;

/**
 */
@IoTDeviceEndpoint
public class HelloWorldEndpointImpl extends AbstractDeviceEndpoint implements HelloWorldEndpoint {
    private static final AtomicBoolean done = new AtomicBoolean(false);
    private static final String MESSAGE = "Hello World.";

    private DataMessage dataMessage;
    private MessagingService messagingService;
    // The attribute holding the message.
    private ReadOnlyDeviceAttribute<String> message;
    private Thread messageThread = null;

    @Inject
    public HelloWorldEndpointImpl(Logger logger, MessagingService messagingService) {
        super();
        this.messagingService = messagingService;
        logger.info("HelloWorldEndpointImpl started.");
    }

    @Override
    public ReadOnlyDeviceAttribute<String> messageProperty() { return message; }

    /**
     * Initializes the device endpoint.
     *
     * @param hardwareId the hardware ID of the device.
     */
    void init(String hardwareId) {}

    @Override
    protected void start() throws Exception {
        done.set(false);
        super.start();

        dataMessage = new DataMessage.Builder()
            .dataItem("message", MESSAGE)
            .format("urn:com:oracle:iot:device:hello_world:attributes")
            .source(getEndpointContext().getEndpointId())
            .build();

        message = new SimpleReadOnlyDeviceAttribute<>(
            this,
            "message",
            getEndpointContext().getEventService(),
            () -> MESSAGE);

        // This thread simulates a humidity device and sends data and alert messages.
        messageThread = new Thread(() -> {
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(2000);
                    } catch (InterruptedException e) {
                    }

                    System.out.println("Sending: " + MESSAGE);
                    messagingService.submit(dataMessage);
                }
            }
        });

        // Get a generated random number for the humidity value.
        messageThread.start();
    }

    @Override
    protected void stop() throws Exception {
        done.set(true);

        if (messageThread != null) {
            synchronized(this) { messageThread.notify(); }
            messageThread.join();
        }

        super.stop();
    }
}
