/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.type.thermometer;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a temperature measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class ThermometerEvent<T extends ThermometerEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:thermometer:attributes";
    private Float temperature = null;

    /**
     * Gets the temperature measurement contained in this event
     *
     * @return the temperature
     */
    public final Float getTemperature() {
        return temperature;
    }

    /**
     * Event type matching any {@code ThermometerEvent}.
     */
    public static final EventType<ThermometerEvent<? extends ThermometerEndpoint>> REPORT =
            new EventType<>("TEMPERATURE", DataEvent.ANY);

    /**
     * Creates a new instance of {@code ThermometerEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected ThermometerEvent(final T source,
                                    final EventType<? extends ThermometerEvent> type,
                                    final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(temperature != null) builder.dataItem("temperature", new Float(temperature));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(temperature != null) out.append("temperature",temperature);
        return out;
    }

    @Override
    public ThermometerEvent<T> copy(final T newSource) {
        ThermometerEvent event = new ThermometerEvent(newSource, ThermometerEvent.REPORT, getTimestamp());
        event.temperature = temperature ;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("ThermometerEvent")).toString();
    }

    public static class Builder<T extends ThermometerEndpoint> {
        T source;
        Instant timestamp;
        private Float  temperature = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder temperature (Float  temperature ) { this.temperature = temperature ; return this;}
        public ThermometerEvent build() {
            ThermometerEvent event = new ThermometerEvent(source, ThermometerEvent.REPORT, timestamp);
            event.temperature  = temperature ;
            return event;
        }
    }
}
