/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.health;

import jdk.phd.Phd;
import jdk.phd.report.ConfigReport;
import jdk.phd.report.NumericMeasurement;
import jdk.phd.report.ScanReport;
import jdk.phd.report.spec.PulseOximeter;
import oracle.iot.beans.property.DoubleProperty;
import oracle.iot.beans.property.IntegerProperty;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.pulseoximeter.PulseOximeterEvent;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.HashMap;
import java.util.logging.Level;

@IoTDeviceEndpoint
public class PulseOximeterEndpoint extends PersonalHealthEndpoint implements com.oracle.iot.sample.daf.type.pulseoximeter.PulseOximeterEndpoint {

    private HashMap<Long, NumericMeasurement> o2Measurements = new HashMap<>();
    private HashMap<Long, NumericMeasurement> heartRateMeasurements = new HashMap<>();

    private final SimpleReadOnlyDeviceAttribute<Float> oxygenSaturation;
    private final SimpleReadOnlyDeviceAttribute<Integer> heartRate;
    private DoubleProperty oxygenSaturationProp;
    private IntegerProperty heartRateProp;
    private EventService es;
    private long timestampOffset = 0;

    @Inject
    public PulseOximeterEndpoint(EventService es)
    {
        super();
        this.es = es;
        oxygenSaturation = new SimpleReadOnlyDeviceAttribute<Float>(this, "oxygen saturation", es);
        heartRate = new SimpleReadOnlyDeviceAttribute<Integer>(this, "heart rate",es);
    }

    public ReadOnlyDeviceAttribute<Float> oxygenSaturationProperty() {
        return oxygenSaturation;
    }

    public Float getOxygenSaturation() {
        return oxygenSaturation.getValue();
    }

    public ReadOnlyDeviceAttribute<Integer> heartRateProperty() {
        return heartRate;
    }

    public Integer getHeartRate() {
        return heartRate.getValue();
    }

    @Override
    public void scanReported(Phd phd, ScanReport scanReport) {
        logger.log(Level.FINE, "PhdHandler.scanReported");

        PulseOximeter poScanReport = (PulseOximeter)scanReport;

        // Add all measurement components
        for(NumericMeasurement nm: poScanReport.getPulse())
            heartRateMeasurements.put(nm.getTimestamp(), nm);
        for(NumericMeasurement nm: poScanReport.getSpo2())
            o2Measurements.put(nm.getTimestamp(), nm);

        reportMeasurements();
    }

    @Override
    public boolean configReported(Phd phd, ConfigReport configReport) {
        logger.log(Level.FINE, "PhdHandler.configReported");
        if(configReport.getConfig().getConfigId() == 0x190) {
            logger.log(Level.INFO, "accepted streaming configuration from pulse oximeter device: " + address);
            return true;
        }
        logger.log(Level.FINE, "Config " + configReport.getConfig().getConfigId()
                + " reported - rejecting.");
        return false;
    }

    /**
     * Search stored individual measurement values and attempt to assemble consolidated events for dispatch
     */
    void reportMeasurements()
    {
        for(Long time : heartRateMeasurements.keySet()) dispatchEvent(time);
        for(Long time : o2Measurements.keySet()) dispatchEvent(time);
    }

    /**
     * Consolidate all measurement events for a given event time and dispatch
     *
     * @param time the time stamp of the measurements to find
     */
    void dispatchEvent(long time)
    {
        int heartRateVal = -1;
        double oxygenSaturationVal = -1;

        NumericMeasurement hr = heartRateMeasurements.remove(time);
        NumericMeasurement oxygenSaturation  = o2Measurements.remove(time);

        if(hr != null) heartRateVal = (int)hr.getValue();
        if(oxygenSaturation != null) oxygenSaturationVal = oxygenSaturation .getValue();


        dispatch(time, heartRateVal, oxygenSaturationVal);
    }


    void dispatch(long time, int heartRate, double oxygenSaturation) {
        logger.log(Level.FINE, "generating event: " +  + heartRate + ":" + oxygenSaturation
                + " from device: " + address);

        // Update the values if signaled, and grab a valid update time.
        if(oxygenSaturation != -1) this.oxygenSaturation.notifyValueUpdated((float)oxygenSaturation);
        if(heartRate != -1) this.heartRate.notifyValueUpdated(heartRate);

        // Pulse oximter in streaming mode use realitve time stamps. normalize these offsets
        // from gateway system time.
        if(timestampOffset == 0) {
            timestampOffset = System.currentTimeMillis() - time;
        }
        es.fire(new PulseOximeterEvent.Builder(this,Instant.ofEpochMilli(time))
                .heartRate(heartRate)
                .oxygenSaturation((float)oxygenSaturation)
                .build());
    }
}

