/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.health;

import jdk.phd.Phd;
import jdk.phd.PhdChannel;
import jdk.phd.PhdConfig;
import jdk.phd.PhdHandler;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.Metadata;

import java.io.InputStream;
import java.io.OutputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Base class for an IEEE 11073 based IoT endpoint device. Contains a default implementation of the
 * PhdHandler class.
 */
public abstract class PersonalHealthEndpoint extends AbstractDeviceEndpoint implements PhdHandler {

    /**
     * IEEE 11073 device type code representing a Pulse Oximeter
     */
    public static final int PULSE_OXIMETER = 4100;
    /**
     * IEEE 11073 device type code representing a Blood Pressure Monitor
     */
    public static final int BLOOD_PRESSURE_MONITOR = 4103;
    /**
     * IEEE 11073 device type code representing a Weighing Scale
     */
    public static final int WEIGHING_SCALE = 4111;

    private String hardwareId;
    protected String address;
    protected int type;
    protected Phd phdDevice;
    protected Logger logger;
    protected Metadata metadata;

    protected PersonalHealthEndpoint() {}

    /**
     * Returns a string representation of the device class name for the given type.
     *
     * @param type The device type constant
     *
     * @return the string representation, never be {@code null}
     */
    public static String getClassName(int type)
    {
        switch(type)
        {
            case PersonalHealthEndpoint.PULSE_OXIMETER: return "PulseOximeter";
            case PersonalHealthEndpoint.BLOOD_PRESSURE_MONITOR: return "BloodPressureMonitor";
            case PersonalHealthEndpoint.WEIGHING_SCALE: return "WeightScale";
            default: return "PersonalHealthDevice";
        }
    }
    /**
     * Initializes  metadata associated with this endpoint.
     *
     * @param hardwareId a globally unique hardware ID.
     * @param metadata the IoT framework metadata associated with this device
     * @param address the bluetooth address for the associated connected device
     * @param type the IEEE 11073 medical device type for this device
     * @param logger the logger to use for this device
     */
    public void init(String hardwareId, Metadata metadata, String address, int type, Logger logger)
    {
        this.hardwareId = hardwareId;
        this.metadata = metadata;
        this.address = address;
        this.type = type;
        this.logger = logger;
    }

    /**
     * Returns the IEEE 11073 type for this device.
     *
     * @return
     */
    public int getType() {
        return type;
    }

    /**
     * Returns the bluetooth address for this device.
     *
     * @return
     */
    public String getAddress() {
        return address;
    }

    public void channelConnect(InputStream is, OutputStream os)
    {
        PhdChannel phdChannel;

        // Create the PHD channel object
        phdChannel = createPhdChannel(is, os);

        /* This doPriv is required because we are running in the context of HDP, which has no knowledge of PHD */
        phdDevice =  AccessController.doPrivileged((PrivilegedAction<Phd>) () -> {
            return Phd.newPhd(phdChannel, this);
        });
    }

    PhdChannel createPhdChannel(InputStream is, OutputStream os) {
        return PhdChannel.newChannel(is, os);
    }

    Phd createPhd(PhdChannel channel) {
        return Phd.newPhd(channel, this);
    }

    public void channelDisconnect()
    {
        if(phdDevice != null)
            phdDevice.close();
    }

    @Override
    public void stateChanged(Phd phd, Phd.PhdState phdState) {
        logger.log(Level.FINE, "BluetoothHealthEndpoint PhdHandler.stateChanged");
    }

    @Override
    public boolean associationRequested(Phd phd, byte[] bytes, int i, PhdConfig phdConfig) {
        logger.log(Level.FINE, "BluetoothHealthEndpoint PhdHandler.associationRequested");
        return false;
    }

    @Override
    public void releaseRequested(Phd phd, byte[] bytes) {
        logger.log(Level.FINE, "BluetoothHealthEndpoint PhdHandler.releaseRequested");
    }
}
