/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.bloodpressuremonitor;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a blood presure measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class BloodPressureMonitorEvent<T extends BloodPressureMonitorEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:blood_pressure_monitoor:attributes";
    private Integer systolicPressure = null;
    private Integer diastolicPressure = null;
    private Integer heartRate = null;

    /**
     * Gets the systolic pressure measurement contained in this event
     *
     * @return the systolic pressure
     */
    public final Integer getSystolicPressure() {
        return systolicPressure;
    }

    /**
     * Gets the diastolic pressure measurement contained in this event
     *
     * @return the diastolic pressure
     */
    public Integer getDiastolicPressure() {
        return diastolicPressure;
    }

    /**
     * Gets the heart rate measurement contained in this event
     *
     * @return the heart rate
     */
    public Integer getHeartRate() {
        return heartRate;
    }
    
    /**
     * Event type matching any {@code BloodPressureMonitorEvent}.
     */
    public static final EventType<BloodPressureMonitorEvent<? extends BloodPressureMonitorEndpoint>> REPORT =
            new EventType<>("BLOOD_PRESSURE", DataEvent.ANY);

    /**
     * Creates a new instance of {@code BloodPressureMonitorEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected BloodPressureMonitorEvent(final T source,
                                      final EventType<? extends BloodPressureMonitorEvent> type,
                                      final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(systolicPressure != null) builder.dataItem("systolicPressure", new Float(systolicPressure));
        if(diastolicPressure != null) builder.dataItem("diastolicPressure", new Float(diastolicPressure));
        if(heartRate != null) builder.dataItem("heartRate", new Float(heartRate));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(systolicPressure != null) out.append("systolicPressure",systolicPressure);
        if(diastolicPressure != null) out.append("diastolicPressure",diastolicPressure);
        if(heartRate != null) out.append("heartRate",heartRate);
        return out;
    }

    @Override
    public BloodPressureMonitorEvent<T> copy(final T newSource) {
        BloodPressureMonitorEvent event = new BloodPressureMonitorEvent(newSource, BloodPressureMonitorEvent.REPORT, getTimestamp());
        event.systolicPressure = systolicPressure ;
        event.diastolicPressure  = diastolicPressure;
        event.heartRate = heartRate;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("BloodPressureMonitorEvent")).toString();
    }

    public static class Builder<T extends BloodPressureMonitorEndpoint> {
        T source;
        Instant timestamp;
        private Integer systolicPressure = null;
        private Integer diastolicPressure = null;
        private Integer heartRate = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder systolicPressure (Integer systolicPressure ) { this.systolicPressure = systolicPressure ; return this;}
        public Builder diastolicPressure (Integer diastolicPressure ) { this.diastolicPressure  = diastolicPressure ; return this;}
        public Builder heartRate(Integer heartRate) { this.heartRate = heartRate; return this;}
        public BloodPressureMonitorEvent build() {
            BloodPressureMonitorEvent event = new BloodPressureMonitorEvent(source, BloodPressureMonitorEvent.REPORT, timestamp);
            event.systolicPressure  = systolicPressure ;
            event.diastolicPressure = diastolicPressure;
            event.heartRate = heartRate;
            return event;
        }
    }
}
